package io.adbrix.sdk.data.repository;

import android.app.Activity;
import android.content.Context;
import android.content.res.Configuration;
import android.net.ConnectivityManager;
import android.net.Network;
import android.net.NetworkCapabilities;
import android.net.NetworkRequest;
import android.os.Build;
import android.util.Pair;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;

import javax.annotation.Nullable;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.EventBuffer;
import io.adbrix.sdk.component.IABXComponentsFactory;
import io.adbrix.sdk.component.SKDataSender;
import io.adbrix.sdk.component.TryOptional;
import io.adbrix.sdk.data.ABXBooleanState;
import io.adbrix.sdk.data.S3ConfigHandler;
import io.adbrix.sdk.data.dataprovider.DeviceRealtimeDataProvider;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.inAppMessage.AbxDialog;
import io.adbrix.sdk.data.inAppMessage.IInAppMessageViewFactory;
import io.adbrix.sdk.data.inAppMessage.InAppMessage;
import io.adbrix.sdk.data.inAppMessage.InAppMessageDAO;
import io.adbrix.sdk.data.inAppMessage.InAppMessageFactoryContainer;
import io.adbrix.sdk.data.modelprovider.ActionHistoryDeleteModelProvider;
import io.adbrix.sdk.data.modelprovider.ActionHistoryQueryModelProvider;
import io.adbrix.sdk.data.modelprovider.DRModelProvider;
import io.adbrix.sdk.data.modelprovider.EventModelProvider;
import io.adbrix.sdk.data.modelprovider.GDPRModelProvider;
import io.adbrix.sdk.data.modelprovider.InAppMessageModelProvider;
import io.adbrix.sdk.data.net.ApiConnection;
import io.adbrix.sdk.data.net.ApiConnectionManager;
import io.adbrix.sdk.data.push.ActionHistoryDAO;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.Repository;
import io.adbrix.sdk.domain.model.ActionHistory;
import io.adbrix.sdk.domain.model.ActionHistoryDeleteModel;
import io.adbrix.sdk.domain.model.ActionHistoryError;
import io.adbrix.sdk.domain.model.ActionHistoryIdType;
import io.adbrix.sdk.domain.model.ActionHistoryQueryModel;
import io.adbrix.sdk.domain.function.Completion;
import io.adbrix.sdk.domain.model.DRModel;
import io.adbrix.sdk.domain.model.DRState;
import io.adbrix.sdk.domain.model.Empty;
import io.adbrix.sdk.domain.model.Error;
import io.adbrix.sdk.domain.model.EventModel;
import io.adbrix.sdk.domain.model.IApiModel;
import io.adbrix.sdk.domain.model.Result;
import io.adbrix.sdk.domain.model.InAppMessageModel;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.domain.model.Success;
import io.adbrix.sdk.domain.model.UserPropertyCommand;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.CoreUtils;

public class RepositoryImpl implements Repository {
    private IABXComponentsFactory factory;
    private DataRegistry dataRegistry;
    private Context androidContext;
    private SKDataSender skDataSender;
    private ConnectivityManager connectivityManager;
    private ConnectivityManager.NetworkCallback networkCallback;
    private InAppMessageDAO inAppMessageDAO;
    private ExecutorService inAppMessageExecutorService;
    private InAppMessageFactoryContainer inAppMessageFactoryContainer;
    private Activity currentActivity;
    private AbxDialog abxDialog;
    private DeviceRealtimeDataProvider deviceRealtimeDataProvider;
    private ActionHistoryDAO actionHistoryDAO;
    private ExecutorService actionHistoryExecutor;

    public RepositoryImpl(IABXComponentsFactory factory) {
        this.factory = factory;

        try {
            this.dataRegistry = factory.createOrGetDataRegistry();
            this.androidContext = factory.getAndroidContext();

            this.inAppMessageDAO = new InAppMessageDAO(androidContext, dataRegistry);
            this.inAppMessageExecutorService = Executors.newSingleThreadExecutor();
            this.inAppMessageFactoryContainer = new InAppMessageFactoryContainer(inAppMessageDAO);
            this.deviceRealtimeDataProvider = factory.createOrGetDeviceRealtimeDataProvider();
            this.actionHistoryDAO = new ActionHistoryDAO(androidContext, dataRegistry);
            this.actionHistoryExecutor = Executors.newSingleThreadExecutor();
        } catch (IABXComponentsFactory.ComponentsCanNotCreateException e) {

        }
    }

    @Override
    public EventModel getEventModel(LogEventParameter logEventParameter) {
        AtomicReference<EventModel> eventModel = new AtomicReference<>();
        TryOptional.of(factory::createOrGetUserPropertyManager).ifPresent(it -> {
            EventModelProvider eventModelProvider = new EventModelProvider(
                    it,
                    this.dataRegistry);

            eventModelProvider.setLogEventParameter(logEventParameter);
            eventModel.getAndSet(eventModelProvider.provide());
        });

        return eventModel.get();
    }

    @Override
    public Boolean saveUserPropertyWithoutEvent(UserPropertyCommand userPropertyCommand) {
        AtomicBoolean result = new AtomicBoolean(false);

        TryOptional.of(factory::createOrGetUserPropertyManager)
                .ifPresent(it -> result.getAndSet(it.merge(userPropertyCommand)));
        return result.get();
    }

    @Override
    public void initDefaultConfigValue() {
        S3ConfigHandler.getInstance().initDefaultConfigValue(dataRegistry);
    }

    @Override
    public void logSameEventWithPaging(String eventName, List<JSONObject> eventParams) {

        if (S3ConfigHandler.getInstance().isBlockedEvent(eventName, "abx")) {
            return;
        }

        boolean isEventBlockedBySamplingFilter = TryOptional.of(factory::createOrGetEventSamplingFilter)
                .map(filter -> filter.isEventBlockedBySamplingFilter("abx:" + eventName))
                .orElse(false);

        if (isEventBlockedBySamplingFilter)
            return;

        long currentTimeMillis = System.currentTimeMillis();
        String eventId = CommonUtils.randomUUIDWithCurrentTime(currentTimeMillis);
        String eventDateTime = CommonUtils.getCurrentUTCInDBFormat(currentTimeMillis);

        String prevId;
        long sessionOrderNo = 0L;
        long eventOrderNo = 0L;
        prevId = dataRegistry.safeGetString(DataRegistryKey.STRING_PREV_ID, null);
        sessionOrderNo = dataRegistry.incrementAndGetLong(DataRegistryKey.LONG_SESSION_ORDER_NO);
        eventOrderNo = dataRegistry.incrementAndGetLong(DataRegistryKey.LONG_EVENT_ORDER_NO);

        int count = 1;
        for (JSONObject eventParamJson : eventParams) {
            try {
                JSONObject temp = new JSONObject();
                temp.put(CoreConstants.PAGE_ABX_KEY, count);
                temp = CommonUtils.parseValueWithDataType(temp, CommonUtils.FixType.PREFIX);

                eventParamJson.put(CoreConstants.PAGE_ABX_KEY, temp.getString(CoreConstants.PAGE_ABX_KEY));
            } catch (JSONException e) {
                AbxLog.e(Arrays.toString(e.getStackTrace()), true);
            }
            LogEventParameter eventParameter = new LogEventParameter(
                    eventId,
                    prevId,
                    CoreConstants.GROUP_ABX,
                    eventName,
                    CommonUtils.getMapFromJSONObject(eventParamJson),
                    0,
                    0,
                    sessionOrderNo,
                    eventOrderNo,
                    eventDateTime
            );

            final EventModel eventModel = getEventModel(eventParameter);

            if (eventModel == null) {
                AbxLog.e("repositoryImpl::logEvent error!! eventModel is null", true);
                return;
            }

            TryOptional.of(factory::createOrGetEventBuffer)
                    .ifPresent(buffer -> buffer.addEventModel(eventModel));

            count++;
        }

        // prev_id를 저장해준다.
        dataRegistry.putDataRegistry(
                new DataUnit(
                        DataRegistryKey.STRING_PREV_ID,
                        eventId,
                        5,
                        this.getClass().getName(),
                        true
                )
        );
    }

    @Override
    public void logEvent(LogEventParameter logEventParameter) {
        if (S3ConfigHandler.getInstance().isBlockedEvent(logEventParameter.eventName, logEventParameter.group)) {
            return;
        }

        boolean isEventBlockedBySamplingFilter = TryOptional.of(factory::createOrGetEventSamplingFilter)
                .map(filter -> filter.isEventBlockedBySamplingFilter(logEventParameter))
                .orElse(false);

        if (isEventBlockedBySamplingFilter)
            return;

        final EventModel eventModel = getEventModel(logEventParameter);

        if (eventModel == null) {
            AbxLog.e("repositoryImpl::logEvent error!! eventModel is null", true);
            return;
        }

        TryOptional.of(factory::createOrGetEventBuffer)
                .ifPresent(buffer -> buffer.addEventModel(eventModel));

        // prev_id를 저장해준다.
        dataRegistry.putDataRegistry(
                new DataUnit(
                        DataRegistryKey.STRING_PREV_ID,
                        logEventParameter.eventId,
                        5,
                        this.getClass().getName(),
                        true
                )
        );

        TryOptional.of(factory::getAndroidContext).ifPresent(context -> {
            boolean isEndSessionNotSentInBackgroundWhenOffline = !CoreUtils.isOnline(context) && !ABXBooleanState.getInstance().inForeground()
                    && logEventParameter.eventName.equals(CoreConstants.EVENT_END_SESSION);

            if (isEndSessionNotSentInBackgroundWhenOffline)
                saveUnsentEvents();
        });
    }

    @Override
    public void flushAllNow() {
        TryOptional.of(factory::createOrGetEventBuffer)
                .ifPresent(EventBuffer::flushAllNow);
    }

    @Override
    public void flushAtIntervals() {
        TryOptional.of(factory::createOrGetEventBuffer)
                .ifPresent(EventBuffer::flushContainerAtIntervals);
    }

    @Override
    public void saveUnsentEvents() {
        TryOptional.of(factory::createOrGetEventBuffer)
                .ifPresent(EventBuffer::saveUnsentEvents);
    }

    @Override
    public void syncApplicationConfigure() {

        if (CoreUtils.isGdprForgetMe(dataRegistry, () -> { })) return;
        if (CoreUtils.isAdbrixAllStop(dataRegistry)) {
            AbxLog.d("ADBRIX ALL STOP :: don't synchronize with server", false);
            return;
        }

        AbxLog.d("Trying to get application settings..", false);
        AbxLog.d("requestAppConfig API called!", true);
        try {
            ApiConnectionManager apiConnectionManager = new ApiConnectionManager(new ApiConnectionManager.Result() {
                @Override
                public void connectSuccess(String responseString, int responseCode) {
                    AbxLog.d("App-Config : connectSuccess ! : " + responseString, true);
                    S3ConfigHandler.getInstance().parseResponseOfS3Config(responseString, dataRegistry);
                    S3ConfigHandler.getInstance().initDefaultConfigValue(dataRegistry);
                }

                @Override
                public void connectFail(int responseCode) {
                    AbxLog.w("AppSyncConfig Failed : Can not get S3 config.json file....  responseCode : " + responseCode, true);
                }
            });

            String appKey = dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY, null);
            String urlString = String.format(CoreConstants.configRequestUrl, appKey);

            ApiConnection apiConnection = new ApiConnection(dataRegistry, androidContext)
                    .get(urlString);

            apiConnectionManager.executeWithRetry(apiConnection);

        } catch (Exception e) {
            AbxLog.e("API Error: " + e.getMessage(), false);
        }
    }

    @Override
    public void gdprForgetMe() {
        try {
            ApiConnectionManager apiConnectionManager = new ApiConnectionManager(new ApiConnectionManager.Result() {
                @Override
                public void connectSuccess(String responseString, int responseCode) {
                    AbxLog.d("GDPR connect success!!", true);
                    dataRegistry.putDataRegistry(
                            new DataUnit(
                                    DataRegistryKey.LONG_GDPR_FORGETME,
                                    1L,
                                    5,
                                    this.getClass().getName(),
                                    true
                            )
                    );

                    dataRegistry.putDataRegistry(
                            new DataUnit(
                                    DataRegistryKey.LONG_GDPR_FORGETME_SERVER_SYNC,
                                    1L,
                                    5,
                                    this.getClass().getName(),
                                    true
                            )
                    );
                }

                @Override
                public void connectFail(int responseCode) {
                    AbxLog.d("GDPR connect failed!! responseCode : " + responseCode, true);
                    dataRegistry.putDataRegistry(
                            new DataUnit(
                                    DataRegistryKey.LONG_GDPR_FORGETME,
                                    1L,
                                    5,
                                    this.getClass().getName(),
                                    true
                            )
                    );

                    dataRegistry.putDataRegistry(
                            new DataUnit(
                                    DataRegistryKey.LONG_GDPR_FORGETME_SERVER_SYNC,
                                    0L,
                                    5,
                                    this.getClass().getName(),
                                    true
                            )
                    );
                }
            });

            IApiModel gdprModel = new GDPRModelProvider(dataRegistry).provide();

            AbxLog.d("GDPRModel : " + gdprModel.getJson().toString(4), true);

            ApiConnection apiConnection = new ApiConnection(dataRegistry, androidContext)
                    .post(gdprModel);

            apiConnectionManager.executeWithRetry(apiConnection);
        } catch (Exception e) {
            AbxLog.e("API Error: " + e.getMessage(), false);
            AbxLog.e("GDPR Request Error: " + e.getMessage(), true);
        }

        dataRegistry.saveRegistry();
    }

//    @Override
//    public void updateFavoriteApplication() {
//        Timer timer = new Timer();
//
//        timer.schedule(new TimerTask() {
//            @Override
//            public void run() {
//                try {
//                    if (ABXBooleanState.getInstance().isAdbrixAllStop() ||
//                            ABXBooleanState.getInstance().isAdbrixError()){
//                        return;
//                    }
//
//                    AbxLog.i("Application Scanning... per " + ABXConstants.BATCHING_INTERVAL / 1000 + " Seconds", false);
//
//                    AbxAppScanHandler appScanHandler = new AbxAppScanHandler(dataRegistry);
//                    if (!ABXBooleanState.getInstance().isAdbrixPause()) {
//                        if (appScanHandler.getAppScanStatus()) {
//                            if (appScanHandler.checkIsTimeToAppScan())
//                                appScanHandler.abxApplicationScan(androidContext);
//                        } else {
//                            boolean userFlag = dataRegistry.safeGetLong(DataRegistryKey.LONG_APP_SCAN_ON_OFF_USER, -1) == 1;
//                            AbxLog.d("Application Scanning Failed : Server Flag = " + !S3ConfigHandler.config_appScanStop, true);
//                            AbxLog.d("Application Scanning Failed : User flag = " + userFlag, true);
//                        }
//                    }
//                } catch (Exception e){
//                    e.printStackTrace();
//                    Log.e(ABXConstants.LOGTAG, e.getMessage());
//                    timer.cancel();
//                }
//            }
//        }, 60_000, 60_000);
//    }

    @Override
    public void registerNetworkCallback() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            skDataSender = new SKDataSender(dataRegistry);

            connectivityManager = (ConnectivityManager) androidContext.getSystemService(Context.CONNECTIVITY_SERVICE);
            networkCallback = null;

            NetworkRequest.Builder builder = new NetworkRequest.Builder();
            builder.addTransportType(NetworkCapabilities.TRANSPORT_WIFI);

            NetworkRequest networkRequest = builder.build();

            networkCallback = new ConnectivityManager.NetworkCallback() {
                @Override
                public void onAvailable(Network network) {
                    super.onAvailable(network);
                    skDataSender.wifiConnectAction();
                }

                @Override
                public void onLost(Network network) {
                    super.onLost(network);
                    skDataSender.wifiDisconnectAction();
                }
            };

            connectivityManager.registerNetworkCallback(networkRequest, networkCallback);
        }
    }

    @Override
    public void unregisterNetworkCallback() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            connectivityManager = (ConnectivityManager) androidContext.getSystemService(Context.CONNECTIVITY_SERVICE);

            if (networkCallback != null)
                connectivityManager.unregisterNetworkCallback(networkCallback);
        }
    }

    @Override
    public void resetInAppMessageFrequencySession() {
        if (inAppMessageExecutorService == null || inAppMessageExecutorService.isShutdown())
            return;

        inAppMessageExecutorService.submit(() -> inAppMessageDAO.resetFrequencyCapPerSession());
    }

    @Override
    public void triggerInAppMessage(String eventName) {
        if (dataRegistry.safeGetLong(DataRegistryKey.LONG_S3_CONFIG_IN_APP_MESSAGE_ACTIVE, -1) != 1) {
            return;
        }

        if (inAppMessageExecutorService == null || inAppMessageExecutorService.isShutdown())
            return;

        if (currentActivity == null || currentActivity.isFinishing()) {
            AbxLog.d("InAppMessage:: Can't use current activity!", true);
            return;
        }

        inAppMessageExecutorService.submit(() -> {
            if (currentActivity == null || currentActivity.isFinishing()) {
                AbxLog.d("InAppMessage:: Can't use current activity!", true);
                return;
            } else if (ABXBooleanState.getInstance().isInAppMessagePresented()) {
                AbxLog.d("InAppMessage already shown", true);
                return;
            }

            InAppMessage inAppMessage;

            try {
                boolean isPortrait = currentActivity.getResources().getConfiguration().orientation == Configuration.ORIENTATION_PORTRAIT;
                inAppMessage = inAppMessageDAO.getInAppMessageByEventName(eventName, isPortrait);
                if (inAppMessage != null) {
                    IInAppMessageViewFactory inAppMessageViewFactory = inAppMessageFactoryContainer.getInAppMessageViewFactory(inAppMessage);
                    inAppMessageViewFactory.createInAppMessage(currentActivity, inAppMessage, dialog -> abxDialog = dialog);
                }
            } catch (NullPointerException nullPointerException) {
                AbxLog.e("triggerInAppMessage: nullpointerException" + Arrays.toString(nullPointerException.getStackTrace()), true);
            }
        });
    }

    @Override
    public void dismissInAppMessageDialog() {
        if (abxDialog != null) {
            if (abxDialog.isShowing())
                abxDialog.dismiss();
            abxDialog = null;
        }
    }

    @Override
    public void shutDownInAppMessageExecutor() {
        if (inAppMessageExecutorService != null)
            inAppMessageExecutorService.shutdown();
        inAppMessageDAO.closeDataBase();
    }

    @Override
    public void updateCurrentActivity(Activity activity) {
        if (inAppMessageExecutorService == null || inAppMessageExecutorService.isShutdown())
            return;

        inAppMessageExecutorService.submit(() -> {
            currentActivity = activity;
        });
    }

    @Override
    public void deleteCurrentActivity(Activity activity) {
        if (inAppMessageExecutorService == null || inAppMessageExecutorService.isShutdown())
            return;

        inAppMessageExecutorService.submit(() -> {
            if (currentActivity == activity) {
                currentActivity = null;
            }
        });
    }

    @Override
    public void connectToInAppMessageApi() {
        if (dataRegistry.safeGetLong(DataRegistryKey.LONG_S3_CONFIG_IN_APP_MESSAGE_ACTIVE, -1) != 1) {
            return;
        }

        AbxLog.d("inAppMessageApiConnection start ::: ", true);
        try {
            ApiConnectionManager apiConnectionManager = new ApiConnectionManager(new ApiConnectionManager.Result() {
                @Override
                public void connectSuccess(String responseString, int responseCode) {
                    AbxLog.d("Terminated inAppMessageApiConnection", true);
                    AbxLog.d("inAppMessageApiConnection connectSuccess !! response: " + responseString, true);

                    if (inAppMessageExecutorService == null || inAppMessageExecutorService.isShutdown())
                        return;

                    inAppMessageExecutorService.submit(() -> {
                        try {
                            inAppMessageDAO.insertApiResponse2Table(responseString);
                        } catch (JSONException e) {
                            AbxLog.d("inserApiResponse2Table jsonerror : " + e, true);
                        }
                    });
                }

                @Override
                public void connectFail(int responseCode) {
                    AbxLog.d("Terminated inAppMessageApiConnection", true);
                    AbxLog.d("inAppMessageApiConnection connectFail !! ", true);
                }
            });

            InAppMessageModel inAppMessageModel = new InAppMessageModelProvider(dataRegistry, androidContext, deviceRealtimeDataProvider).provide();

            AbxLog.d("inAppMessageRequest: " + inAppMessageModel.getJson().toString(4), true);
            AbxLog.d("inAppMessageRequest: " + inAppMessageModel.getUrlString(), true);

            ApiConnection apiConnection = new ApiConnection(dataRegistry, androidContext)
                    .post(inAppMessageModel);

            if (inAppMessageExecutorService == null || inAppMessageExecutorService.isShutdown()) {
                AbxLog.d("inAppMessageExecutorService is null or shutdowned", true);
                return;
            }

            inAppMessageExecutorService.submit(() ->
                    apiConnectionManager.executeWithRetry(apiConnection)
            );

        } catch (JSONException e) {
            AbxLog.e("inAppMessageApiConnection Request Error: " + e.getMessage(), true);
        }
    }

    @Override
    public Result<Empty> deleteUserData(String userId) {
        try {
            DRModel deleteUserDataModel = new DRModelProvider(factory,
                    userId, DRModel.OperationType.DELETE)
                    .provide();

            AbxLog.d(deleteUserDataModel.getJson().toString(4), true);
            AbxLog.d(deleteUserDataModel.getUrlString(), true);

            ApiConnection apiConnection = new ApiConnection(dataRegistry, androidContext)
                    .post(deleteUserDataModel);

            Result<ApiConnectionManager.Response> result = new ApiConnectionManager()
                    .executeWithResult(apiConnection)
                    .onSuccess(response -> {
                        AbxLog.d("Delete User Data API connect success!!", true);
                        onDeleteComplete(true);
                    })
                    .onFailure(error -> {
                        AbxLog.w("Delete User Data API connect failed!!", true);
                        onDeleteComplete(false);
                    });

            if (result.isSucceeded())
                return Success.empty();
            else
                return Error.of("Delete API Request Error");

        } catch (Exception e) {
            AbxLog.e("Delete API Request Error: " + e.getMessage(), true);
            return Error.of(e);
        }
    }

    public void onDeleteComplete(boolean isDeleteSynced) {

        if (CoreUtils.getDRState(dataRegistry) == DRState.DELETE_NOT_SYNCED && !isDeleteSynced) {
            AbxLog.e("Delete API failed again!", true);
            return;
        }

        //Restart 시 appKey, secretKey 가 필요하다.
        String appKey = dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY, null);
        String secretKey = dataRegistry.safeGetString(DataRegistryKey.STRING_SECRETKEY, null);

        //in app message 처리
        inAppMessageDAO.deleteDB();
        if (inAppMessageExecutorService != null)
            inAppMessageExecutorService.shutdownNow();
        currentActivity = null;

        //in app message 처리
        actionHistoryDAO.deleteDB();
        if (actionHistoryExecutor != null)
            actionHistoryExecutor.shutdownNow();

        //local data 삭제.
        dataRegistry.clearRegistry();

        //SKDataSender 중지
        unregisterNetworkCallback();

        //다음 SDK 시작시에 stop state를 판단하기 위한 flag 저장.
        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.BOOLEAN_IS_SDK_STOPPED,
                true,
                5,
                this.getClass().getName(),
                true
        ));

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.BOOLEAN_IS_SDK_STOPPED_SERVER_SYNC,
                isDeleteSynced,
                5,
                this.getClass().getName(),
                true
        ));

        //appKey, secretKey 저장
        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_APPKEY,
                appKey,
                5,
                this.getClass().getName(),
                true
        ));

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_SECRETKEY,
                secretKey,
                5,
                this.getClass().getName(),
                true
        ));

        dataRegistry.saveRegistry();
    }

    @Override
    public Result<Empty> restartSDK(String userId) {
        try {
            DRModel restartModel = getRestartApiModel(userId);

            AbxLog.d(restartModel.getJson().toString(4), true);
            AbxLog.d(restartModel.getUrlString(), true);

            ApiConnection apiConnection = new ApiConnection(dataRegistry, androidContext)
                    .post(restartModel);

            Result<ApiConnectionManager.Response> result = new ApiConnectionManager()
                    .executeWithResult(apiConnection)
                    .onSuccess(response -> {
                        AbxLog.d("Restart SDK API connect success!!", true);
                        onRestartComplete(true);
                    })
                    .onFailure(error -> {
                        AbxLog.w("Restart SDK API connect failed!!", true);
                        onRestartComplete(false);
                    });

            if (result.isSucceeded())
                return Success.empty();
            else
                return Error.of("Restart API Request Error");

        } catch (Exception e) {
            AbxLog.e("Restart API Request Error: " + e.getMessage(), true);
            return Error.of(e);
        }
    }

    public DRModel getRestartApiModel(String userId) {

        if (CoreUtils.getDRState(dataRegistry) != DRState.NORMAL) {
            DRModelProvider drModelProvider = new DRModelProvider(factory, userId, DRModel.OperationType.INITIALIZE);
            //restart - initialize
            if (CoreUtils.getDRState(dataRegistry) == DRState.INIT_RESTART_NOT_SYNCED) {
                return drModelProvider.getNotSyncedInitRestartModel();
            } else {
                return drModelProvider.provide();
            }
        } else {
            //duplicated restart
            return new DRModelProvider(factory, userId, DRModel.OperationType.RESTART).provide();
        }
    }

    public void onRestartComplete(boolean isRestartSynced) {
        if (CoreUtils.getDRState(dataRegistry) == DRState.INIT_RESTART_NOT_SYNCED && !isRestartSynced) {
            AbxLog.e("Initialize restart failed again!", true);
            return;
        }

        if (isRestartSynced) {
            //Init Restart 성공했으므로 Init Restart Event Datetime 삭제.
            dataRegistry.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.STRING_INIT_RESTART_EVENT_DATETIME,
                            null,
                            5,
                            this.getClass().getName(),
                            true
                    )
            );

            if (CoreUtils.getDRState(dataRegistry) != DRState.NORMAL) {
                inAppMessageDAO.restartInAppMessage();
                inAppMessageExecutorService = Executors.newSingleThreadExecutor();

                actionHistoryDAO.restartDB();
                actionHistoryExecutor = Executors.newSingleThreadExecutor();
            }
        }

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.BOOLEAN_IS_SDK_STOPPED,
                false,
                5,
                this.getClass().getName(),
                true
        ));

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.BOOLEAN_IS_SDK_STOPPED_SERVER_SYNC,
                !isRestartSynced,
                5,
                this.getClass().getName(),
                true
        ));

        dataRegistry.saveRegistry();
    }

    @Override
    public void fetchActionHistoryFromServer(@Nullable String token, ActionHistoryIdType idType, List<String> actionType, Completion<Result<List<ActionHistory>>> completion) {
        if (actionHistoryExecutor == null || actionHistoryExecutor.isShutdown()) {
            completion.handle(Error.of("actionHistoryExecutor is not available now!"));
            return;
        }

        actionHistoryExecutor.submit(() -> {
            Result<Pair<String, Integer>> result = null;
            Map<String, String> header = new HashMap<>();

            ActionHistoryQueryModel actionHistoryQueryModel = new ActionHistoryQueryModelProvider(
                    dataRegistry,
                    androidContext,
                    deviceRealtimeDataProvider,
                    factory,
                    idType,
                    actionType,
                    0,
                    0,
                    0
            ).provide();

            try {
                AbxLog.d(actionHistoryQueryModel.getJson().toString(4), true);
                AbxLog.d(actionHistoryQueryModel.getUrlString(), true);
            } catch (JSONException e) {
                e.printStackTrace();
            }

            if (token != null)
                header.put("Authorization", token);

            ApiConnection apiConnection = new ApiConnection(dataRegistry, androidContext)
                    .header(header)
                    .post(actionHistoryQueryModel);

            new ApiConnectionManager()
                    .executeWithResult(apiConnection)
                    .onSuccess(response -> {
                        if (response == null) {
                            completion.handle(ActionHistoryError.NULL_RESPONSE_ERROR.getError());
                            return;
                        }

                        JSONArray actionHistoryJSONArray = getActionHistoryJSONArray(response.message);

                        AbxLog.d(actionHistoryJSONArray.toString(), true);

                        actionHistoryDAO.insertActionHistory(actionHistoryJSONArray);

                        List<ActionHistory> actionHistories = new ArrayList<>();
                        for (int i = 0; i < actionHistoryJSONArray.length(); i++) {
                            JSONObject actionHistoryJSONObject = actionHistoryJSONArray.optJSONObject(i);
                            actionHistories.add(ActionHistory.fromJSONObject(actionHistoryJSONObject, true));
                        }

                        completion.handle(Success.of(actionHistories));
                    })
                    .onFailure(error -> {
                        completion.handle(ActionHistoryError.SYNC_SERVER_ERROR.getError());
                    });

            actionHistoryDAO.getLastServerTimeStamp().onSuccess(lastTimestamp -> {
                AbxLog.d("getLastServerTimeStamp : " + lastTimestamp, true);
                actionHistoryDAO.deleteOutdatedPushData(lastTimestamp);
            }).onFailure(it -> {
                AbxLog.d("getLastServerTimeStamp failed!", true);
            });
        });
    }

    private JSONArray getActionHistoryJSONArray(String response) {
        try {
            JSONObject responseJson = new JSONObject(response);

            return responseJson.optJSONArray(CoreConstants.ACTION_HISTORY_RESPONSE_HISTORIES);
        } catch (JSONException e) {
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
            return new JSONArray();
        }
    }

    @Override
    public void insertPushData(String pushDataString) {
        if (actionHistoryExecutor == null || actionHistoryExecutor.isShutdown())
            return;

        actionHistoryExecutor.submit(() -> actionHistoryDAO.insertPushData(pushDataString));
    }

    @Override
    public void getActionHistory(int skip, int limit, List<String> actionType, Completion<Result<List<ActionHistory>>> completion) {
        if (actionHistoryExecutor == null || actionHistoryExecutor.isShutdown()) {
            completion.handle(Error.of("actionHistoryExecutor is not available now!"));
            return;
        }

        actionHistoryExecutor.submit(() -> {
            completion.handle(actionHistoryDAO.getActionHistory(skip, limit, actionType));
        });
    }

    @Override
    public void getAllActionHistory(List<String> actionType, Completion<Result<List<ActionHistory>>> completion) {
        if (actionHistoryExecutor == null || actionHistoryExecutor.isShutdown()) {
            completion.handle(Error.of("actionHistoryExecutor is not available now!"));
            return;
        }

        actionHistoryExecutor.submit(() -> {
            completion.handle(actionHistoryDAO.getAllActionHistory(actionType));
        });
    }

    @Override
    public void deleteActionHistory(@Nullable String token, String historyId, long timestamp, Completion<Result<Empty>> completion) {
        if (actionHistoryExecutor == null || actionHistoryExecutor.isShutdown()) {
            completion.handle(Error.of("actionHistoryExecutor is not available now!"));
            return;
        }

        actionHistoryExecutor.submit(() -> {
            if (!actionHistoryDAO.deleteActionHistory(historyId, timestamp).isSucceeded()) {
                completion.handle(ActionHistoryError.SQLITE_QUERY_ERROR.getError());
                return;
            }

            ActionHistoryDeleteModel actionHistoryDeleteModel = null;
            try {
                actionHistoryDeleteModel = new ActionHistoryDeleteModelProvider(
                        dataRegistry,
                        androidContext,
                        deviceRealtimeDataProvider,
                        factory,
                        null,
                        historyId,
                        timestamp
                ).provide();
            } catch (Exception e) {
                completion.handle(Error.of(e));
            }

            if (actionHistoryDeleteModel == null) {
                completion.handle(ActionHistoryError.NULL_ACTION_HISTORY_API_MODEL_ERROR.getError());
                return;
            }

            Map<String, String> header = new HashMap<>();
            if (token != null)
                header.put("Authorization", token);

            try {
                AbxLog.d(actionHistoryDeleteModel.getJson().toString(4), true);
                AbxLog.d(actionHistoryDeleteModel.getUrlString(), true);
            } catch (JSONException e) {
                e.printStackTrace();
            }

            ApiConnection apiConnection = new ApiConnection(dataRegistry, androidContext)
                    .header(header)
                    .delete(actionHistoryDeleteModel);

            new ApiConnectionManager()
                    .executeWithResult(apiConnection)
                    .onSuccess(result -> {
                        AbxLog.d("deleteActionHistory connect success!!", true);
                        completion.handle(Success.empty());
                    })
                    .onFailure(result -> {
                        AbxLog.w("deleteActionHistory connect failed!!", true);
                        completion.handle(ActionHistoryError.DELETE_REMOTE_DB_FAIL_ERROR.getError());
                    });
        });
    }

    @Override
    public void deleteAllActionHistory(@Nullable String token, @Nullable ActionHistoryIdType idType, Completion<Result<Empty>> completion) {
        if (actionHistoryExecutor == null || actionHistoryExecutor.isShutdown()) {
            completion.handle(Error.of("actionHistoryExecutor is not available now!"));
            return;
        }

        actionHistoryExecutor.submit(() -> {
            if (!actionHistoryDAO.clearSyncedActionHistoryInLocalDB().isSucceeded()) {
                completion.handle(ActionHistoryError.SQLITE_QUERY_ERROR.getError());
                return;
            }

            ActionHistoryDeleteModel actionHistoryDeleteModel = null;
            try {
                actionHistoryDeleteModel = new ActionHistoryDeleteModelProvider(
                        dataRegistry,
                        androidContext,
                        deviceRealtimeDataProvider,
                        factory,
                        idType,
                        null,
                        0
                ).provide();
            } catch (Exception e) {
                completion.handle(Error.of(e));
            }

            if (actionHistoryDeleteModel == null) {
                completion.handle(ActionHistoryError.NULL_ACTION_HISTORY_API_MODEL_ERROR.getError());
                return;
            }

            Map<String, String> header = new HashMap<>();
            if (token != null)
                header.put("Authorization", token);

            ApiConnection apiConnection = new ApiConnection(dataRegistry, androidContext)
                    .header(header)
                    .delete(actionHistoryDeleteModel);

            try {
                AbxLog.d(actionHistoryDeleteModel.getJson().toString(4), true);
                AbxLog.d(actionHistoryDeleteModel.getUrlString(), true);
            } catch (JSONException e) {
                e.printStackTrace();
            }

            new ApiConnectionManager()
                    .executeWithResult(apiConnection)
                    .onSuccess(response -> {
                        AbxLog.d("deleteAllActionHistory connect success!!", true);
                        completion.handle(Success.empty());
                    })
                    .onFailure(error -> {
                        AbxLog.d("deleteAllActionHistory connect failed!!", true);
                        completion.handle(ActionHistoryError.DELETE_REMOTE_DB_FAIL_ERROR.getError());
                    });
        });
    }

    @Override
    public void clearSyncedActionHistoryInLocalDB(Completion<Result<Empty>> completion) {
        if (actionHistoryExecutor == null || actionHistoryExecutor.isShutdown()) {
            completion.handle(Error.of("actionHistoryExecutor is not available now!"));
            return;
        }

        actionHistoryExecutor.submit(() -> {
            completion.handle(actionHistoryDAO.clearSyncedActionHistoryInLocalDB());
        });
    }

    @Override
    public void clearAllActionHistoryInLocalDB() {
        if (actionHistoryExecutor == null || actionHistoryExecutor.isShutdown())
            return;

        actionHistoryExecutor.submit(() -> {
            actionHistoryDAO.clearAllActionHistoryInLocalDB()
                    .onSuccess(empty -> AbxLog.d("clearAllActionHistoryInLocalDB onSuccess!", true))
                    .onFailure(error -> AbxLog.d("clearAllActionHistoryInLocalDB onFailure: " + error.getMessage(), true));
        });
    }
}
