package io.adbrix.sdk.data.net;

import java.util.Arrays;
import java.util.Timer;
import java.util.TimerTask;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.configuration.DefaultABXContextController;
import io.adbrix.sdk.data.ABXBooleanState;
import io.adbrix.sdk.domain.model.Error;
import io.adbrix.sdk.domain.model.Success;

public class ApiConnectionManager {
    private static final int MAX_RETRY = 3;
    private Result result;
    private int attempt = 0;
    private Timer retryTimer;

    public ApiConnectionManager(Result result) {
        if (result == null)
            this.result = new NullResult();
        else this.result = result;
    }

    public ApiConnectionManager() {
        this.result = new NullResult();
    }

    public void executeWithRetry(ApiConnection apiConnection) {

        String response = null;
        try {
            response = apiConnection.request();
        } catch (Exception e) {
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
            return;
        }

        if (apiConnection.isHttpOK()) {
            if (response != null && response.equals("{\"deeplink\":null}")) {
                retry(apiConnection);
            } else
                result.connectSuccess(response, apiConnection.getResponseCode());
        } else if (apiConnection.isWrongAppkey()) {
            // HTTP 500: wrong appKey
            ABXBooleanState.getInstance().adbrixError.getAndSet(true);
            AbxLog.w("Warning!! :: wrong appkey : " + response, true);

            retry(apiConnection);
        } else if (apiConnection.isInvalidAppkey()) {
            //AppKey ERR
            ABXBooleanState.getInstance().adbrixError.getAndSet(true);
            AbxLog.w("Warning!! :: Appkey is not valid. " + response, true);
            // not retry
            AbxLog.d("Broken connection to AdBrixRm. Skip retry uploading events. " + response, true);

            DefaultABXContextController.getInstance().disableSDK("AdbrixAllStop");
            result.connectFail(apiConnection.getResponseCode());
        } else {
            // not retry
            AbxLog.d("Broken connection to AdBrixRm. Skip retry uploading events. " + response, true);
            result.connectFail(apiConnection.getResponseCode());
        }
    }

    public void execute(ApiConnection apiConnection) {

        String response = null;
        try {
            response = apiConnection.request();
        } catch (Exception e) {
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
            return;
        }

        if (apiConnection.isHttpOK()) {
            result.connectSuccess(response, apiConnection.getResponseCode());
        } else if (apiConnection.isWrongAppkey()) {
            // HTTP 500: wrong appKey
            ABXBooleanState.getInstance().adbrixError.getAndSet(true);
            AbxLog.w("Warning!! :: wrong appkey : " + response, true);
            result.connectFail(apiConnection.getResponseCode());
        } else if (apiConnection.isInvalidAppkey()) {
            //AppKey ERR
            ABXBooleanState.getInstance().adbrixError.getAndSet(true);
            AbxLog.w("Warning!! :: Appkey is not valid. " + response, true);
            // not retry
            AbxLog.w("Broken connection to AdBrixRm. Skip retry uploading events. " + response, true);

            DefaultABXContextController.getInstance().disableSDK("AdbrixAllStop");
            result.connectFail(apiConnection.getResponseCode());
        } else {
            // not retry
            AbxLog.d("Broken connection to AdBrixRm. Skip retry uploading events. " + response, true);
            result.connectFail(apiConnection.getResponseCode());
        }
    }

    public io.adbrix.sdk.domain.model.Result<Response> executeWithResult(ApiConnection apiConnection) {

        String response = null;
        try {
            response = apiConnection.request();
        } catch (Exception e) {
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
            return Error.of(e);
        }

        if (apiConnection.isHttpOK()) {
            return Success.of(new Response(apiConnection.getResponseCode(), response));
        } else if (apiConnection.isWrongAppkey()) {
            // HTTP 500: wrong appKey
            ABXBooleanState.getInstance().adbrixError.getAndSet(true);
            AbxLog.w("Warning!! :: wrong appkey : " + response, true);
            return Error.of(response);
        } else if (apiConnection.isInvalidAppkey()) {
            //AppKey ERR
            ABXBooleanState.getInstance().adbrixError.getAndSet(true);
            AbxLog.w("Warning!! :: Appkey is not valid. " + response, true);
            // not retry
            AbxLog.w("Broken connection to AdBrixRm. Skip retry uploading events. " + response, true);

            DefaultABXContextController.getInstance().disableSDK("AdbrixAllStop");
            return Error.of(response);
        } else {
            // not retry
            AbxLog.d("Broken connection to AdBrixRm. Skip retry uploading events. " + response, true);
            return Error.of(response);
        }
    }

    private void retry(ApiConnection apiConnection) {
        if (attempt >= MAX_RETRY) {
            if (retryTimer != null) {
                retryTimer.cancel();
            }
            AbxLog.d("ApiConnectionManager retry failed. Skip retry uploading events.", true);
            result.connectFail(apiConnection.getResponseCode());
            return;
        }

        retryTimer = new Timer();

        retryTimer.schedule(new TimerTask() {
            @Override
            public void run() {
                AbxLog.d("ApiConnectionManager retry : (" + (attempt + 1) + "/3)", true);
                attempt++;
                executeWithRetry(apiConnection);
            }
        }, 10_000);
    }

    public static class Response {
        public final int code;
        public final String message;

        public Response(int code, String message) {
            this.code = code;
            this.message = message;
        }
    }

    public interface Result {
        void connectSuccess(String responseString, int responseCode);

        void connectFail(int responseCode);
    }

    private static class NullResult implements Result {
        @Override
        public void connectSuccess(String responseString, int responseCode) {
            //do nothing
        }

        @Override
        public void connectFail(int responseCode) {
            //do nothing
        }
    }
}
