package io.adbrix.sdk.data.inAppMessage;

import android.app.Activity;
import android.content.ActivityNotFoundException;
import android.content.Intent;
import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.graphics.Color;
import android.net.Uri;
import android.os.Build;
import android.os.CountDownTimer;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;
import android.widget.FrameLayout;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import io.adbrix.sdk.R;
import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.IObserver;
import io.adbrix.sdk.component.InAppMessageClickPostingObservable;
import io.adbrix.sdk.configuration.DefaultABXContextController;
import io.adbrix.sdk.data.ABXBooleanState;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.utils.CommonUtils;

import static android.view.View.GONE;
import static android.view.View.VISIBLE;

public class StickyBannerMessageViewFactory implements IInAppMessageViewFactory {
    private final String alignTop = "top";
    private List<IObserver<LogEventParameter>> observers = new ArrayList<>();
    private InAppMessageDAO inAppMessageDAO;
    private CountDownTimer imageCountDownTimer;

    public StickyBannerMessageViewFactory(InAppMessageDAO inAppMessageDAO) {
        add(DefaultABXContextController.getInstance());
        this.inAppMessageDAO = inAppMessageDAO;
    }

    @Override
    public void createInAppMessage(Activity activity, InAppMessage inAppMessage, IObserver<AbxDialog> abxDialogIObserver) {
        LayoutInflater inflater = activity.getLayoutInflater();

        View view = inflater.inflate(R.layout.layout_slideup_message, null);

        activity.runOnUiThread(() -> {
            if (ABXBooleanState.getInstance().isInAppMessagePresented()) {
                AbxLog.d("InAppMessage already shown", true);
                return;
            }

            try {
                AbxDialog dialog = new AbxDialog(activity);

                dialog.setContentView(view);
                dialog.setCanceledOnTouchOutside(false);

                dialog.setOnCancelListener(v -> {
                    setDialogTerminateState(abxDialogIObserver);
                });

                dialog.setOnDismissListener(v -> {
                    setDialogTerminateState(abxDialogIObserver);
                });

                boolean isPortrait = activity.getResources().getConfiguration().orientation == Configuration.ORIENTATION_PORTRAIT;

                setImage(view, isPortrait, inAppMessage, dialog, activity);

                Runnable onClickRunnable = () -> AbxLog.d("StickyBanner : unknown button action", true);

                if (ActionType.CLOSE.getValue().equals(inAppMessage.getStickBannerClickActionActionType())) {
                    onClickRunnable = () -> {
                        dialog.dismiss();

                        HashMap<String, Object> eventParam = new HashMap<>();
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_EVENT_TYPE_ABX_KEY, EventType.CLICK.getValue());
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_CAMPAIGN_ID_KEY, inAppMessage.getCampaignId());
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_ACTION_ID_ABX_KEY, inAppMessage.getStickBannerClickActionActionID());
                        notifyObserver(new LogEventParameter(
                                CoreConstants.GROUP_ABX,
                                CoreConstants.EVENT_IN_APP_MESSAGE,
                                CommonUtils.parseValueWithDataType(eventParam, CommonUtils.FixType.PREFIX),
                                0,
                                0
                        ));

                        InAppMessageClickPostingObservable.getInstance().postInAppMessageClickToMainThread(
                                inAppMessage.getStickBannerClickActionActionID(),
                                inAppMessage.getStickBannerClickActionActionType(),
                                inAppMessage.getStickBannerClickOptionActionArg(),
                                true
                        );
                    };
                } else if (
                        ActionType.DEEPLINK_AND_CLOSE.getValue().equals(inAppMessage.getStickBannerClickActionActionType())
                                || ActionType.WEBLINK.getValue().equals(inAppMessage.getStickBannerClickActionActionType())
                                || ActionType.WEBLINK_AND_CLOSE.getValue().equals(inAppMessage.getStickBannerClickActionActionType())) {
                    onClickRunnable = () -> {
                        String url = inAppMessage.getStickBannerClickOptionActionArg();
                        Intent intent = new Intent(Intent.ACTION_VIEW, Uri.parse(url));

                        HashMap<String, Object> eventParam = new HashMap<>();
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_EVENT_TYPE_ABX_KEY, EventType.CLICK.getValue());
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_CAMPAIGN_ID_KEY, inAppMessage.getCampaignId());
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_ACTION_ID_ABX_KEY, inAppMessage.getStickBannerClickActionActionID());
                        notifyObserver(new LogEventParameter(
                                CoreConstants.GROUP_ABX,
                                CoreConstants.EVENT_IN_APP_MESSAGE,
                                CommonUtils.parseValueWithDataType(eventParam, CommonUtils.FixType.PREFIX),
                                0,
                                0
                        ));

                        try {
                            view.getContext().startActivity(intent);
                        } catch (ActivityNotFoundException exception) {
                            AbxLog.d("ModalMessageViewFactory: ActivityNotFoundException, button action url was " + url, true);
                        }

                        InAppMessageClickPostingObservable.getInstance().postInAppMessageClickToMainThread(
                                inAppMessage.getStickBannerClickActionActionID(),
                                inAppMessage.getStickBannerClickActionActionType(),
                                inAppMessage.getStickBannerClickOptionActionArg(),
                                true
                        );
                    };
                } else {
                    AbxLog.d("StickyBannerMessageViewFactory: unknown button action '" + inAppMessage.getStickBannerClickActionActionType() + "'", true);
                }

                setMessage(view, inAppMessage);

                if (alignTop.equals(inAppMessage.getStickyBannerAlign()))
                    setOnTop(dialog, onClickRunnable, abxDialogIObserver);
                else
                    setOnBottom(dialog, onClickRunnable, abxDialogIObserver);

                dialog.show();

                if (!dialog.isShowing()) {
                    return;
                }

                HashMap<String, Object> eventParam = new HashMap<>();
                eventParam.put(CoreConstants.IN_APP_MESSAGE_EVENT_TYPE_ABX_KEY, EventType.IMPRESSION.getValue());
                eventParam.put(CoreConstants.IN_APP_MESSAGE_CAMPAIGN_ID_KEY, inAppMessage.getCampaignId());

                if (isPortrait) {
                    if (inAppMessage.getPortraitImage() != null) {
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_URL_ABX_KEY, inAppMessage.getPortraitImageURL());
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_IS_IMAGE_DOWNLOADED_ABX_KEY, inAppMessage.getsIsPortraitImageDownloaded());
                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                            eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_SIZE_ABX_KEY, inAppMessage.getPortraitImage().getAllocationByteCount());
                        } else {
                            eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_SIZE_ABX_KEY, inAppMessage.getPortraitImage().getByteCount());
                        }
                    }
                } else {
                    if (inAppMessage.getLandScapeImage() != null) {
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_URL_ABX_KEY, inAppMessage.getLandscapeImageURL());
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_IS_IMAGE_DOWNLOADED_ABX_KEY, inAppMessage.getsIsLandscapeImageDownloaded());
                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                            eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_SIZE_ABX_KEY, inAppMessage.getLandScapeImage().getAllocationByteCount());
                        } else {
                            eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_SIZE_ABX_KEY, inAppMessage.getLandScapeImage().getByteCount());
                        }
                    }
                }

                notifyObserver(new LogEventParameter(
                        CoreConstants.GROUP_ABX,
                        CoreConstants.EVENT_IN_APP_MESSAGE,
                        CommonUtils.parseValueWithDataType(eventParam, CommonUtils.FixType.PREFIX),
                        0,
                        0
                ));

                abxDialogIObserver.update(dialog);
            } catch (NullPointerException nullPointerException) {
                AbxLog.d("CreateInAppMessage : nullPointerException\n " + nullPointerException, true);
            }
        });
    }

    private void setImage(View view, boolean isPortrait, InAppMessage inAppMessage, AbxDialog dialog, Activity activity) {
        Bitmap bitmapImage = isPortrait ? inAppMessage.getPortraitImage() : inAppMessage.getLandScapeImage();

        ImageView imageView = view.findViewById(R.id.com_adbrix_inappmessage_slideup_imageview);
        FrameLayout progressBarLayout = view.findViewById(R.id.com_adbirx_inappmessage_slideup_progressbar_layout);
        FrameLayout defaultImageViewLayout = view.findViewById(R.id.com_adbrix_inappmessage_slideup_default_imageview_layout);

        if (bitmapImage == null) {
            progressBarLayout.setVisibility(VISIBLE);
            imageView.setVisibility(GONE);
            defaultImageViewLayout.setVisibility(GONE);

            imageCountDownTimer = new CountDownTimer(9_000, 3_000) {
                @Override
                public void onTick(long millisUntilFinished) {
                    if (dialog != null && dialog.isShowing()) {

                        inAppMessageDAO.imageDownloadExecutorService.submit(() -> {
                            if (imageCountDownTimer == null)
                                return;

                            HashMap<String, Object> imageHashMap = inAppMessageDAO.getAndSaveImageInfo(inAppMessage.getCampaignId(), isPortrait ? InAppMessageDAO.PORTRAIT : InAppMessageDAO.LANDSCAPE, true);

                            if (imageHashMap != null) {
                                if (isPortrait) {
                                    inAppMessage.setPortraitImage((Bitmap) imageHashMap.get(ImageInfoHashmapValue.IMAGE_BITMAP.getValue()));
                                    inAppMessage.setPortraitImageURL((String) imageHashMap.get(ImageInfoHashmapValue.IMAGE_URL.getValue()));
                                    inAppMessage.setIsPortraitImageDownloaded((Boolean) imageHashMap.get(ImageInfoHashmapValue.IS_DOWNLOADED.getValue()));
                                } else {
                                    inAppMessage.setLandscapeImage((Bitmap) imageHashMap.get(ImageInfoHashmapValue.IMAGE_BITMAP.getValue()));
                                    inAppMessage.setLandscapeImageURL((String) imageHashMap.get(ImageInfoHashmapValue.IMAGE_URL.getValue()));
                                    inAppMessage.setIsLandscapeImageDownloaded((Boolean) imageHashMap.get(ImageInfoHashmapValue.IS_DOWNLOADED.getValue()));
                                }

                                inAppMessage.setImageClickAction((String) imageHashMap.get(ImageInfoHashmapValue.CLICK_ACTION.getValue()));

                                activity.runOnUiThread(() -> setImage(view, isPortrait, inAppMessage, dialog, activity));

                                if (imageCountDownTimer != null) {
                                    imageCountDownTimer.cancel();
                                    imageCountDownTimer = null;
                                }
                            }
                        });
                    }
                }

                @Override
                public void onFinish() {
                    if (dialog != null && dialog.isShowing()) {
                        activity.runOnUiThread(() -> {
                            defaultImageViewLayout.setVisibility(VISIBLE);
                            progressBarLayout.setVisibility(GONE);
                            imageView.setVisibility(GONE);
                        });

                        defaultImageViewLayout.setOnClickListener(v -> activity.runOnUiThread(() ->
                                setImage(view, isPortrait, inAppMessage, dialog, activity)
                        ));
                    }
                    this.cancel();
                }
            }.start();

            return;
        } else {
            if (imageCountDownTimer != null) {
                imageCountDownTimer.cancel();
                imageCountDownTimer = null;
            }
            progressBarLayout.setVisibility(GONE);
            imageView.setVisibility(VISIBLE);
            defaultImageViewLayout.setVisibility(GONE);
        }

        imageView.setImageBitmap(bitmapImage);

        if (inAppMessage.getImageClickActionId() != null) {
            imageView.setOnClickListener(new IInAppMessageViewFactory.InAppMessageOnClickListener(
                    inAppMessage.getCampaignId(),
                    inAppMessage.getImageClickActionId(),
                    inAppMessage.getImageClickActionType(),
                    inAppMessage.getImageClickActionArg(),
                    dialog,
                    () -> {
                        HashMap<String, Object> eventParam = new HashMap<>();
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_ACTION_ID_ABX_KEY, inAppMessage.getImageClickActionId());
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_EVENT_TYPE_ABX_KEY, EventType.CLICK.getValue());
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_CAMPAIGN_ID_KEY, inAppMessage.getCampaignId());
                        notifyObserver(new LogEventParameter(
                                CoreConstants.GROUP_ABX,
                                CoreConstants.EVENT_IN_APP_MESSAGE,
                                CommonUtils.parseValueWithDataType(eventParam, CommonUtils.FixType.PREFIX),
                                0,
                                0
                        ));
                        if (
                                ActionType.CLOSE.getValue().equals(inAppMessage.getImageClickActionType())
                                        || ActionType.WEBLINK_AND_CLOSE.getValue().equals(inAppMessage.getImageClickActionType())
                                        || ActionType.DEEPLINK_AND_CLOSE.getValue().equals(inAppMessage.getImageClickActionType())
                                        || ActionType.DONT_SHOW_ME_TODAY_AND_CLOSE.getValue().equals(inAppMessage.getImageClickActionType())
                        ) {
                            if (imageCountDownTimer != null) {
                                imageCountDownTimer.cancel();
                                imageCountDownTimer = null;
                            }
                        }
                    },
                    inAppMessageDAO
            ));
        }
    }

    private void setOnTop(AbxDialog dialog, Runnable onClickRunnable, IObserver<AbxDialog> abxDialogIObserver) {
        Window window = dialog.getWindow();
        window.getAttributes().windowAnimations = R.style.TopSlideUpAnimation;

        window.getDecorView().setOnTouchListener(new SwipeListener() {
            @Override
            public void onSwipeLeft(MotionEvent motionEvent) {
                if (dialog != null)
                    dialog.cancel();
                setDialogTerminateState(abxDialogIObserver);
            }

            @Override
            public void onSwipeRight(MotionEvent motionEvent) {
                if (dialog != null)
                    dialog.cancel();
                setDialogTerminateState(abxDialogIObserver);
            }

            @Override
            public void onClick() {
                super.onClick();
                if (onClickRunnable != null)
                    onClickRunnable.run();
                if (dialog != null)
                    dialog.cancel();
                setDialogTerminateState(abxDialogIObserver);
            }
        });

        WindowManager.LayoutParams wlp = window.getAttributes();

        wlp.gravity = Gravity.TOP;
        wlp.flags &= ~WindowManager.LayoutParams.FLAG_DIM_BEHIND;

        window.setFlags(WindowManager.LayoutParams.FLAG_NOT_TOUCH_MODAL,
                WindowManager.LayoutParams.FLAG_NOT_TOUCH_MODAL);
        window.setLayout(WindowManager.LayoutParams.MATCH_PARENT, WindowManager.LayoutParams.WRAP_CONTENT);

    }

    private void setOnBottom(AbxDialog dialog, Runnable onClickRunnable, IObserver<AbxDialog> abxDialogIObserver) {
        Window window = dialog.getWindow();
        window.getAttributes().windowAnimations = R.style.BottomSlideUpAnimation;

        window.getDecorView().setOnTouchListener(new SwipeListener() {
            @Override
            public void onSwipeLeft(MotionEvent motionEvent) {
                if (dialog != null)
                    dialog.cancel();
                setDialogTerminateState(abxDialogIObserver);
            }

            @Override
            public void onSwipeRight(MotionEvent motionEvent) {
                if (dialog != null)
                    dialog.cancel();
                setDialogTerminateState(abxDialogIObserver);
            }

            @Override
            public void onClick() {
                super.onClick();
                if (onClickRunnable != null)
                    onClickRunnable.run();
                if (dialog != null)
                    dialog.cancel();
                setDialogTerminateState(abxDialogIObserver);
            }
        });

        WindowManager.LayoutParams wlp = window.getAttributes();

        wlp.gravity = Gravity.BOTTOM;
        wlp.flags &= ~WindowManager.LayoutParams.FLAG_DIM_BEHIND;
        window.setFlags(WindowManager.LayoutParams.FLAG_NOT_TOUCH_MODAL,
                WindowManager.LayoutParams.FLAG_NOT_TOUCH_MODAL);
        window.setLayout(WindowManager.LayoutParams.MATCH_PARENT, WindowManager.LayoutParams.WRAP_CONTENT);

    }

    private void setMessage(View view, InAppMessage inAppMessage) {
        TextView textView = view.findViewById(R.id.com_adbrix_inappmessage_slideup_textview);
        textView.setText(inAppMessage.getBodyText());
        textView.setTextColor(Color.parseColor(inAppMessage.getBodyTextColor()));

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            if (TextAlign.CENTER.getValue().equals(inAppMessage.getTextAlign())) {
                textView.setTextAlignment(View.TEXT_ALIGNMENT_CENTER);
            } else if (TextAlign.RIGHT.getValue().equals(inAppMessage.getTextAlign())) {
                textView.setTextAlignment(View.TEXT_ALIGNMENT_VIEW_END);
            } else if (TextAlign.LEFT.getValue().equals(inAppMessage.getTextAlign())) {
                textView.setTextAlignment(View.TEXT_ALIGNMENT_VIEW_START);
            }
        }


        LinearLayout frame = view.findViewById(R.id.com_adbrix_inappmessage_slideup_frame);
        frame.setBackgroundColor(Color.parseColor(inAppMessage.getBgColor()));
    }

    private void setDialogTerminateState(IObserver<AbxDialog> abxDialogIObserver) {
        ABXBooleanState.getInstance().isInAppMessagePresented.getAndSet(false);
        if (imageCountDownTimer != null) {
            imageCountDownTimer.cancel();
            imageCountDownTimer = null;
        }
        if (abxDialogIObserver != null)
            abxDialogIObserver.update(null);
    }

    @Override
    public void add(IObserver<LogEventParameter> observer) {
        observers.add(observer);
    }

    @Override
    public void delete(IObserver<LogEventParameter> observer) {
        observers.remove(observer);
    }

    @Override
    public void notifyObserver(LogEventParameter logEventParameter) {
        for (IObserver<LogEventParameter> observer : observers) {
            observer.update(logEventParameter);
        }
    }
}
