package io.adbrix.sdk.data.inAppMessage;

import android.content.ContentValues;
import android.content.Context;
import android.content.ContextWrapper;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.util.Pair;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Locale;
import java.util.Queue;
import java.util.TimeZone;
import java.util.UUID;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;

public class InAppMessageDAO {
    public static final String RESPONSE_DATA = "data";
    public static final String RESPONSE_CHECKSUM = "checksum";
    public static final String RESPONSE_STATUS = "status";
    public static final String RESPONSE_MINUTES_TO_EXPIRY = "minutes_to_expiry";
    public static final String RESPONSE_IAM = "in_app_messages";
    public static final String RESPONSE_IAM_CAMPAIGN_ID = "campaign_id";
    public static final String RESPONSE_IAM_PAGE_ORIENTATION = "page_orientation";
    public static final String RESPONSE_IAM_TYPE = "type";
    public static final String RESPONSE_IAM_LAYOUT = "layout";
    public static final String RESPONSE_IAM_CONTENTS = "contents";
    public static final String RESPONSE_IAM_CONTENTS_IMAGE = "image";
    public static final String RESPONSE_IAM_CONTENTS_IMAGE_CLICK_ACTION = "click_action";
    public static final String RESPONSE_IAM_CONTENTS_IMAGE_CLICK_ACTION_ACTION_ID = "action_id";
    public static final String RESPONSE_IAM_CONTENTS_IMAGE_CLICK_ACTION_ACTION_TYPE = "action_type";
    public static final String RESPONSE_IAM_CONTENTS_IMAGE_CLICK_ACTION_ACTION_ARG = "action_arg";
    public static final String RESPONSE_IAM_CONTENTS_IMAGE_URL = "url";
    public static final String RESPONSE_IAM_CONTENTS_IMAGE_URL_PORTRAIT = "portrait";
    public static final String RESPONSE_IAM_CONTENTS_IMAGE_URL_LANDSCAPE = "landscape";
    public static final String RESPONSE_IAM_CONTENTS_BG_STYLE = "bg_style";
    public static final String RESPONSE_IAM_CONTENTS_BG_STYLE_BG_COLOR = "bg_color";
    public static final String RESPONSE_IAM_CONTENTS_BG_STYLE_OVERLAY_COLOR = "overlay_color";
    public static final String RESPONSE_IAM_CONTENTS_TEXT = "text";
    public static final String RESPONSE_IAM_CONTENTS_TEXT_TITLE = "title";
    public static final String RESPONSE_IAM_CONTENTS_TEXT_BODY = "body";
    public static final String RESPONSE_IAM_CONTENTS_TEXT_TITLE_COLOR = "title_color";
    public static final String RESPONSE_IAM_CONTENTS_TEXT_BODY_COLOR = "body_color";
    public static final String RESPONSE_IAM_CONTENTS_TEXT_ALIGN = "align";
    public static final String RESPONSE_IAM_CONTENTS_BUTTONS = "buttons";
    public static final String RESPONSE_IAM_CONTENTS_BUTTONS_BUTTON_ID = "button_id";
    public static final String RESPONSE_IAM_CONTENTS_BUTTONS_TEXT = "text";
    public static final String RESPONSE_IAM_CONTENTS_BUTTONS_CLICK_ACTION = "click_action";
    public static final String RESPONSE_IAM_CONTENTS_BUTTONS_CLICK_ACTION_ID = "action_id";
    public static final String RESPONSE_IAM_CONTENTS_BUTTONS_CLICK_ACTION_ACTION_TYPE = "action_type";
    public static final String RESPONSE_IAM_CONTENTS_BUTTONS_CLICK_ACTION_ACTION_ARG = "action_arg";
    public static final String RESPONSE_IAM_CONTENTS_BUTTONS_TEXT_COLOR = "text_color";
    public static final String RESPONSE_IAM_CONTENTS_BUTTONS_BG_COLOR = "bg_color";
    public static final String RESPONSE_IAM_CONTENTS_BUTTONS_BORDER_COLOR = "border_color";
    public static final String RESPONSE_IAM_CONTENTS_DEFAULT_CLOSE_BUTTON = "default_close_button";
    public static final String RESPONSE_IAM_CONTENTS_DEFAULT_CLOSE_BUTTON_COLOR = "color";
    public static final String RESPONSE_IAM_CONTENTS_DEFAULT_CLOSE_BUTTON_BG_COLOR = "bg_color";
    public static final String RESPONSE_IAM_CONTENTS_STICKY_BANNER_OPTION = "sticky_banner_option";
    public static final String RESPONSE_IAM_CONTENTS_STICKY_BANNER_OPTION_ALIGN = "align";
    public static final String RESPONSE_IAM_CONTENTS_STICKY_BANNER_OPTION_CLICK_ACTION = "click_action";
    public static final String RESPONSE_IAM_CONTENTS_STICKY_BANNER_OPTION_CLICK_ACTION_ACTION_ID = "action_id";
    public static final String RESPONSE_IAM_CONTENTS_STICKY_BANNER_OPTION_CLICK_ACTION_ACTION_TYPE = "action_type";
    public static final String RESPONSE_IAM_CONTENTS_STICKY_BANNER_OPTION_CLICK_ACTION_ACTION_ARG = "action_arg";
    public static final String RESPONSE_IAM_TRIGGERS = "triggers";
    public static final String RESPONSE_IAM_TRIGGERS_TYPE = "type";
    public static final String RESPONSE_IAM_TRIGGERS_EVT = "evt";
    public static final String RESPONSE_IAM_TRIGGERS_PRIORITY = "priority";
    public static final String RESPONSE_IAM_EXT_ATTR = "ext_attr";
    public static final String RESPONSE_IAM_FREQUENCY_CAP = "frequency_cap";
    public static final String RESPONSE_IAM_FREQUENCY_CAP_PER_SESSION = "per_session";
    public static final String RESPONSE_IAM_FREQUENCY_CAP_PER_USER = "per_user";
    public static final String RESPONSE_IAM_FREQUENCY_CAP_PER_PERIOD = "per_period";
    public static final String RESPONSE_IAM_FREQUENCY_CAP_PER_PERIOD_PERIOD_IN_MINUTES = "period_in_minutes";
    public static final String RESPONSE_IAM_FREQUENCY_CAP_PER_PERIOD_PER_PERIOD = "per_period";
    public static final String RESPONSE_IAM_TIMEZONE_OFFSET = "timezone_offset";
    public static final String RESPONSE_IAM_TIMEZONE_TYPE = "timezone_type";
    public static final String RESPONSE_IAM_AVAILABLE_TIME = "available_time";
    public static final String RESPONSE_IAM_AVAILABLE_TIME_DAYS = "days";
    public static final String RESPONSE_IAM_AVAILABLE_TIME_START_MIN = "start_min";
    public static final String RESPONSE_IAM_AVAILABLE_TIME_END_MIN = "end_min";
    public static final String RESPONSE_IAM_START_DATETIME = "start_datetime";
    public static final String RESPONSE_IAM_END_DATETIME = "end_datetime";
    public static final String RESPONSE_IAM_LAST_UPDATED_DATETIME = "last_updated_datetime";
    public static final String PORTRAIT = "portrait";
    public static final String LANDSCAPE = "landscape";
    private static final String AVAILABLE_TIME_ALL = "all";
    private static final String NO_IMAGE = "no_image";
    private static final String TIMEZONE_GLOBAL = "Global";
    private static final String TIMEZONE_LOCAL = "Local";
    private static final String CHECKSUM_UNCHANGED = "unchanged";
    private static final String CHECKSUM_NO_ACTIVE_CAMPAIGN = "__no_active_campaign__";
    private static final String EVENT_ABX_PREFIX = "abx:";
    private static final String EVENT_CUSTOM_PREFIX = "custom:";
    private static final String NULL_STRING = "null";
    InAppMessageSqliteStore dbHelper;
    DataRegistry dataRegistry;
    Context context;
    ExecutorService imageDownloadExecutorService;

    public InAppMessageDAO(Context context, DataRegistry dataRegistry) {
        dbHelper = new InAppMessageSqliteStore(context);
        this.dataRegistry = dataRegistry;
        this.context = context;
        this.imageDownloadExecutorService = Executors.newSingleThreadExecutor();
    }

    public void insertApiResponse2Table(String response) throws JSONException {
        ArrayList<String> responseCampaignIdArrayList = new ArrayList<>();

        JSONObject responseJsonObject = new JSONObject(response);

        JSONObject data = responseJsonObject.getJSONObject(RESPONSE_DATA);

        String status = data.getString(RESPONSE_STATUS);
        String checksum = data.getString(RESPONSE_CHECKSUM);
        int minutesToExpiry = data.getInt(RESPONSE_MINUTES_TO_EXPIRY);

        if (CHECKSUM_UNCHANGED.equals(status) || CHECKSUM_NO_ACTIVE_CAMPAIGN.equals(checksum)) {
            if (CHECKSUM_UNCHANGED.equals(status))
                AbxLog.d("insertApiResponse2Table: inAppMessage Campaign unchanged", true);
            else
                AbxLog.d("insertApiResponse2Table: No Active Campaign", true);

            dataRegistry.putDataRegistry(new DataUnit(
                    DataRegistryKey.STRING_IN_APP_MESSAGE_CHECKSUM,
                    checksum,
                    5,
                    this.getClass().getName(),
                    true
            ));

            dataRegistry.putDataRegistry(new DataUnit(
                    DataRegistryKey.INT_IN_APP_MESSAGE_MINUTES_TO_EXPIRY,
                    minutesToExpiry,
                    5,
                    this.getClass().getName(),
                    true
            ));

            dataRegistry.putDataRegistry(new DataUnit(
                    DataRegistryKey.LONG_IN_APP_MESSAGE_LAST_RESPONSE_TIME,
                    System.currentTimeMillis(),
                    5,
                    this.getClass().getName(),
                    true
            ));

            return;
        }

        JSONArray inAppMessages = data.getJSONArray(RESPONSE_IAM);

        deleteBeforeInsertApiResponse();

        ConcurrentLinkedQueue<Runnable> imageDownloadRunnableQueue = new ConcurrentLinkedQueue<>();

        for (int inAppMessagesIdx = 0; inAppMessagesIdx < inAppMessages.length(); inAppMessagesIdx++) {
            JSONObject inAppMessage = inAppMessages.getJSONObject(inAppMessagesIdx);
            JSONObject contents = inAppMessage.getJSONObject(RESPONSE_IAM_CONTENTS);
            JSONObject frequencyCap = inAppMessage.getJSONObject(RESPONSE_IAM_FREQUENCY_CAP);
            JSONArray triggers = inAppMessage.getJSONArray(RESPONSE_IAM_TRIGGERS);
            JSONArray availableTimes = inAppMessage.getJSONArray(RESPONSE_IAM_AVAILABLE_TIME);

            long startDateTime = getStartDateTimeLong(inAppMessage.getString(RESPONSE_IAM_START_DATETIME));
            long endDateTime = getEndDateTimeLong(inAppMessage.optString(RESPONSE_IAM_END_DATETIME));
            long lastUpdatedDateTime = getLastUpdatedDateTimeLong(inAppMessage.getString(RESPONSE_IAM_LAST_UPDATED_DATETIME));

            responseCampaignIdArrayList.add(inAppMessage.getString(RESPONSE_IAM_CAMPAIGN_ID));

            if (startDateTime == 0 || endDateTime == -1 || lastUpdatedDateTime == 0) {
                AbxLog.d("insertApiResponse2Table: DateTime Parse Error", true);
                return;
            }

            String campaignId = inAppMessage.getString(RESPONSE_IAM_CAMPAIGN_ID);

            boolean portraitAvailable = inAppMessage.getJSONArray(RESPONSE_IAM_PAGE_ORIENTATION).toString().contains(PORTRAIT);
            boolean landscapeAvailable = inAppMessage.getJSONArray(RESPONSE_IAM_PAGE_ORIENTATION).toString().contains(LANDSCAPE);

            insertInAppMessageTable(
                    campaignId,
                    portraitAvailable ? 1 : 0,
                    landscapeAvailable ? 1 : 0,
                    inAppMessage.optJSONObject(RESPONSE_IAM_EXT_ATTR) == null ? null : inAppMessage.getJSONObject(RESPONSE_IAM_EXT_ATTR).toString(),
                    inAppMessage.optInt(RESPONSE_IAM_TIMEZONE_OFFSET),
                    inAppMessage.optString(RESPONSE_IAM_TIMEZONE_TYPE),
                    inAppMessage.optString(RESPONSE_IAM_TYPE),
                    inAppMessage.optString(RESPONSE_IAM_LAYOUT),
                    contents.optJSONObject(RESPONSE_IAM_CONTENTS_BG_STYLE) == null ? null : contents.getJSONObject(RESPONSE_IAM_CONTENTS_BG_STYLE).getString(RESPONSE_IAM_CONTENTS_BG_STYLE_BG_COLOR),
                    contents.optJSONObject(RESPONSE_IAM_CONTENTS_BG_STYLE) == null ? null : contents.getJSONObject(RESPONSE_IAM_CONTENTS_BG_STYLE).getString(RESPONSE_IAM_CONTENTS_BG_STYLE_OVERLAY_COLOR),
                    contents.optJSONObject(RESPONSE_IAM_CONTENTS_TEXT) == null ? null : contents.getJSONObject(RESPONSE_IAM_CONTENTS_TEXT).toString(),
                    contents.optJSONArray(RESPONSE_IAM_CONTENTS_BUTTONS) == null ? null : contents.getJSONArray(RESPONSE_IAM_CONTENTS_BUTTONS).toString(),
                    contents.optJSONObject(RESPONSE_IAM_CONTENTS_DEFAULT_CLOSE_BUTTON) == null ? null : contents.getJSONObject(RESPONSE_IAM_CONTENTS_DEFAULT_CLOSE_BUTTON).optString(RESPONSE_IAM_CONTENTS_DEFAULT_CLOSE_BUTTON_COLOR),
                    contents.optJSONObject(RESPONSE_IAM_CONTENTS_DEFAULT_CLOSE_BUTTON) == null ? null : contents.getJSONObject(RESPONSE_IAM_CONTENTS_DEFAULT_CLOSE_BUTTON).optString(RESPONSE_IAM_CONTENTS_DEFAULT_CLOSE_BUTTON_BG_COLOR),
                    contents.optJSONObject(RESPONSE_IAM_CONTENTS_STICKY_BANNER_OPTION) == null ? null : contents.getJSONObject(RESPONSE_IAM_CONTENTS_STICKY_BANNER_OPTION).toString(),
                    frequencyCap.getInt(RESPONSE_IAM_FREQUENCY_CAP_PER_SESSION),
                    frequencyCap.getInt(RESPONSE_IAM_FREQUENCY_CAP_PER_USER),
                    frequencyCap.optJSONObject(RESPONSE_IAM_FREQUENCY_CAP_PER_PERIOD) == null ? 0 : frequencyCap.getJSONObject(RESPONSE_IAM_FREQUENCY_CAP_PER_PERIOD).getInt(RESPONSE_IAM_FREQUENCY_CAP_PER_PERIOD_PERIOD_IN_MINUTES),
                    frequencyCap.optJSONObject(RESPONSE_IAM_FREQUENCY_CAP_PER_PERIOD) == null ? 0 : frequencyCap.getJSONObject(RESPONSE_IAM_FREQUENCY_CAP_PER_PERIOD).getInt(RESPONSE_IAM_FREQUENCY_CAP_PER_PERIOD_PER_PERIOD),
                    startDateTime,
                    endDateTime,
                    lastUpdatedDateTime
            );

            for (int triggersIdx = 0; triggersIdx < triggers.length(); triggersIdx++) {
                JSONObject trigger = triggers.getJSONObject(triggersIdx);
                String eventName = trigger.getString(RESPONSE_IAM_TRIGGERS_EVT);

                if (eventName.startsWith(EVENT_ABX_PREFIX)) {
                    eventName = eventName.replace(EVENT_ABX_PREFIX, "");
                } else if (eventName.startsWith(EVENT_CUSTOM_PREFIX)) {
                    eventName = eventName.replace(EVENT_CUSTOM_PREFIX, "");
                }

                insertTriggersTable(
                        campaignId,
                        trigger.getString(RESPONSE_IAM_TRIGGERS_TYPE),
                        eventName,
                        trigger.getInt(RESPONSE_IAM_TRIGGERS_PRIORITY)
                );
            }

            if (availableTimes.length() == 0) {
                insertAvailableTimeTable(
                        campaignId,
                        AVAILABLE_TIME_ALL,
                        0,
                        0
                );
            } else {
                for (int availableTimeIdx = 0; availableTimeIdx < availableTimes.length(); availableTimeIdx++) {
                    JSONObject availableTime = availableTimes.getJSONObject(availableTimeIdx);
                    JSONArray days = availableTime.getJSONArray(RESPONSE_IAM_AVAILABLE_TIME_DAYS);

                    for (int dayIdx = 0; dayIdx < days.length(); dayIdx++) {
                        insertAvailableTimeTable(
                                campaignId,
                                days.getString(dayIdx),
                                availableTime.getInt(RESPONSE_IAM_AVAILABLE_TIME_START_MIN),
                                availableTime.getInt(RESPONSE_IAM_AVAILABLE_TIME_END_MIN)
                        );
                    }
                }
            }

            if (!isCurrentFrequencyExist(inAppMessage.getString(RESPONSE_IAM_CAMPAIGN_ID))) {
                insertCurrentFrequencyCapTable(
                        campaignId,
                        0,
                        0,
                        0,
                        0
                );
            }

            if (portraitAvailable && contents.optJSONObject(RESPONSE_IAM_CONTENTS_IMAGE) != null) {
                String portraitUrlString = contents.getJSONObject(RESPONSE_IAM_CONTENTS_IMAGE).getJSONObject(RESPONSE_IAM_CONTENTS_IMAGE_URL).getString(RESPONSE_IAM_CONTENTS_IMAGE_URL_PORTRAIT);

                if (!getImageUrlFromImageTable(inAppMessage.getString(RESPONSE_IAM_CAMPAIGN_ID), PORTRAIT).equals(portraitUrlString)) {
                    deleteImageTableWithOrientation(inAppMessage.getString(RESPONSE_IAM_CAMPAIGN_ID), PORTRAIT);

                    insertImageTable(
                            campaignId,
                            PORTRAIT,
                            portraitUrlString,
                            NO_IMAGE,
                            contents.getJSONObject(RESPONSE_IAM_CONTENTS_IMAGE).optJSONObject(RESPONSE_IAM_CONTENTS_IMAGE_CLICK_ACTION) == null ? null : contents.getJSONObject(RESPONSE_IAM_CONTENTS_IMAGE).getJSONObject(RESPONSE_IAM_CONTENTS_IMAGE_CLICK_ACTION).toString(),
                            false
                    );

                    imageDownloadRunnableQueue.add(() -> getAndSaveImageInfo(campaignId, PORTRAIT, false));
                }
            }

            if (landscapeAvailable && contents.optJSONObject(RESPONSE_IAM_CONTENTS_IMAGE) != null) {
                String landscapeUrlString = contents.getJSONObject(RESPONSE_IAM_CONTENTS_IMAGE).getJSONObject(RESPONSE_IAM_CONTENTS_IMAGE_URL).getString(RESPONSE_IAM_CONTENTS_IMAGE_URL_LANDSCAPE);

                if (!getImageUrlFromImageTable(inAppMessage.getString(RESPONSE_IAM_CAMPAIGN_ID), LANDSCAPE).equals(landscapeUrlString)) {
                    deleteImageTableWithOrientation(inAppMessage.getString(RESPONSE_IAM_CAMPAIGN_ID), LANDSCAPE);

                    insertImageTable(
                            campaignId,
                            LANDSCAPE,
                            landscapeUrlString,
                            NO_IMAGE,
                            contents.getJSONObject(RESPONSE_IAM_CONTENTS_IMAGE).optJSONObject(RESPONSE_IAM_CONTENTS_IMAGE_CLICK_ACTION) == null ? null : contents.getJSONObject(RESPONSE_IAM_CONTENTS_IMAGE).getJSONObject(RESPONSE_IAM_CONTENTS_IMAGE_CLICK_ACTION).toString(),
                            false
                    );

                    imageDownloadRunnableQueue.add(() -> getAndSaveImageInfo(campaignId, LANDSCAPE, false));
                }
            }
        }

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_IN_APP_MESSAGE_CHECKSUM,
                checksum,
                5,
                this.getClass().getName(),
                true
        ));

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.INT_IN_APP_MESSAGE_MINUTES_TO_EXPIRY,
                minutesToExpiry,
                5,
                this.getClass().getName(),
                true
        ));

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.LONG_IN_APP_MESSAGE_LAST_RESPONSE_TIME,
                System.currentTimeMillis(),
                5,
                this.getClass().getName(),
                true
        ));

        deleteStoppedOrDeletedInAppMessage(responseCampaignIdArrayList);

        while (!imageDownloadRunnableQueue.isEmpty()) {
            Runnable imageDownloadRunnable = imageDownloadRunnableQueue.poll();
            imageDownloadExecutorService.submit(imageDownloadRunnable);
        }
    }

    public long getStartDateTimeLong(String startDateString) {
        SimpleDateFormat startAndEndDateFormat = new SimpleDateFormat(CoreConstants.IAM_START_AND_END_DATE_FORMAT, Locale.US);
        startAndEndDateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
        SimpleDateFormat startAndEndDateFormatNoMills = new SimpleDateFormat(CoreConstants.IAM_START_AND_END_DATE_FORMAT_NO_MILLIS, Locale.US);
        startAndEndDateFormatNoMills.setTimeZone(TimeZone.getTimeZone("UTC"));
        Date startDate;
        try {
            startDate = startAndEndDateFormat.parse(startDateString);
        } catch (ParseException e) {
            try {
                startDate = startAndEndDateFormatNoMills.parse(startDateString);
            } catch (ParseException parseException) {
                AbxLog.w("getStartDateTimeLong: parse Error", true);
                return 0;
            }
        }

        if (startDate != null) {
            return startDate.getTime();
        }

        return 0;
    }

    public long getEndDateTimeLong(String endDateString) {
        if (endDateString == null || endDateString.equals(NULL_STRING))
            return 0;

        SimpleDateFormat startAndEndDateFormat = new SimpleDateFormat(CoreConstants.IAM_START_AND_END_DATE_FORMAT, Locale.US);
        startAndEndDateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
        SimpleDateFormat startAndEndDateFormatNoMills = new SimpleDateFormat(CoreConstants.IAM_START_AND_END_DATE_FORMAT_NO_MILLIS, Locale.US);
        startAndEndDateFormatNoMills.setTimeZone(TimeZone.getTimeZone("UTC"));

        Date endDate;
        try {
            endDate = startAndEndDateFormat.parse(endDateString);
        } catch (ParseException e) {
            try {
                endDate = startAndEndDateFormatNoMills.parse(endDateString);
            } catch (ParseException parseException) {
                AbxLog.w("getEndDateTimeLong: parse Error", true);
                return -1;
            }
        }

        if (endDate != null) {
            return endDate.getTime();
        }

        return 0;
    }

    public long getLastUpdatedDateTimeLong(String lastUpdatedDateString) {
        SimpleDateFormat lastUpdatedDateFormat = new SimpleDateFormat(CoreConstants.IAM_LAST_UPDATE_DATE_FORMAT, Locale.US);
        Date lastUpdatedDate;
        try {
            lastUpdatedDate = lastUpdatedDateFormat.parse(lastUpdatedDateString);
        } catch (ParseException e) {
            AbxLog.w("getLastUpdatedDateTimeLong: parse Error", true);
            return 0;
        }

        if (lastUpdatedDate != null) {
            return lastUpdatedDate.getTime();
        }

        return 0;
    }


    public void insertInAppMessageTable(
            String campaignId,
            int portraitAvailable,
            int landscapeAvailable,
            String externalAttr,
            Integer timezoneOffset,
            String timezoneType,
            String type,
            String layout,
            String bgColor,
            String overlayColor,
            String contentsText,
            String contentsButtons,
            String defaultCloseButtonColor,
            String defaultCloseButtonBgColor,
            String stickyBannerOption,
            int frequencyCapPerSession,
            int frequencyCapPerUser,
            int frequencyCapForPeriodInMinutes,
            int frequencyCapForPeriodPerPeriod,
            long startDateTime,
            long endDateTime,
            long lastUpdatedDateTime
    ) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        ContentValues contentValues = new ContentValues();
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CAMPAIGN_ID, campaignId);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_PORTRAIT_AVAILABLE, portraitAvailable);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_LANDSCAPE_AVAILABLE, landscapeAvailable);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_EXTERNAL_ATTR, externalAttr);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_TIMEZONE_OFFSET, timezoneOffset);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_TIMEZONE_TYPE, timezoneType);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_TYPE, type);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_LAYOUT, layout);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_BG_STYLE_BG_COLOR, bgColor);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_BG_STYLE_OVERLAY_COLOR, overlayColor);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CONTENTS_TEXT, contentsText);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CONTENTS_BUTTONS, contentsButtons);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_DEFAULT_CLOSE_BUTTON_COLOR, defaultCloseButtonColor);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_DEFAULT_CLOSE_BUTTON_BG_COLOR, defaultCloseButtonBgColor);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_STICKY_BANNER_OPTION, stickyBannerOption);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_FREQUENCY_CAP_PER_SESSION, frequencyCapPerSession);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_FREQUENCY_CAP_PER_USER, frequencyCapPerUser);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_FREQUENCY_CAP_FOR_PERIOD_IN_MINUTES, frequencyCapForPeriodInMinutes);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_FREQUENCY_CAP_FOR_PERIOD_PER_PERIOD, frequencyCapForPeriodPerPeriod);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_START_DATETIME, startDateTime);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_END_DATETIME, endDateTime);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_LAST_UPDATED_DATETIME, lastUpdatedDateTime);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_UNAVAILABLE_DAY, "");
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_UNAVAILABLE_TIME, 0);

        long newRowId = db.insertWithOnConflict(InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME, null, contentValues, SQLiteDatabase.CONFLICT_REPLACE);

        if (newRowId == -1)
            AbxLog.d("Insert to InAppMessageTable fail", true);
        else
            AbxLog.d("Insert Success, newRowId in InAppMessageTable : " + newRowId, true);
    }

    public void updateOneDayCloseInAppMessageTable(
            String campaignId,
            String unavailableDay,
            long unavailableTime
    ) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        ContentValues contentValues = new ContentValues();
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_UNAVAILABLE_DAY, unavailableDay);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_UNAVAILABLE_TIME, unavailableTime);

        db.update(InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME, contentValues,
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CAMPAIGN_ID + "= ? ", new String[]{campaignId});
    }

    public void insertTriggersTable(
            String campaignId,
            String type,
            String eventName,
            int priority
    ) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        ContentValues contentValues = new ContentValues();
        contentValues.put(InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_CAMPAIGN_ID, campaignId);
        contentValues.put(InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_TYPE, type);
        contentValues.put(InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_EVENT_NAME, eventName);
        contentValues.put(InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_PRIORITY, priority);

        long newRowId = db.insertWithOnConflict(InAppMessageSqliteStore.TriggersTable.TABLE_NAME, null, contentValues, SQLiteDatabase.CONFLICT_REPLACE);

        if (newRowId == -1)
            AbxLog.d("Insert to TriggersTable fail", true);
        else
            AbxLog.d("Insert Success, newRowId in TriggersTable : " + newRowId, true);
    }

    public void insertAvailableTimeTable(
            String campaignId,
            String day,
            int startMin,
            int endMin
    ) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        ContentValues contentValues = new ContentValues();
        contentValues.put(InAppMessageSqliteStore.AvailableTimeTable.COLUMN_NAME_CAMPAIGN_ID, campaignId);
        contentValues.put(InAppMessageSqliteStore.AvailableTimeTable.COLUMN_NAME_DAY, day);
        contentValues.put(InAppMessageSqliteStore.AvailableTimeTable.COLUMN_NAME_START_MIN, startMin);
        contentValues.put(InAppMessageSqliteStore.AvailableTimeTable.COLUMN_NAME_END_MIN, endMin);

        long newRowId = db.insertWithOnConflict(InAppMessageSqliteStore.AvailableTimeTable.TABLE_NAME, null, contentValues, SQLiteDatabase.CONFLICT_REPLACE);

        if (newRowId == -1)
            AbxLog.d("insert to AvailableTimeTable fail", true);
        else
            AbxLog.d("Insert Success, newRowId in AvailableTimeTable : " + newRowId, true);
    }

    public void insertCurrentFrequencyCapTable(
            String campaignId,
            int capPerSession,
            int capPerUser,
            long lastGroupStartTime,
            int lastGroupCount
    ) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        ContentValues contentValues = new ContentValues();
        contentValues.put(InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CAMPAIGN_ID, campaignId);
        contentValues.put(InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CURRENT_FREQUENCY_CAP_PER_SESSION, capPerSession);
        contentValues.put(InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CURRENT_FREQUENCY_CAP_PER_USER, capPerUser);
        contentValues.put(InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_FREQUENCY_LAST_GROUP_START_TIME, lastGroupStartTime);
        contentValues.put(InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_FREQUENCY_LAST_GROUP_COUNT, lastGroupCount);

        long newRowId = db.insertWithOnConflict(InAppMessageSqliteStore.CurrentFrequencyCapTable.TABLE_NAME, null, contentValues, SQLiteDatabase.CONFLICT_REPLACE);

        if (newRowId == -1)
            AbxLog.d("insert to CurrentFrequencyCapTable fail", true);
        else
            AbxLog.d("Insert Success, newRowId in CurrentFrequencyCapTable : " + newRowId, true);
    }

    public synchronized void insertImageTable(
            String campaignId,
            String orientation,
            String urlString,
            String imagePathString,
            String clickAction,
            boolean isFirstShow
    ) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        ContentValues contentValues = new ContentValues();
        contentValues.put(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CAMPAIGN_ID, campaignId);
        contentValues.put(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_ORIENTATION, orientation);
        contentValues.put(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_URL, urlString);
        contentValues.put(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IMAGE_PATH_STRING, imagePathString);
        contentValues.put(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CLICK_ACTION, clickAction);
        contentValues.put(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IS_FIRST_SHOW, isFirstShow ? 1 : 0);

        long newRowId = db.insertWithOnConflict(InAppMessageSqliteStore.ImageTable.TABLE_NAME, null, contentValues, SQLiteDatabase.CONFLICT_REPLACE);

        if (newRowId == -1)
            AbxLog.d("insert to ImageTable fail", true);
        else
            AbxLog.d("Insert Success, newRowId in ImageTable : " + newRowId, true);
    }

    public synchronized void updateImageTable(
            String campaignId,
            String orientation,
            String urlString,
            String imageString,
            String clickAction,
            boolean isFirstShow
    ) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        ContentValues contentValues = new ContentValues();
        contentValues.put(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CAMPAIGN_ID, campaignId);
        contentValues.put(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_ORIENTATION, orientation);
        contentValues.put(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_URL, urlString);
        contentValues.put(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IMAGE_PATH_STRING, imageString);
        contentValues.put(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CLICK_ACTION, clickAction);
        contentValues.put(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IS_FIRST_SHOW, isFirstShow ? 1 : 0);

        long numberOfRowsAffected = db.update(InAppMessageSqliteStore.ImageTable.TABLE_NAME, contentValues,
                InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CAMPAIGN_ID + " = ? AND " +
                        InAppMessageSqliteStore.ImageTable.COLUMN_NAME_ORIENTATION + " = ?",
                new String[]{campaignId, orientation});

        if (numberOfRowsAffected == 0) {
            AbxLog.d("updateImageTable Fail", true);
        }
    }

    public void deleteInAppMessageTable(String campaignId) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        int deletedRowCount = db.delete(InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME, InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CAMPAIGN_ID + "= '" + campaignId + "'", null);

        AbxLog.d("deleteAvailableTimeTable: Deleted row count = " + deletedRowCount, true);
    }

    public void deleteAllInAppMessageTable() {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        int deletedRowCount = db.delete(InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME, null, null);

        AbxLog.d("deleteAvailableTimeTable: Deleted row count = " + deletedRowCount, true);
    }


    public void deleteAvailableTimeTable(String campaignId) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        int deletedRowCount = db.delete(InAppMessageSqliteStore.AvailableTimeTable.TABLE_NAME, InAppMessageSqliteStore.AvailableTimeTable.COLUMN_NAME_CAMPAIGN_ID + "= '" + campaignId + "'", null);


        AbxLog.d("deleteAvailableTimeTable: Deleted row count = " + deletedRowCount, true);
    }

    public void deleteAllAvailableTimeTable() {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        int deletedRowCount = db.delete(InAppMessageSqliteStore.AvailableTimeTable.TABLE_NAME, null, null);


        AbxLog.d("deleteAvailableTimeTable: Deleted row count = " + deletedRowCount, true);
    }


    public void deleteTriggersTable(String campaignId) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        int deletedRowCount = db.delete(InAppMessageSqliteStore.TriggersTable.TABLE_NAME, InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_CAMPAIGN_ID + "= '" + campaignId + "'", null);


        AbxLog.d("deleteTriggersTable: Deleted row count = " + deletedRowCount, true);
    }

    public void deleteAllTriggersTable() {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        int deletedRowCount = db.delete(InAppMessageSqliteStore.TriggersTable.TABLE_NAME, null, null);


        AbxLog.d("deleteTriggersTable: Deleted row count = " + deletedRowCount, true);
    }

    public void deleteCurrentFrequencyCapTable(String campaignId) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        int deletedRowCount = db.delete(InAppMessageSqliteStore.CurrentFrequencyCapTable.TABLE_NAME, InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CAMPAIGN_ID + "= '" + campaignId + "'", null);


        AbxLog.d("deleteAvailableTimeTable: Deleted row count = " + deletedRowCount, true);
    }

    public void deleteImageTable(String campaignId) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        String portraitImagePathQueue = getImagePathFromImageTable(campaignId, PORTRAIT);
        String landscapeImagePathQueue = getImagePathFromImageTable(campaignId, LANDSCAPE);

        deleteImageFromInternalStorage(context, portraitImagePathQueue);
        deleteImageFromInternalStorage(context, landscapeImagePathQueue);

        int deletedRowCount = db.delete(InAppMessageSqliteStore.ImageTable.TABLE_NAME, InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CAMPAIGN_ID + "= '" + campaignId + "'", null);

        AbxLog.d("deleteImageTable: Deleted row count = " + deletedRowCount, true);
    }

    public String getImageUrlFromImageTable(String campaignId, String orientation) {
        SQLiteDatabase db = dbHelper.getReadableDatabase();

        String[] projection = {
                InAppMessageSqliteStore.ImageTable.COLUMN_NAME_URL,
        };

        String selection;

        selection = InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CAMPAIGN_ID + " = ? AND "
                + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_ORIENTATION + " = ? ";

        String[] selectionArgs = {campaignId, orientation};

        Cursor cursor = db.query(
                InAppMessageSqliteStore.ImageTable.TABLE_NAME,
                projection,
                selection,
                selectionArgs,
                null,
                null,
                null
        );

        if (cursor.moveToNext()) {
            String imageUrl = cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_URL));
            if (imageUrl != null) {
                return imageUrl;
            }
        }

        cursor.close();

        return "";
    }

    public String getImagePathFromImageTable(String campaignId, String orientation) {
        SQLiteDatabase db = dbHelper.getReadableDatabase();

        String[] projection = {
                InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IMAGE_PATH_STRING,
        };

        String selection;

        selection = InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CAMPAIGN_ID + " = ? AND "
                + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_ORIENTATION + " = ? ";

        String[] selectionArgs = {campaignId, orientation};

        Cursor cursor = db.query(
                InAppMessageSqliteStore.ImageTable.TABLE_NAME,
                projection,
                selection,
                selectionArgs,
                null,
                null,
                null
        );

        if (cursor.moveToNext()) {
            String imagePath = cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IMAGE_PATH_STRING));
            return imagePath;
        }

        cursor.close();

        return null;
    }

    public Queue<String> getAllImagePathFromImageTable() {
        SQLiteDatabase db = dbHelper.getReadableDatabase();

        String[] projection = {
                InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IMAGE_PATH_STRING,
        };

        Cursor cursor = db.query(
                InAppMessageSqliteStore.ImageTable.TABLE_NAME,
                projection,
                null,
                null,
                null,
                null,
                null
        );

        Queue<String> imagePaths = new ConcurrentLinkedQueue<>();

        while (cursor.moveToNext()) {
            String imagePath = cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IMAGE_PATH_STRING));
            imagePaths.add(imagePath);
        }

        cursor.close();

        return imagePaths;
    }

    public void deleteImageTableWithOrientation(String campaignId, String orientation) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        String imagePath = getImagePathFromImageTable(campaignId, orientation);

        if (imagePath != null) {
            boolean isInternalStorageDeleted = deleteImageFromInternalStorage(context, imagePath);
            AbxLog.d("deleteImageTableWithOrientation: isInternalStorageDeleted" + isInternalStorageDeleted, true);
        }

        int deletedRowCount = db.delete(InAppMessageSqliteStore.ImageTable.TABLE_NAME,
                InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CAMPAIGN_ID + "= '" + campaignId + "' "
                        + " AND " + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_ORIENTATION + "= '" + orientation + "' ", null);

        AbxLog.d("deleteImageTableWithOrientation: Deleted row count = " + deletedRowCount, true);
    }


    public void deleteCampaign(String campaignId) {
        deleteInAppMessageTable(campaignId);
        deleteAvailableTimeTable(campaignId);
        deleteTriggersTable(campaignId);
        deleteCurrentFrequencyCapTable(campaignId);
        deleteImageTable(campaignId);
    }

    public void deletePerUserFrequencyExpiredCampaign(String campaignId) {
        deleteInAppMessageTable(campaignId);
        deleteAvailableTimeTable(campaignId);
        deleteTriggersTable(campaignId);
        deleteImageTable(campaignId);
    }

    public void deleteBeforeInsertApiResponse() {
        deleteAllInAppMessageTable();
        deleteAllTriggersTable();
        deleteAllAvailableTimeTable();
    }

    public void deleteStoppedOrDeletedInAppMessage(ArrayList<String> responseCampaignIdArrayList) {
        SQLiteDatabase db = dbHelper.getReadableDatabase();

        String getCampaignIdFromImageTableQuery = "SELECT " + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CAMPAIGN_ID
                + " FROM " + InAppMessageSqliteStore.ImageTable.TABLE_NAME;

        Cursor campaignIdCursor = db.rawQuery(getCampaignIdFromImageTableQuery, null);

        while (campaignIdCursor.moveToNext()) {
            String campaignId = campaignIdCursor.getString(campaignIdCursor.getColumnIndex(InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_CAMPAIGN_ID));

            if (!responseCampaignIdArrayList.contains(campaignId)) {
                deleteCampaign(campaignId);
            }
        }

        campaignIdCursor.close();
    }

    public void deleteDB() {
        imageDownloadExecutorService.shutdownNow();
        Queue<String> imagePathQueue;

        try {
            imagePathQueue = getAllImagePathFromImageTable();
        } catch (Exception e) {
            AbxLog.d("InAppMessage DB is already deleted", true);
            return;
        }

        if (imagePathQueue != null) {
            while (!imagePathQueue.isEmpty()) {
                String imagePath = imagePathQueue.poll();
                deleteImageFromInternalStorage(context, imagePath);
            }
        }

        dbHelper.deleteDB();
    }

    public void restartInAppMessage() {
        imageDownloadExecutorService = Executors.newSingleThreadExecutor();
        dbHelper.restartDB();
    }

    public boolean isCampaignIdTimeAvailable(String campaignId) {
        SQLiteDatabase db = dbHelper.getReadableDatabase();

        String getTimeInfoQuery = "SELECT " + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_TIMEZONE_OFFSET + ", "
                + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_TIMEZONE_TYPE + ", "
                + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_START_DATETIME + ", "
                + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_END_DATETIME + ", "
                + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_UNAVAILABLE_DAY + ", "
                + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_UNAVAILABLE_TIME
                + " FROM " + InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME
                + " WHERE " + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CAMPAIGN_ID + " = '" + campaignId + "'";

        Cursor getTimeCursor = db.rawQuery(getTimeInfoQuery, null);

        long startDateTime;
        long endDateTime;
        String timezoneType;
        int timezoneOffset;
        String currentDay;
        String currentMin;
        long currentTimeMillis;
        String unavailableDay;
        long unavailableTime;

        if (getTimeCursor.moveToNext()) {
            startDateTime = getTimeCursor.getLong(getTimeCursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_START_DATETIME));
            endDateTime = getTimeCursor.getLong(getTimeCursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_END_DATETIME));
            timezoneType = getTimeCursor.getString(getTimeCursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_TIMEZONE_TYPE));
            timezoneOffset = getTimeCursor.getInt(getTimeCursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_TIMEZONE_OFFSET));
            unavailableDay = getTimeCursor.getString(getTimeCursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_UNAVAILABLE_DAY));
            unavailableTime = getTimeCursor.getLong(getTimeCursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_UNAVAILABLE_TIME));

            if (timezoneType.equals(TIMEZONE_GLOBAL)) {
                Calendar utcCalendar = Calendar.getInstance(TimeZone.getTimeZone("UTC"));
                SimpleDateFormat utcSimpleDateFormat = new SimpleDateFormat("EE", Locale.ENGLISH);
                utcSimpleDateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));

                currentTimeMillis = System.currentTimeMillis();
                currentDay = utcSimpleDateFormat.format(System.currentTimeMillis() + timezoneOffset * 3600 * 1000);
                currentMin = Integer.toString((utcCalendar.get(Calendar.MINUTE) + (utcCalendar.get(Calendar.HOUR_OF_DAY) + timezoneOffset) * 60) % 1440);
            } else if (timezoneType.equals(TIMEZONE_LOCAL)) {
                Calendar localCalendar = Calendar.getInstance();
                SimpleDateFormat localSimpleDateFormat = new SimpleDateFormat("EE", Locale.ENGLISH);

                currentTimeMillis = System.currentTimeMillis();
                currentDay = localSimpleDateFormat.format(System.currentTimeMillis());
                currentMin = Integer.toString(localCalendar.get(Calendar.MINUTE) + localCalendar.get(Calendar.HOUR_OF_DAY) * 60);
            } else {
                AbxLog.d("isCampaignIdTimeAvailable: invalid timezoneOffset: " + timezoneOffset, true);
                return false;
            }

            //one day close check
            if (currentDay.equals(unavailableDay) && unavailableTime + 2 * 24 * 3600 * 1000 > currentTimeMillis) {
                AbxLog.d("isCampaignIdTimeAvailable: one day close ", true);
                return false;
            } else {
                updateOneDayCloseInAppMessageTable(campaignId, "", 0);
            }


            //start, end time available check
            if (currentTimeMillis < startDateTime) {
                AbxLog.d("isCampaignIdTimeAvailable: currentTimeMillis is earlier than startDateTime", true);
                return false;
            } else if (endDateTime < currentTimeMillis && endDateTime != 0) {
                AbxLog.d("isCampaignIdTimeAvailable: currentTimeMillis over endDateTime", true);
                deleteCampaign(campaignId);
                return false;
            }
        } else {
            AbxLog.d("isCampaignIdTimeAvailable: campaignId dateTime inAvailable", true);
            return false;
        }

        if (currentDay.equals("Thu")) {
            currentDay = "Thur";
        }

        AbxLog.d("currentDay: " + currentDay + ", currentMin: " + currentMin, true);

        String isCampaignIdValidQuery = "SELECT * FROM "
                + InAppMessageSqliteStore.AvailableTimeTable.TABLE_NAME
                + " WHERE " + InAppMessageSqliteStore.AvailableTimeTable.TABLE_NAME + "." + InAppMessageSqliteStore.AvailableTimeTable.COLUMN_NAME_CAMPAIGN_ID + " = '" + campaignId + "' "
                + " AND (" + InAppMessageSqliteStore.AvailableTimeTable.TABLE_NAME + "." + InAppMessageSqliteStore.AvailableTimeTable.COLUMN_NAME_DAY + " = '" + currentDay + "'"
                + " AND " + InAppMessageSqliteStore.AvailableTimeTable.TABLE_NAME + "." + InAppMessageSqliteStore.AvailableTimeTable.COLUMN_NAME_START_MIN + " <= " + currentMin
                + " AND " + InAppMessageSqliteStore.AvailableTimeTable.TABLE_NAME + "." + InAppMessageSqliteStore.AvailableTimeTable.COLUMN_NAME_END_MIN + " >= " + currentMin
                + " OR " + InAppMessageSqliteStore.AvailableTimeTable.TABLE_NAME + "." + InAppMessageSqliteStore.AvailableTimeTable.COLUMN_NAME_DAY + " = '" + AVAILABLE_TIME_ALL + "')";


        Cursor cursor = db.rawQuery(isCampaignIdValidQuery, null);

        boolean isCampaignIdValid = cursor.moveToNext();

        getTimeCursor.close();

        cursor.close();

        return isCampaignIdValid;
    }

    public boolean isCampaignIdFrequencyAvailable(String campaignId) {
        SQLiteDatabase db = dbHelper.getReadableDatabase();

        String isCampaignIdFrequencyAvailableQuery = "SELECT " + InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_FREQUENCY_LAST_GROUP_COUNT + ", "
                + InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_FREQUENCY_LAST_GROUP_START_TIME + ", "
                + InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CURRENT_FREQUENCY_CAP_PER_SESSION + ", "
                + InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CURRENT_FREQUENCY_CAP_PER_USER + ", "
                + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_FREQUENCY_CAP_FOR_PERIOD_IN_MINUTES + ", "
                + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_FREQUENCY_CAP_FOR_PERIOD_PER_PERIOD + ", "
                + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_FREQUENCY_CAP_PER_USER + ", "
                + InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CURRENT_FREQUENCY_CAP_PER_USER
                + " FROM " + InAppMessageSqliteStore.CurrentFrequencyCapTable.TABLE_NAME
                + " JOIN " + InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME
                + " ON " + InAppMessageSqliteStore.CurrentFrequencyCapTable.TABLE_NAME + "." + InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CAMPAIGN_ID
                + " = " + InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME + "." + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CAMPAIGN_ID
                + " WHERE " + InAppMessageSqliteStore.CurrentFrequencyCapTable.TABLE_NAME + "." + InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CAMPAIGN_ID + " = '" + campaignId + "' "
                + " AND (" + InAppMessageSqliteStore.CurrentFrequencyCapTable.TABLE_NAME + "." + InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CURRENT_FREQUENCY_CAP_PER_SESSION + " "
                + " < " + InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME + "." + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_FREQUENCY_CAP_PER_SESSION
                + " OR " + InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME + "." + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_FREQUENCY_CAP_PER_SESSION + " = 0" //0일 경우 무제한 노출
                + " )";

        AbxLog.d("isCampaignIdFrequencyAvailableQuery: " + isCampaignIdFrequencyAvailableQuery, true);
        Cursor cursor = db.rawQuery(isCampaignIdFrequencyAvailableQuery, null);


        if (cursor.moveToNext()) {
            long lastGroupStartTime = cursor.getLong(cursor.getColumnIndex(InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_FREQUENCY_LAST_GROUP_START_TIME));
            int lastGroupCount = cursor.getInt(cursor.getColumnIndex(InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_FREQUENCY_LAST_GROUP_COUNT));
            int currentSessionCount = cursor.getInt(cursor.getColumnIndex(InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CURRENT_FREQUENCY_CAP_PER_SESSION));
            int currentUserCount = cursor.getInt(cursor.getColumnIndex(InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CURRENT_FREQUENCY_CAP_PER_USER));
            int currentPerUser = cursor.getInt(cursor.getColumnIndex(InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CURRENT_FREQUENCY_CAP_PER_USER));
            int periodInMinutes = cursor.getInt(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_FREQUENCY_CAP_FOR_PERIOD_IN_MINUTES));
            int periodPerPeriod = cursor.getInt(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_FREQUENCY_CAP_FOR_PERIOD_PER_PERIOD));
            int perUser = cursor.getInt(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_FREQUENCY_CAP_PER_USER));
            long currentTimeInMillis = System.currentTimeMillis();

            cursor.close();

            //perUser valid check
            if (perUser != 0 && currentPerUser >= perUser) {
                AbxLog.d("isCampaignIdFrequencyAvailable: frequency perUser is invalid", true);
                deletePerUserFrequencyExpiredCampaign(campaignId);
                return false;
            }

            //perperiod valid check
            if (periodPerPeriod == 0) { //0일 경우 무제한 노출
                insertCurrentFrequencyCapTable(campaignId, currentSessionCount + 1, currentUserCount + 1, currentTimeInMillis, lastGroupCount + 1);
                AbxLog.d("isCampaignIdFrequencyAvailable: frequency is valid", true);
                return true;
            } else if (lastGroupStartTime + periodInMinutes * 60 * 1000 < currentTimeInMillis) {
                insertCurrentFrequencyCapTable(campaignId, currentSessionCount + 1, currentUserCount + 1, currentTimeInMillis, 1);
                AbxLog.d("isCampaignIdFrequencyAvailable: frequency is valid", true);
                return true;
            } else if (lastGroupCount < periodPerPeriod) {
                insertCurrentFrequencyCapTable(campaignId, currentSessionCount + 1, currentUserCount + 1, lastGroupStartTime, lastGroupCount + 1);
                AbxLog.d("isCampaignIdFrequencyAvailable: frequency is valid", true);
                return true;
            } else {
                AbxLog.d("isCampaignIdFrequencyAvailable: frequency period is invalid", true);
                return false;
            }
        } else {
            cursor.close();
            AbxLog.d("isCampaignIdFrequencyAvailable: frequency perSession is invalid", true);
            return false;
        }
    }

    public boolean isCurrentFrequencyExist(String campaignId) {
        SQLiteDatabase db = dbHelper.getReadableDatabase();

        String isCurrentFrequencyExistQuery = "SELECT * FROM " + InAppMessageSqliteStore.CurrentFrequencyCapTable.TABLE_NAME
                + " WHERE " + InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CAMPAIGN_ID + " = '" + campaignId + "'";

        AbxLog.d("isCurrentFrequencyExistQuery: " + isCurrentFrequencyExistQuery, true);
        Cursor cursor = db.rawQuery(isCurrentFrequencyExistQuery, null);
        boolean isExist = cursor.moveToNext();
        cursor.close();
        return isExist;
    }

    public void resetFrequencyCapPerSession() {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        ContentValues contentValues = new ContentValues();
        contentValues.put(InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CURRENT_FREQUENCY_CAP_PER_SESSION, 0);
        int updateResult = db.update(InAppMessageSqliteStore.CurrentFrequencyCapTable.TABLE_NAME, contentValues, null, null);

        if (updateResult == -1) {
            AbxLog.d("resetFrequencyCapPerSessionQuery: resetFrequencyCapPerSession Fail!", true);
        } else {
            AbxLog.d("resetFrequencyCapPerSessionQuery: resetFrequencyCapPerSession SUCCESS", true);
        }
    }

    public InAppMessage getInAppMessageByEventName(String triggeredEventName, boolean isPortrait) {
        HashMap<String, Pair<Long, Long>> triggeredCampaignIdMap = getCampaignIdFromTriggerEvent(triggeredEventName);

        for (String campaignId : sortMapByValue(triggeredCampaignIdMap)) {
            if (!isCampaignIdTimeAvailable(campaignId))
                continue;

            InAppMessage inAppMessage = getInAppMessageByCampaignId(campaignId, isPortrait);

            if (inAppMessage == null) {
                AbxLog.d("getInAppMessageByCampaignId Fail, campaignId = " + campaignId, true);
            } else {

                if (!isCampaignIdFrequencyAvailable(campaignId))
                    continue;

                AbxLog.d("getInAppMessageByCampaignId Success, campaignId = " + campaignId, true);

                return inAppMessage;
            }
        }

        AbxLog.d("getInAppMessageByCampaignId Fail, No Valid InAppMessage", true);

        return null;
    }

    public HashMap<String, Pair<Long, Long>> getCampaignIdFromTriggerEvent(String triggeredEventName) {
        HashMap<String, Pair<Long, Long>> campaignIdHashMap = new HashMap<>();

        SQLiteDatabase db = dbHelper.getReadableDatabase();

        String getCampaignIdWithPriority = "SELECT "
                + InAppMessageSqliteStore.TriggersTable.TABLE_NAME + "." + InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_CAMPAIGN_ID + ", "
                + InAppMessageSqliteStore.TriggersTable.TABLE_NAME + "." + InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_PRIORITY + ", "
                + InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME + "." + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_LAST_UPDATED_DATETIME
                + " FROM " + InAppMessageSqliteStore.TriggersTable.TABLE_NAME
                + " JOIN " + InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME
                + " ON " + InAppMessageSqliteStore.TriggersTable.TABLE_NAME + "." + InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_CAMPAIGN_ID
                + " = " + InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME + "." + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CAMPAIGN_ID
                + " WHERE " + InAppMessageSqliteStore.TriggersTable.TABLE_NAME + "." + InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_EVENT_NAME + " = '" + triggeredEventName + "'"
                + " AND " + InAppMessageSqliteStore.TriggersTable.TABLE_NAME + "." + InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_TYPE + " = 'inappevent'";

        AbxLog.d("getCampaignIdHashMapFromTriggerEvent query:" + getCampaignIdWithPriority, true);

        Cursor cursor = db.rawQuery(getCampaignIdWithPriority, null);

        while (cursor.moveToNext()) {
            campaignIdHashMap.put(
                    cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_CAMPAIGN_ID)),
                    new Pair<>(
                            cursor.getLong(cursor.getColumnIndex(InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_PRIORITY)),
                            cursor.getLong(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_LAST_UPDATED_DATETIME))
                    )
            );
        }

        cursor.close();
        return campaignIdHashMap;
    }

    public InAppMessage getInAppMessageByCampaignId(String campaignId, boolean isPortrait) {
        InAppMessage inAppMessage = null;

        SQLiteDatabase db = dbHelper.getReadableDatabase();

        // Define a projection that specifies which columns from the database
        // you will actually use after this query.
        String[] projection = {
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_TYPE,
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_LAYOUT,
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_BG_STYLE_BG_COLOR,
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_BG_STYLE_OVERLAY_COLOR,
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CONTENTS_TEXT,
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CONTENTS_BUTTONS,
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_DEFAULT_CLOSE_BUTTON_COLOR,
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_DEFAULT_CLOSE_BUTTON_BG_COLOR,
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_STICKY_BANNER_OPTION
        };

        String selection;
        if (isPortrait) {
            selection = InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CAMPAIGN_ID + " = ? AND "
                    + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_PORTRAIT_AVAILABLE + " = ? ";
        } else {
            selection = InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CAMPAIGN_ID + " = ? AND "
                    + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_LANDSCAPE_AVAILABLE + " = ? ";
        }

        String[] selectionArgs = {campaignId, Integer.toString(1)};

        Cursor cursor = db.query(
                InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME,
                projection,
                selection,
                selectionArgs,
                null,
                null,
                null
        );

        if (cursor.moveToNext()) {
            HashMap<String, Object> imageInfoPortrait = null;
            HashMap<String, Object> imageInfoLandscape = null;
            String imageAction;
            if (isPortrait) {
                imageInfoPortrait = getImageInfo(campaignId, PORTRAIT, true);
                imageAction = imageInfoPortrait == null ? null : (String) imageInfoPortrait.get(IInAppMessageViewFactory.ImageInfoHashmapValue.CLICK_ACTION.getValue());
            } else {
                imageInfoLandscape = getImageInfo(campaignId, LANDSCAPE, true);
                imageAction = imageInfoLandscape == null ? null : (String) imageInfoLandscape.get(IInAppMessageViewFactory.ImageInfoHashmapValue.CLICK_ACTION.getValue());
            }
            inAppMessage = new InAppMessage(
                    campaignId,
                    cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_TYPE)),
                    cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_LAYOUT)),
                    imageInfoPortrait == null ? null : (Bitmap) imageInfoPortrait.get(IInAppMessageViewFactory.ImageInfoHashmapValue.IMAGE_BITMAP.getValue()),
                    imageInfoPortrait == null ? null : (String) imageInfoPortrait.get(IInAppMessageViewFactory.ImageInfoHashmapValue.IMAGE_URL.getValue()),
                    imageInfoPortrait == null ? null : (Boolean) imageInfoPortrait.get(IInAppMessageViewFactory.ImageInfoHashmapValue.IS_DOWNLOADED.getValue()),
                    imageInfoLandscape == null ? null : (Bitmap) imageInfoLandscape.get(IInAppMessageViewFactory.ImageInfoHashmapValue.IMAGE_BITMAP.getValue()),
                    imageInfoLandscape == null ? null : (String) imageInfoLandscape.get(IInAppMessageViewFactory.ImageInfoHashmapValue.IMAGE_URL.getValue()),
                    imageInfoLandscape == null ? null : (Boolean) imageInfoLandscape.get(IInAppMessageViewFactory.ImageInfoHashmapValue.IS_DOWNLOADED.getValue()),
                    imageAction,
                    cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_BG_STYLE_BG_COLOR)),
                    cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_BG_STYLE_OVERLAY_COLOR)),
                    cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CONTENTS_TEXT)),
                    cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CONTENTS_BUTTONS)),
                    cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_DEFAULT_CLOSE_BUTTON_COLOR)),
                    cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_DEFAULT_CLOSE_BUTTON_BG_COLOR)),
                    cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_STICKY_BANNER_OPTION))
            );
        }

        cursor.close();

        return inAppMessage;
    }

    public synchronized HashMap<String, Object> getAndSaveImageInfo(
            String campaignId,
            String orientation,
            boolean isImageBeUsed
    ) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();
        HashMap<String, Object> imageHashMap = new HashMap<>();
        Bitmap bitmapImage = null;

        String getImageQuery = "SELECT "
                + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IMAGE_PATH_STRING + ", "
                + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CLICK_ACTION + ", "
                + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_URL + ", "
                + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IS_FIRST_SHOW
                + " FROM "
                + InAppMessageSqliteStore.ImageTable.TABLE_NAME
                + " WHERE " + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CAMPAIGN_ID + " = '" + campaignId + "'"
                + " AND " + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_ORIENTATION + " = '" + orientation + "'";

        Cursor cursor = db.rawQuery(getImageQuery, null);

        if (cursor.moveToNext()) {
            String imagePathString = cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IMAGE_PATH_STRING));
            String clickAction = cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CLICK_ACTION));
            String imageUrlString = cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_URL));
            boolean isDownloaded = cursor.getInt(cursor.getColumnIndex(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IS_FIRST_SHOW)) == 1;

            cursor.close();

            if (imagePathString.equals(NO_IMAGE)) {
                URL url;
                HttpURLConnection connection = null;

                try {
                    url = new URL(imageUrlString);
                    connection = (HttpURLConnection) url.openConnection();
                    connection.setDoInput(true);
                    connection.connect();
                    InputStream input = connection.getInputStream();
                    bitmapImage = BitmapFactory.decodeStream(input);
                    if (bitmapImage != null)
                        imagePathString = saveToInternalStorage(bitmapImage, context);
                } catch (IOException e) {
                    if (connection != null) {
                        try {
                            AbxLog.d("InAppMessageDAO getAndSaveImageInfo: httpconnection error, responsecode: " + connection.getResponseCode() + ", responseMessage: " + connection.getResponseMessage() + ", errorMessage: " + e.toString(), true);
                        } catch (IOException ioException) {
                            AbxLog.d("InAppMessageDAO getAndSaveImageInfo IOException: " + e.getMessage(), true);
                        }
                    } else {
                        AbxLog.d("InAppMessageDAO getAndSaveImageInfo: url.openConnection error", true);
                    }
                    return null;
                }

                if (bitmapImage == null) {
                    AbxLog.d("InAppMessageDAO getAndSaveImageInfo: bitmapImage is null", true);
                    return null;
                } else if (imagePathString == null) {
                    AbxLog.d("InAppMessageDAO getAndSaveImageInfo: bitmapImage is null", true);
                    return null;
                }

                updateImageTable(campaignId, orientation, imageUrlString, imagePathString, clickAction, !isImageBeUsed);

                isDownloaded = true;
            } else if (isDownloaded && isImageBeUsed) {
                updateImageTable(campaignId, orientation, imageUrlString, imagePathString, clickAction, false);
                bitmapImage = getImageBitmapFromInternalStorage(context, imagePathString);
            } else {
                bitmapImage = getImageBitmapFromInternalStorage(context, imagePathString);
            }

            if (bitmapImage == null)
                return null;

            imageHashMap.put(IInAppMessageViewFactory.ImageInfoHashmapValue.IMAGE_BITMAP.getValue(), bitmapImage);
            imageHashMap.put(IInAppMessageViewFactory.ImageInfoHashmapValue.IMAGE_URL.getValue(), imageUrlString);
            imageHashMap.put(IInAppMessageViewFactory.ImageInfoHashmapValue.IS_DOWNLOADED.getValue(), isDownloaded);
            imageHashMap.put(IInAppMessageViewFactory.ImageInfoHashmapValue.CLICK_ACTION.getValue(), clickAction);

            return imageHashMap;
        } else {
            AbxLog.d("InAppMessage No image with campaignId = " + campaignId + ", orientation = " + orientation, true);
            cursor.close();

            return null;
        }
    }

    public HashMap<String, Object> getImageInfo(
            String campaignId,
            String orientation,
            boolean isImageBeUsed
    ) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();
        HashMap<String, Object> imageHashMap = new HashMap<>();

        String getImageQuery = "SELECT "
                + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IMAGE_PATH_STRING + ", "
                + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CLICK_ACTION + ", "
                + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_URL + ", "
                + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IS_FIRST_SHOW
                + " FROM "
                + InAppMessageSqliteStore.ImageTable.TABLE_NAME
                + " WHERE " + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CAMPAIGN_ID + " = '" + campaignId + "'"
                + " AND " + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_ORIENTATION + " = '" + orientation + "'";

        Cursor cursor = db.rawQuery(getImageQuery, null);

        if (cursor.moveToNext()) {
            String imagePathString = cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IMAGE_PATH_STRING));
            String clickAction = cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CLICK_ACTION));
            String imageUrlString = cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_URL));
            boolean isDownloaded = cursor.getInt(cursor.getColumnIndex(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IS_FIRST_SHOW)) == 1;

            cursor.close();

            if (NO_IMAGE.equals(imagePathString)) {
                return null;
            } else if (isDownloaded && isImageBeUsed) {
                updateImageTable(campaignId, orientation, imageUrlString, imagePathString, clickAction, false);
            }

            Bitmap imageBitmap = getImageBitmapFromInternalStorage(context, imagePathString);

            if (imageBitmap == null) {
                AbxLog.d("InAppMessage getImageInfo : can not get image from internal storage", true);
                return null;
            }

            imageHashMap.put(IInAppMessageViewFactory.ImageInfoHashmapValue.IMAGE_BITMAP.getValue(), imageBitmap);
            imageHashMap.put(IInAppMessageViewFactory.ImageInfoHashmapValue.IMAGE_URL.getValue(), imageUrlString);
            imageHashMap.put(IInAppMessageViewFactory.ImageInfoHashmapValue.IS_DOWNLOADED.getValue(), isDownloaded);
            imageHashMap.put(IInAppMessageViewFactory.ImageInfoHashmapValue.CLICK_ACTION.getValue(), clickAction);

            return imageHashMap;
        } else {
            AbxLog.d("InAppMessage No image with campaignId = " + campaignId + ", orientation = " + orientation, true);
            cursor.close();

            return null;
        }

    }

    public void clearInAppMessageDataBase() {
        dbHelper.onUpgrade(dbHelper.getWritableDatabase(), 0, 0);
    }

    public void closeDataBase() {
        dbHelper.getWritableDatabase().close();
    }

    public ArrayList<String> sortMapByValue(final HashMap<String, Pair<Long, Long>> map) {
        ArrayList<String> list = new ArrayList<>(map.keySet());
        Collections.sort(list, (key1, key2) -> {
            Pair<Long, Long> value1 = map.get(key1);
            Pair<Long, Long> value2 = map.get(key2);

            if (value1 == null || value2 == null) {
                AbxLog.e("sorMapByValue null value error", true);
                return 1;
            }

            if (value1.first.equals(value2.first)) {
                if (value1.second < value2.second)
                    return 1;
                else
                    return -1;
            } else {
                if (value1.first < value2.first)
                    return 1;
                else
                    return -1;
            }
        });

        return list;
    }

    public String saveToInternalStorage(Bitmap bitmapImage, Context applicationContext) {
        String fileName = UUID.randomUUID() + ".png";

        ContextWrapper contextWrapper = new ContextWrapper(applicationContext);
        // path to /data/data/yourapp/app_data/imageDir
        File imageDirectory = contextWrapper.getDir("imageDir", Context.MODE_PRIVATE);
        // Create imageDir
        File imagePath = new File(imageDirectory, fileName);

        try (FileOutputStream fileOutputStream = new FileOutputStream(imagePath)) {
            // Use the compress method on the BitMap object to write image to the OutputStream
            bitmapImage.compress(Bitmap.CompressFormat.PNG, 100, fileOutputStream);
        } catch (Exception e) {
            AbxLog.d("saveToInternalStorage error : " + e.toString(), true);
            return null;
        }

        return fileName;
    }

    public Bitmap getImageBitmapFromInternalStorage(Context applicationContext, String imagePathString) {
        ContextWrapper contextWrapper = new ContextWrapper(applicationContext);

        try {
            File imageDirectory = contextWrapper.getDir("imageDir", Context.MODE_PRIVATE);
            File imagePath = new File(imageDirectory, imagePathString);
            Bitmap imageBitmap = BitmapFactory.decodeStream(new FileInputStream(imagePath));
            return imageBitmap;
        } catch (Exception e) {
            AbxLog.d("getImageBitmapFromInternalStorage exception: " + e.toString(), true);
            return null;
        }
    }

    public boolean deleteImageFromInternalStorage(Context applicationContext, String fileName) {
        ContextWrapper contextWrapper = new ContextWrapper(applicationContext);

        try {
            File imageDirectory = contextWrapper.getDir("imageDir", Context.MODE_PRIVATE);
            File imageFile = new File(imageDirectory, fileName);
            return imageFile.delete();
        } catch (NullPointerException nullPointerException) {
            return false;
        }
    }
}
