package io.adbrix.sdk.data;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.Arrays;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.CoreUtils;

public class S3ConfigHandler {
    public JSONArray configBlockedEventList;
    public long configAppScanPeriod;
    public boolean configAppScanStop;
    public int configPropertyMaxSize = ConfigConstants.CONFIG_DEFAULT_PROPERTY_MAX_SIZE;
    public String configSkIpPort = ConfigConstants.CONFIG_DEFAULT_SK_IP_PORT;
    public int configSkPingVersion = ConfigConstants.CONFIG_DEFAULT_SK_PING_VERSION;
    public int configSkPingPeriodSecond = ConfigConstants.CONFIG_DEFAULT_SK_PING_PERIOD_SECOND;
    public boolean configInAppMessageActive = false;
    protected boolean configAdbrixPause;
    protected boolean configAdbrixAllStop;
    public JSONObject configEventSamplings;

    private S3ConfigHandler() {
    }

    public static S3ConfigHandler getInstance() {
        return Singleton.INSTANCE;
    }

    public void initDefaultConfigValue(DataRegistry dataRegistry) {
        long tempLong;
        String tempStringArray;
        String tempString;

        tempStringArray = dataRegistry.safeGetString(DataRegistryKey.STRING_S3_CONFIG_BLOCKED_EVTLIST, null);
        if (tempStringArray != null) {
            try {
                configBlockedEventList = new JSONArray(tempStringArray);
            } catch (JSONException e) {
                AbxLog.e(Arrays.toString(e.getStackTrace()), true);
            }
        } else configBlockedEventList = ConfigConstants.CONFIG_DEFAULT_BLOCKED_EVENT_LIST;

        tempLong = dataRegistry.safeGetLong(DataRegistryKey.LONG_S3_CONFIG_APP_SCAN_PERIOD, -1);
        if (tempLong == -1)
            configAppScanPeriod = ConfigConstants.CONFIG_DEFAULT_APP_SCAN_PERIOD * 3600 * 1000;
        else configAppScanPeriod = (int) tempLong * 3600 * 1000;

        tempLong = dataRegistry.safeGetLong(DataRegistryKey.LONG_S3_CONFIG_APP_SCAN_STOP, -1);
        if (tempLong == -1) configAppScanStop = ConfigConstants.CONFIG_DEFAULT_APP_SCAN_ON_OFF;
        else configAppScanStop = tempLong == 1;

        tempLong = dataRegistry.safeGetLong(DataRegistryKey.LONG_S3_CONFIG_PROPERTY_KEY_MAX, -1);
        if (tempLong == -1)
            configPropertyMaxSize = ConfigConstants.CONFIG_DEFAULT_PROPERTY_MAX_SIZE;
        else configPropertyMaxSize = (int) tempLong;

        tempLong = dataRegistry.safeGetLong(DataRegistryKey.LONG_S3_CONFIG_ADBRIX_PAUSE, -1);
        if (tempLong == -1) configAdbrixPause = ConfigConstants.CONFIG_DEFAULT_ADBRIX_PAUSE;
        else configAdbrixPause = tempLong == 1;

        tempLong = dataRegistry.safeGetLong(DataRegistryKey.LONG_S3_CONFIG_ADBRIX_ALL_STOP, -1);
        if (tempLong == -1)
            configAdbrixAllStop = ConfigConstants.CONFIG_DEFAULT_ADBRIX_ALL_STOP;
        else configAdbrixAllStop = tempLong == 1;

        if (configAdbrixAllStop) configAdbrixPause = true;

        tempString = dataRegistry.safeGetString(DataRegistryKey.STRING_S3_CONFIG_SK_IP_PORT, null);
        if (tempString != null) {
            configSkIpPort = tempString;
        } else configSkIpPort = ConfigConstants.CONFIG_DEFAULT_SK_IP_PORT;

        tempLong = dataRegistry.safeGetInt(DataRegistryKey.INT_S3_CONFIG_SK_PING_VERSION, -1);
        if (tempLong == -1) {
            configSkPingVersion = ConfigConstants.CONFIG_DEFAULT_SK_PING_VERSION;
        } else configSkPingVersion = (int) tempLong;

        tempLong = dataRegistry.safeGetInt(DataRegistryKey.INT_S3_CONFIG_SK_PING_PERIOD_SECOND, -1);
        if (tempLong == -1)
            configSkPingPeriodSecond = ConfigConstants.CONFIG_DEFAULT_SK_PING_PERIOD_SECOND * 1000;
        else configSkPingPeriodSecond = (int) tempLong * 1000;

        tempLong = dataRegistry.safeGetLong(DataRegistryKey.LONG_S3_CONFIG_IN_APP_MESSAGE_ACTIVE, -1);
        if (tempLong == -1)
            configInAppMessageActive = ConfigConstants.CONFIG_DEFAULT_IN_APP_MESSAGE_ACTIVE;
        else configInAppMessageActive = tempLong == 1;

        tempString = dataRegistry.safeGetString(DataRegistryKey.STRING_S3_CONFIG_EVENT_SAMPLING_DATA, null);
        if (tempString != null) {
            try {
                configEventSamplings = new JSONObject(tempString);
            } catch (JSONException e) {
                AbxLog.e(Arrays.toString(e.getStackTrace()), true);
            }
        } else configEventSamplings = new JSONObject();

        AbxLog.d("Initialized default value - adbrix pause       : " + configAdbrixPause, true);
        AbxLog.d("Initialized default value - banned event list  : " + configBlockedEventList, true);
//        AbxLog.d("Initialized default value - app scan period    : " + config_appScanPeriod/3600/1000,    true);
//        AbxLog.d("Initialized default value - app scan stop      : " + config_appScanStop,                true);
        AbxLog.d("Initialized default value - property max size  : " + configPropertyMaxSize, true);
        AbxLog.d("Initialized default value - adbrix all stop    : " + configAdbrixAllStop, true);
        AbxLog.d("Initialized default value - sk ip port         : " + configSkIpPort, true);
        AbxLog.d("Initialized default value - sk ping version    : " + configSkPingVersion, true);
        AbxLog.d("Initialized default value - sk ping period sec : " + configSkPingPeriodSecond / 1000, true);
        AbxLog.d("Initialized default value - inAppMessage active : " + configInAppMessageActive, true);
        AbxLog.d("Initialized default value - eventSamplings : " + configEventSamplings.toString(), true);
    }

    public void parseResponseOfS3Config(String responseString, DataRegistry dataRegistry) {

        try {
            int temp;
            JSONObject jsonObjectFromS3Server;

            if (!CommonUtils.isNullOrEmpty(responseString))
                jsonObjectFromS3Server = new JSONObject(responseString);
            else return;

            temp = getDecryptedConfigValue(jsonObjectFromS3Server, ConfigConstants.S3_CONFIG_ADBRIX_PAUSE);
            if (temp != -1) {
                configAdbrixPause = temp == 1;

                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.LONG_S3_CONFIG_ADBRIX_PAUSE,
                                (long) temp,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
                AbxLog.d("CONFIG::ADBRIX PAUSE : " + configAdbrixPause, true);
            }

            temp = getDecryptedConfigValue(jsonObjectFromS3Server, ConfigConstants.S3_CONFIG_ADBRIX_ALL_STOP);
            if (temp != -1) {
                configAdbrixAllStop = temp == 1;

                if (configAdbrixAllStop) {
                    configAdbrixPause = true;

                    dataRegistry.putDataRegistry(
                            new DataUnit(
                                    DataRegistryKey.LONG_S3_CONFIG_ADBRIX_PAUSE,
                                    (long) temp,
                                    5,
                                    this.getClass().getName(),
                                    true
                            )
                    );
                }

                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.LONG_S3_CONFIG_ADBRIX_ALL_STOP,
                                (long) temp,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
                AbxLog.d("CONFIG::ADBRIX ALL STOP : " + configAdbrixAllStop, true);
            }

            temp = getDecryptedConfigValue(jsonObjectFromS3Server, ConfigConstants.S3_CONFIG_APP_SCAN_PERIOD);
            if (temp != -1) {
                configAppScanPeriod = temp * 3600 * 1000; // period : per hour

                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.LONG_S3_CONFIG_APP_SCAN_PERIOD,
                                (long) temp,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
//                AbxLog.d("CONFIG::APPSCAN PERIOD : " + temp, true);

                AbxLog.d("AbxApplicationScan:: DEPRECATED!!!", true);
            }

            temp = getDecryptedConfigValue(jsonObjectFromS3Server, ConfigConstants.S3_CONFIG_APP_SCAN_STOP);
            if (temp != -1) {
                configAppScanStop = temp == 1;

                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.LONG_S3_CONFIG_APP_SCAN_STOP,
                                (long) temp,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
//                AbxLog.d("CONFIG::APPSCAN STOP : " + config_appScanStop, true);
                AbxLog.d("AbxApplicationScan:: DEPRECATED!!!", true);
            }

            temp = getDecryptedConfigValue(jsonObjectFromS3Server, ConfigConstants.S3_CONFIG_PROPERTY_KEY_MAX);
            if (temp != -1) {
                configPropertyMaxSize = temp;

                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.LONG_S3_CONFIG_PROPERTY_KEY_MAX,
                                (long) temp,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
                AbxLog.d("CONFIG::PROPERTY MAX SIZE : " + configPropertyMaxSize, true);
            }

            if (jsonObjectFromS3Server.has(ConfigConstants.S3_CONFIG_BLOCKED_EVTLIST)) {
                if (jsonObjectFromS3Server.get(ConfigConstants.S3_CONFIG_BLOCKED_EVTLIST) instanceof JSONArray) {
                    configBlockedEventList = (JSONArray) jsonObjectFromS3Server.get(ConfigConstants.S3_CONFIG_BLOCKED_EVTLIST);

                    dataRegistry.putDataRegistry(
                            new DataUnit(
                                    DataRegistryKey.STRING_S3_CONFIG_BLOCKED_EVTLIST,
                                    configBlockedEventList.toString(),
                                    5,
                                    this.getClass().getName(),
                                    true
                            )
                    );
                    AbxLog.d("CONFIG::BANNED LIST : " + configBlockedEventList.toString(), true);
                }
            }

            if (jsonObjectFromS3Server.has(ConfigConstants.S3_CONFIG_SK_IP_PORT)) {
                if (jsonObjectFromS3Server.get(ConfigConstants.S3_CONFIG_SK_IP_PORT) instanceof String) {
                    configSkIpPort = (String) jsonObjectFromS3Server.get(ConfigConstants.S3_CONFIG_SK_IP_PORT);
                    if (configSkIpPort.length() != 17) {
                        AbxLog.d("CONFIG::SK IP PORT : " + "Wrong SK_IP_PORT!!", true);
                    } else {
                        dataRegistry.putDataRegistry(
                                new DataUnit(
                                        DataRegistryKey.STRING_S3_CONFIG_SK_IP_PORT,
                                        configSkIpPort,
                                        5,
                                        this.getClass().getName(),
                                        true
                                )
                        );
                        AbxLog.d("CONFIG::SK IP PORT : " + configSkIpPort, true);
                    }
                }
            }

            temp = getDecryptedConfigValue(jsonObjectFromS3Server, ConfigConstants.S3_CONFIG_SK_PING_VERSION);
            if (temp != -1) {
                configSkPingVersion = temp;

                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.INT_S3_CONFIG_SK_PING_VERSION,
                                temp,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
                AbxLog.d("CONFIG::SK PING VER : " + configSkPingVersion, true);
            }

            temp = getDecryptedConfigValue(jsonObjectFromS3Server, ConfigConstants.S3_CONFIG_SK_PING_PERIOD_SECOND);
            if (temp != -1) {
                configSkPingPeriodSecond = temp;

                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.INT_S3_CONFIG_SK_PING_PERIOD_SECOND,
                                temp,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
                AbxLog.d("CONFIG::SK PING PERIOD SEC : " + configSkPingPeriodSecond, true);
            }

            temp = getDecryptedConfigValue(jsonObjectFromS3Server, ConfigConstants.S3_CONFIG_IN_APP_MESSAGE_ACTIVE);
            if (temp != -1) {
                configInAppMessageActive = temp == 1;

                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.LONG_S3_CONFIG_IN_APP_MESSAGE_ACTIVE,
                                (long) temp,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
                AbxLog.d("CONFIG::InAppMessage Active : " + configInAppMessageActive, true);
            }

            if (jsonObjectFromS3Server.optJSONObject(ConfigConstants.S3_CONFIG_EVENT_SAMPLING) != null) {
                JSONObject eventSamplingJson = jsonObjectFromS3Server.getJSONObject(ConfigConstants.S3_CONFIG_EVENT_SAMPLING);

                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.STRING_S3_CONFIG_EVENT_SAMPLING_DATA,
                                eventSamplingJson.toString(),
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
                AbxLog.d("CONFIG::EVENT SAMPLING FILTER : " + eventSamplingJson.toString(), true);
            }

            dataRegistry.saveRegistry();

        } catch (JSONException e) {
            AbxLog.d("CONFIG : The format received from server is not valid. Can't create new JSONObject.", true);
            AbxLog.d(Arrays.toString(e.getStackTrace()), true);
        }
    }

    private int getDecryptedConfigValue(JSONObject objFromServer, String type) {
        String md5Hash;
        if (objFromServer.has(type)) {
            try {
                Object temp = objFromServer.get(type);
                String tempToString = "";
                if (temp instanceof String) tempToString = (String) temp;
                md5Hash = CoreUtils.get02XMD5(tempToString);
                if (ConfigConstants.S3_KEY.containsKey(md5Hash)) {
                    return ConfigConstants.S3_KEY.get(md5Hash);
                }
            } catch (JSONException e) {
                AbxLog.e(Arrays.toString(e.getStackTrace()), true);
                return -1;
            }
        }
        return -1;
    }

    public boolean isBlockedEvent(String eventName, String category) {

        if (configBlockedEventList != null && configBlockedEventList.length() != 0) {
            for (int i = 0; i < configBlockedEventList.length(); i++) {
                try {
                    String bannedEvent = configBlockedEventList.getString(i);
                    String[] evtSplitByCategory = bannedEvent.split(":");

                    if (evtSplitByCategory.length != 2) continue;
                    if (!evtSplitByCategory[0].equals("abx") && !evtSplitByCategory[0].equals("custom"))
                        continue;
                    if (category.equals(CoreConstants.GROUP_CUSTOM) && evtSplitByCategory[0].equals("custom")) {
                        if (evtSplitByCategory[1].equals(eventName)) {
                            AbxLog.d("S3ConfigHandler :: custom event is blocked ! : " + eventName, true);
                            return true;
                        }
                    } else if (category.equals(CoreConstants.GROUP_ABX) && evtSplitByCategory[0].equals("abx")) {
                        if (evtSplitByCategory[1].equals(eventName)) {
                            AbxLog.d("S3ConfigHandler :: ABX event is blocked ! : " + eventName, true);
                            return true;
                        }
                    }
                } catch (JSONException e) {
                    AbxLog.d("Blocked Event Exception " + e.getMessage(), true);
                    AbxLog.d(Arrays.toString(e.getStackTrace()), true);
                }
            }
        }
        return false;
    }

    private static class Singleton {
        private static final S3ConfigHandler INSTANCE = new S3ConfigHandler();
    }
}
