package io.adbrix.sdk.configuration;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;

import org.json.JSONObject;

import java.util.Arrays;
import java.util.List;

import javax.annotation.Nullable;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.IABXComponentsFactory;
import io.adbrix.sdk.component.IObserver;
import io.adbrix.sdk.component.TryOptional;
import io.adbrix.sdk.component.autoEvent.DeeplinkAutoEventGenerator;
import io.adbrix.sdk.component.autoEvent.SessionAutoEventGenerator;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.LogMessageFormat;
import io.adbrix.sdk.domain.Repository;
import io.adbrix.sdk.domain.interactor.ClearSyncedActionHistoryInLocalDBUseCase;
import io.adbrix.sdk.domain.interactor.ClearAllActionHistoryInLocalDBUseCase;
import io.adbrix.sdk.domain.interactor.DeleteActionHistoryUseCase;
import io.adbrix.sdk.domain.interactor.DeleteAllActionHistoryUseCase;
import io.adbrix.sdk.domain.interactor.DeleteUserDataUseCase;
import io.adbrix.sdk.domain.interactor.FetchActionHistoryFromServerUseCase;
import io.adbrix.sdk.domain.interactor.FlushAllNowUseCase;
import io.adbrix.sdk.domain.interactor.FlushAtIntervalsUseCase;
import io.adbrix.sdk.domain.interactor.GetActionHistoryUseCase;
import io.adbrix.sdk.domain.interactor.GetAllActionHistoryUseCase;
import io.adbrix.sdk.domain.interactor.InsertPushDataUseCase;
import io.adbrix.sdk.domain.interactor.LogEventUseCase;
import io.adbrix.sdk.domain.interactor.LogSameEventWithPagingUseCase;
import io.adbrix.sdk.domain.interactor.RegisterNetworkCallbackUseCase;
import io.adbrix.sdk.domain.interactor.RestartSDKUseCase;
import io.adbrix.sdk.domain.interactor.SaveUnsentEventsUseCase;
import io.adbrix.sdk.domain.interactor.SaveUserPropertyUseCaseWithoutEvent;
import io.adbrix.sdk.domain.model.ActionHistory;
import io.adbrix.sdk.domain.model.ActionHistoryIdType;
import io.adbrix.sdk.domain.function.Completion;
import io.adbrix.sdk.domain.model.Empty;
import io.adbrix.sdk.domain.model.Error;
import io.adbrix.sdk.domain.model.Result;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.domain.model.UserPropertyCommand;
import io.adbrix.sdk.utils.CoreUtils;

public class ActiveABXContext implements IABXContext, IObserver<LogEventParameter> {
    private Context androidContext;
    private Repository repository;
    private IABXComponentsFactory componentsFactory;

    public ActiveABXContext(
            IABXComponentsFactory componentsFactory,
            Repository repository
    ) {
        this.componentsFactory = componentsFactory;
        this.repository = repository;

        TryOptional.of(componentsFactory::createOrGetSessionAutoEventGenerator)
                .ifPresent(it -> it.add(this));
        TryOptional.of(componentsFactory::createOrGetDeeplinkAutoEventGenerator)
                .ifPresent(it -> it.add(this));
        TryOptional.of(componentsFactory::createOrGetDailyFirstOpenAutoEventGenerator)
                .ifPresent(it -> it.add(this));

        TryOptional.of(componentsFactory::getAndroidContext).ifPresent(context -> {
            if (CoreUtils.isOnline(context)) {
                new FlushAllNowUseCase(repository).execute();
            }
        });

        new RegisterNetworkCallbackUseCase(repository).execute();
    }

    @Override
    public void update(LogEventParameter logEventParameter) {
        this.logEvent(logEventParameter);
    }

    @Override
    public Result<String> initialize(Context context, String appkey, String secretkey) {
        AbxLog.w(String.format(LogMessageFormat.INVALID_CONTEXT, "Active", "initialize"), true);
        return Error.of(String.format(LogMessageFormat.INVALID_CONTEXT, "Active", "initialize"));
    }

    @Override
    public void saveUserProperty(UserPropertyCommand userPropertyCommand) {

        if (saveUserPropertyWithoutEvent(userPropertyCommand)) {
            LogEventParameter eventParameter = new LogEventParameter(
                    CoreConstants.GROUP_ABX,
                    CoreConstants.EVENT_USER_PROPERTY_CHANGED,
                    null,
                    0,
                    0
            );

            logEvent(eventParameter);
        }
    }

    @Override
    public Boolean saveUserPropertyWithoutEvent(UserPropertyCommand userPropertyCommand) {
        return new SaveUserPropertyUseCaseWithoutEvent(repository).execute(userPropertyCommand);
    }

    @Override
    public void logEvent(LogEventParameter logEventParameter) {
        new LogEventUseCase(repository).execute(logEventParameter);
    }

    @Override
    public void logSameEventWithPaging(String eventName, List<JSONObject> eventParamList) {
        new LogSameEventWithPagingUseCase(repository).execute(eventName, eventParamList);
    }

    @Override
    public void runOnIdleTime() {
        new FlushAtIntervalsUseCase(repository).execute();
    }

    @Override
    public void onResume(Activity activity) {
        TryOptional.of(componentsFactory::createOrGetSessionAutoEventGenerator)
                .ifPresent(it -> it.onResume(activity));

        TryOptional.of(componentsFactory::createOrGetDailyFirstOpenAutoEventGenerator)
                .ifPresent(it -> it.process());
    }

    @Override
    public void onPause() {
        TryOptional.of(componentsFactory::createOrGetSessionAutoEventGenerator)
                .ifPresent(SessionAutoEventGenerator::onPause);

        TryOptional.of(componentsFactory::getAndroidContext).ifPresent(context -> {
            if (CoreUtils.isOnline(context)) {
                new FlushAllNowUseCase(repository).execute();
            }
        });

        new SaveUnsentEventsUseCase(repository).execute();

        TryOptional.of(componentsFactory::createOrGetDataRegistry)
                .ifPresent(DataRegistry::saveRegistry);
    }

    @Override
    public void deeplink(Activity deeplinkActivity) {
        try {
            SessionAutoEventGenerator sessionAutoEventGenerator = componentsFactory.createOrGetSessionAutoEventGenerator();
            DeeplinkAutoEventGenerator deeplinkAutoEventGenerator = componentsFactory.createOrGetDeeplinkAutoEventGenerator();

            if (sessionAutoEventGenerator.isStartSessionByDeeplink(deeplinkActivity)) {
                return;
            }
            deeplinkAutoEventGenerator.process(deeplinkActivity);

        } catch (IABXComponentsFactory.ComponentsCanNotCreateException e) {
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
        }
    }

    @Override
    public void deeplinkWithIntent(Intent deeplinkIntent) {
        try {
            SessionAutoEventGenerator sessionAutoEventGenerator = componentsFactory.createOrGetSessionAutoEventGenerator();
            DeeplinkAutoEventGenerator deeplinkAutoEventGenerator = componentsFactory.createOrGetDeeplinkAutoEventGenerator();

            if (sessionAutoEventGenerator.isStartSessionByDeeplinkIntent(deeplinkIntent)) {
                return;
            }
            deeplinkAutoEventGenerator.process(deeplinkIntent);

        } catch (IABXComponentsFactory.ComponentsCanNotCreateException e) {
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
        }
    }

    @Override
    public void putDataRegistry(DataUnit dataUnit) {
        TryOptional.of(componentsFactory::createOrGetDataRegistry)
                .ifPresent(dataRegistry -> dataRegistry.putDataRegistry(dataUnit));
    }

    @Override
    public Result<Empty> deleteUserData(String userId) {
        AbxLog.d("Delete user data api called!", true);

        return new DeleteUserDataUseCase(repository, userId).execute();
    }

    @Override
    public Result<Empty> restartSDK(String userId) {
        AbxLog.d("Restart SDK api called!", true);

        return new RestartSDKUseCase(repository, userId).execute();
    }

    @Override
    public void fetchActionHistoryFromServer(@Nullable String token, ActionHistoryIdType idType, List<String> actionType, Completion<Result<List<ActionHistory>>> completion) {
        new FetchActionHistoryFromServerUseCase(repository, token, idType, actionType, completion).execute();
    }

    @Override
    public void insertPushData(String pushDataString) {
        new InsertPushDataUseCase(repository, pushDataString).execute();
    }

    @Override
    public void getActionHistory(int skip, int limit, List<String> actionType, Completion<Result<List<ActionHistory>>> completion) {
        new GetActionHistoryUseCase(repository, skip, limit, actionType, completion).execute();
    }

    @Override
    public void getAllActionHistory(List<String> actionType, Completion<Result<List<ActionHistory>>> completion) {
        new GetAllActionHistoryUseCase(repository, actionType, completion).execute();
    }

    @Override
    public void deleteActionHistory(@Nullable String token, String historyId, long timestamp, Completion<Result<Empty>> completion) {
        new DeleteActionHistoryUseCase(repository, token, historyId, timestamp, completion).execute();
    }

    @Override
    public void deleteAllActionHistory(@Nullable String token, @Nullable ActionHistoryIdType idType, Completion<Result<Empty>> completion) {
        new DeleteAllActionHistoryUseCase(repository, token, idType, completion).execute();
    }

    @Override
    public void clearSyncedActionHistoryInLocalDB(Completion<Result<Empty>> completion) {
        new ClearSyncedActionHistoryInLocalDBUseCase(repository, completion).execute();
    }

    @Override
    public void clearAllActionHistoryInLocalDB() {
        new ClearAllActionHistoryInLocalDBUseCase(repository).execute();
    }
}
