package io.adbrix.sdk.component.autoEvent;

import android.app.Activity;
import android.content.Intent;
import android.net.Uri;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.DeeplinkParameterSet;
import io.adbrix.sdk.component.DeeplinkPostingObservable;
import io.adbrix.sdk.component.IObservable;
import io.adbrix.sdk.component.IObserver;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.CoreUtils;

public class DeeplinkAutoEventGenerator implements IObservable<LogEventParameter> {
    private DataRegistry dataRegistry;
    private DeeplinkParameterSet deeplinkParameterSet;
    private List<IObserver<LogEventParameter>> observers;

    public DeeplinkAutoEventGenerator(
            DataRegistry dataRegistry,
            DeeplinkParameterSet deeplinkParameterSet
    ) {
        this.dataRegistry = dataRegistry;
        this.deeplinkParameterSet = deeplinkParameterSet;
        observers = new ArrayList<>();
    }

    @Override
    public void add(IObserver<LogEventParameter> observer) {
        observers.add(observer);
    }

    @Override
    public void delete(IObserver<LogEventParameter> observer) {
        observers.remove(observer);
    }

    @Override
    public void notifyObserver(LogEventParameter logEventParameter) {
        for (IObserver<LogEventParameter> observer : observers) {
            observer.update(logEventParameter);
        }
    }

    public void process(Activity deeplinkActivity) {
        if (!CoreUtils.isActivityStartedByDeeplink(deeplinkActivity, dataRegistry)) {
            AbxLog.e("Check adbrix Parameters :: There is no adbrix deeplink parameter in Activity!", true);
            return;
        }

        Uri uri = CoreUtils.getDeeplinkUriFromCurrentActivity(deeplinkActivity);

        if (uri == null) {
            AbxLog.d("deeplink uri is null", true);
            return;
        }

        String deeplinkUrl = uri.toString();

        Map<String, Object> eventParam = CoreUtils.getDeeplinkParameterFromUriString(deeplinkUrl);

        if (deeplinkParameterSet.isAlreadyUsedDeeplinkParameter(eventParam)) {
            return;
        }

        LogEventParameter eventParameter = new LogEventParameter(
                CoreConstants.GROUP_ABX,
                CoreConstants.EVENT_DEEPLINK_OPEN,
                CommonUtils.parseValueWithDataType(eventParam, CommonUtils.FixType.PREFIX),
                0,
                0);

        dataRegistry.putDataRegistry(
                new DataUnit(
                        DataRegistryKey.STRING_LAST_DEEPLINK_ID,
                        eventParameter.eventId,
                        5,
                        this.getClass().getName(),
                        true
                ));

        dataRegistry.putDataRegistry(
                new DataUnit(
                        DataRegistryKey.STRING_LAST_OPEN_ID,
                        eventParameter.eventId,
                        5,
                        this.getClass().getName(),
                        true
                ));

        notifyObserver(eventParameter);

        DeeplinkPostingObservable.getInstance().notifyObserver(CoreUtils.getDeeplinkUriFromCurrentActivity(deeplinkActivity).toString());

        CoreUtils.clearAdbrixTrackingParamFromCurrentActivity(deeplinkActivity, dataRegistry);
    }

    public void process(Intent deeplinkIntent) {
        if (!CoreUtils.isStartedByDeeplinkIntent(deeplinkIntent, dataRegistry)) {
            AbxLog.e("Check adbrix Parameters :: There is no adbrix deeplink parameter in Intent!", true);
            return;
        }

        Uri uri = CoreUtils.getDeeplinkUriFromCurrentIntent(deeplinkIntent);

        if (uri == null) {
            AbxLog.d("deeplink uri is null", true);
            return;
        }

        String deeplinkUrl = uri.toString();

        Map<String, Object> eventParam = CoreUtils.getDeeplinkParameterFromUriString(deeplinkUrl);

        if (deeplinkParameterSet.isAlreadyUsedDeeplinkParameter(eventParam)) {
            return;
        }

        LogEventParameter eventParameter = new LogEventParameter(
                CoreConstants.GROUP_ABX,
                CoreConstants.EVENT_DEEPLINK_OPEN,
                CommonUtils.parseValueWithDataType(eventParam, CommonUtils.FixType.PREFIX),
                0,
                0);

        dataRegistry.putDataRegistry(
                new DataUnit(
                        DataRegistryKey.STRING_LAST_DEEPLINK_ID,
                        eventParameter.eventId,
                        5,
                        this.getClass().getName(),
                        true
                ));

        dataRegistry.putDataRegistry(
                new DataUnit(
                        DataRegistryKey.STRING_LAST_OPEN_ID,
                        eventParameter.eventId,
                        5,
                        this.getClass().getName(),
                        true
                ));

        notifyObserver(eventParameter);

        DeeplinkPostingObservable.getInstance().notifyObserver(CoreUtils.getDeeplinkUriFromCurrentIntent(deeplinkIntent).toString());

        CoreUtils.clearAdbrixTrackingParamFromCurrentIntent(deeplinkIntent, dataRegistry);
    }
}
