package io.adbrix.sdk.component;

import java.util.Timer;
import java.util.TimerTask;

import javax.annotation.Nullable;

import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.net.SocketClient;
import io.adbrix.sdk.data.net.SocketClientManager;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.utils.Base64;
import io.adbrix.sdk.utils.CoreUtils;

public class SKDataSender implements ITimerManager {
    @Nullable
    private Timer wifiTimer;
    private DataRegistry dataRegistry;
    private SocketClientManager clientManager;

    public SKDataSender(DataRegistry dataRegistry) {
        this.dataRegistry = dataRegistry;
        TimerManager.getInstance().add(this);

        clientManager = new SocketClientManager(dataRegistry, new SocketClientManager.Result() {
            @Override
            public void connectSuccess(int responseCode) {
                //SK 공유기의 로컬 아이피에 연결이 성공했을 때

                String ipAndPort = dataRegistry.safeGetString(DataRegistryKey.STRING_S3_CONFIG_SK_IP_PORT, "12700000000104200");
                int pingPeriodSecond = dataRegistry.safeGetInt(DataRegistryKey.INT_S3_CONFIG_SK_PING_PERIOD_SECOND, 300);

                scheduleWifiTimer(ipAndPort, pingPeriodSecond);
            }

            @Override
            public void connectFail() {
                //SK 공유기의 로컬 아이피에 연결이 실패했을 때
                AbxLog.d("SKB:: Connect fail! Cancel repeat!", true);
                setTimerToNull();
            }
        });
    }

    //multi thread에서 timer에 동시 접근하는 것을 막기 위해 synchronized 처리
    //https://iga-dev.atlassian.net/wiki/spaces/DFNDEV/pages/2729738241/2022-1-21+Android+SDK+2.1.2.5+QA+Guide
    private synchronized void scheduleWifiTimer(String ipAndPort, int pingPeriodSecond) {
        if (wifiTimer == null) {
            wifiTimer = new Timer();
        }

        wifiTimer.schedule(new TimerTask() {
            @Override
            public void run() {

                SocketClient client = new SocketClient(ipAndPort, getMessage());

                client.run();

                if (client.getResponseCode() == 200) {
                    AbxLog.d("SKB:: Sending igaw_id is complete.", true);
                } else if (client.getResponseCode() == 201) {
                    AbxLog.d("SKB:: Sending igaw_id is skipped.", true);
                } else {
                    AbxLog.d("SKB:: Sending igaw_id is failed.", true);
                    cancelConnectToSKB();
                }
            }
        }, pingPeriodSecond * 1000, pingPeriodSecond * 1000);
        AbxLog.d("SKB:: Connect success, ping period " + pingPeriodSecond + "s, start repeat!", true);
    }

    public void wifiDisconnectAction() {
        AbxLog.d("SKB:: Wifi is disconnected! Cancel repeat.", true);
        setTimerToNull();
    }

    public void wifiConnectAction() {
        if (CoreUtils.isGdprForgetMe(dataRegistry, () -> { })) {
            AbxLog.d("SKB:: Cannot connect due to gdpr setting || DRState. Cancel repeat.", true);
            return;
        }

        AbxLog.d("SKB:: Wifi is connected!", true);

        String ipAndPort = dataRegistry.safeGetString(DataRegistryKey.STRING_S3_CONFIG_SK_IP_PORT, "12700000000104200");

        SocketClient client = new SocketClient(ipAndPort, getMessage());

        clientManager.execute(client);
    }

    private String getMessage() {
        String igawId = dataRegistry.safeGetString(DataRegistryKey.STRING_IGAW_ID, null);
        String gaid = dataRegistry.safeGetString(DataRegistryKey.STRING_GAID, null);
        int skPingVersion = dataRegistry.safeGetInt(DataRegistryKey.INT_S3_CONFIG_SK_PING_VERSION, -1);

        switch (skPingVersion) {
            case 0:
                return igawId;
            case 1:
                if (gaid != null)
                    return "0001" + gaid + "0036" + Base64.encode("\"os\":\"0\",\"ccode\":\"C03000\"");
                else
                    return null;
            default:
                return null;
        }
    }

    private void cancelConnectToSKB() {
        setTimerToNull();
    }

    //multi thread에서 timer에 동시 접근하는 것을 막기 위해 synchronized 처리
    //https://iga-dev.atlassian.net/wiki/spaces/DFNDEV/pages/2729738241/2022-1-21+Android+SDK+2.1.2.5+QA+Guide
    @Override
    public synchronized void setTimerToNull() {
        if (wifiTimer != null) {
            wifiTimer.cancel();
            wifiTimer = null;
        }
    }
}
