package io.adbrix.sdk.component;

import android.content.Context;
import android.net.Uri;
import android.os.RemoteException;

import com.android.installreferrer.api.InstallReferrerClient;
import com.android.installreferrer.api.InstallReferrerStateListener;
import com.android.installreferrer.api.ReferrerDetails;

import java.util.Arrays;
import java.util.concurrent.CountDownLatch;

import io.adbrix.sdk.utils.CommonUtils;

public class GooglePlayReferrerProperties {
    public String referrerUrl;
    public long referrerClickTime = -1;
    public long appInstallTime = -1;
    public String clickAdKey;
    private Context context;

    public GooglePlayReferrerProperties(Context context) {
        this.context = context;
    }

    public void startConnection() {
        try {
            InstallReferrerClient referrerClient = InstallReferrerClient.newBuilder(context).build();
            CountDownLatch latch = new CountDownLatch(1);

            referrerClient.startConnection(new InstallReferrerStateListener() {
                @Override
                public void onInstallReferrerSetupFinished(int responseCode) {
                    switch (responseCode) {
                        case InstallReferrerClient.InstallReferrerResponse.OK:
                            onInstallReferrerResponseOK(referrerClient);
                            referrerClient.endConnection();
                            latch.countDown();
                            break;
                        default:
                            // API not available on the current Play Store app. or Connection couldn't be established.
                            onInstallReferrerResponseError(responseCode);
                            referrerClient.endConnection();
                            latch.countDown();
                            break;
                    }
                }

                @Override
                public void onInstallReferrerServiceDisconnected() {
                    // Try to restart the connection on the next request to
                    // Google Play by calling the startConnection() method.
                    onInstallReferrerResponseError(InstallReferrerClient.InstallReferrerResponse.SERVICE_DISCONNECTED);
                    latch.countDown();
                }
            });

            latch.await();
        } catch (Exception e) {
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
        }
    }

    private void onInstallReferrerResponseOK(InstallReferrerClient referrerClient) {
        try {
            ReferrerDetails response = referrerClient.getInstallReferrer();
            if (response != null) {
                referrerUrl = response.getInstallReferrer();
                if (!CommonUtils.isNullOrEmpty(referrerUrl)) {
                    try {
                        String abxTid = Uri.parse("http://referrer.ad-brix.com?" + referrerUrl).getQueryParameter("abx_tid");
                        if (!CommonUtils.isNullOrEmpty(abxTid)) {
                            clickAdKey = abxTid;
                        }
                    } catch (Exception e) {
                        AbxLog.e(Arrays.toString(e.getStackTrace()), true);
                    }
                }

                long referrerClickTimestampSeconds = response.getReferrerClickTimestampSeconds();
                if (referrerClickTimestampSeconds != 0) {
                    referrerClickTime = referrerClickTimestampSeconds;
                }

                long installBeginTimestampSeconds = response.getInstallBeginTimestampSeconds();
                if (installBeginTimestampSeconds != 0) {
                    appInstallTime = installBeginTimestampSeconds;
                }
            }
        } catch (RemoteException e) {
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
        }
    }

    private void onInstallReferrerResponseError(int responseCode) {
        switch (responseCode) {
            case -1:
                AbxLog.d("InstallReferrerResponse : SERVICE_DISCONNECTED", true);
                break;
            case 0:
                AbxLog.d("InstallReferrerResponse : OK", true);
                break;
            case 1:
                AbxLog.d("InstallReferrerResponse : SERVICE_UNAVAILABLE", true);
                break;
            case 2:
                AbxLog.d("InstallReferrerResponse : FEATURE_NOT_SUPPORTED", true);
                break;
            case 3:
                AbxLog.d("InstallReferrerResponse : DEVELOPER_ERROR", true);
                break;
            default:
                break;
        }
    }
}
