package io.adbrix.sdk.component;

import org.json.JSONArray;
import org.json.JSONException;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;

import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.DRState;
import io.adbrix.sdk.domain.model.EventModel;
import io.adbrix.sdk.domain.model.EventPackage;
import io.adbrix.sdk.utils.CoreUtils;

public class EventPackageContainer {
    private Queue<EventPackage> packageQueue;
    private HashSet<String> successPackageIdTable;
    private IEventSender eventSender;
    private DataRegistry dataRegistry;

    public EventPackageContainer(IEventSender eventSender, DataRegistry dataRegistry) {
        this.dataRegistry = dataRegistry;
        this.eventSender = eventSender;
        packageQueue = getPreviousPackageQueue();
        AbxLog.d("getPreviousPackageQueue : " + packageQueue.size(), true);
        successPackageIdTable = new HashSet<>();
    }

    public void mergeV1EventsAndV2Events(EventPackage unsentV1EventPackage) {
        try {
            if (unsentV1EventPackage == null)
                return;

            Queue<EventPackage> mergedQueue = new ConcurrentLinkedQueue<>();
            Queue<EventPackage> v2Queue = new ConcurrentLinkedQueue<>(packageQueue);

            mergedQueue.offer(unsentV1EventPackage);

            while (!v2Queue.isEmpty()) {
                mergedQueue.offer(v2Queue.poll());
            }

            packageQueue = formatEventPakageQueue(mergedQueue);

            AbxLog.d("Merging v1 events and v2 events is finished. EventQueue Size : " + packageQueue.size(), true);
        } catch (Exception e) {
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
        }
    }

    public void updateUnsentEventPackagesInDataRegistry() {

        JSONArray jsonArray = new JSONArray();

        for (EventPackage eventPackage : packageQueue) {
            try {
                jsonArray.put(eventPackage.toJson());
            } catch (JSONException e) {
                AbxLog.e(Arrays.toString(e.getStackTrace()), true);
            }
        }

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_UNSENT_EVENTPACKAGES,
                jsonArray.toString(),
                5,
                this.getClass().getName(),
                true
        ));
    }

    private Queue<EventPackage> getPreviousPackageQueue() {
        String unsentEventPackagesString = dataRegistry
                .safeGetString(DataRegistryKey.STRING_UNSENT_EVENTPACKAGES, null);

        if (unsentEventPackagesString == null) return new ConcurrentLinkedQueue<>();

        JSONArray jsonArray = null;
        try {
            jsonArray = new JSONArray(unsentEventPackagesString);
        } catch (JSONException e) {
            AbxLog.d(Arrays.toString(e.getStackTrace()), true);
        }

        if (jsonArray == null) return new ConcurrentLinkedQueue<>();

        Queue<EventPackage> tempQueue = new ConcurrentLinkedQueue<>();
        try {
            for (int i = 0; i < jsonArray.length(); i++) {
                EventPackage eventPackage = EventPackage.fromJson(jsonArray.getJSONObject(i));

                if (eventPackage == null) continue;

                tempQueue.offer(eventPackage);
            }
        } catch (Exception e) {
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
        }

        return formatEventPakageQueue(tempQueue);
    }

    private Queue<EventPackage> formatEventPakageQueue(Queue<EventPackage> unformattedQueue) {
        Queue<EventModel> flattedEventModelQueue = new ConcurrentLinkedQueue<>();

        for (EventPackage eventPackage : unformattedQueue) {
            flattedEventModelQueue.addAll(eventPackage.eventModels);
        }

        while (flattedEventModelQueue.size() > CoreConstants.MAX_EVENT_COUNT)
            flattedEventModelQueue.poll();

        Queue<EventPackage> formattedEventPackageQueue = partition(flattedEventModelQueue);

        return formattedEventPackageQueue;
    }

    private Queue<EventPackage> partition(Queue<EventModel> eventModels) {
        Queue<EventPackage> eventPackages = new ConcurrentLinkedQueue<>();
        Queue<EventModel> chunked = new ConcurrentLinkedQueue<>();

        for (EventModel eventModel : eventModels) {
            chunked.add(eventModel);

            if (chunked.size() == CoreConstants.EVENT_BUFFER_COUNT) {
                eventPackages.add(new EventPackage(chunked));
                chunked = new ConcurrentLinkedQueue<>();
            }
        }

        if (!chunked.isEmpty()) {
            eventPackages.add(new EventPackage(chunked));
        }

        return eventPackages;
    }

    public void addEventPackage(EventPackage sendWaitEventPackage) {
        if (!successPackageIdTable.contains(sendWaitEventPackage.packageId)) {
            this.packageQueue.offer(sendWaitEventPackage);
        }
    }

    public void flushAtIntervals() {

        Runnable eventSendingTask = new Runnable() {
            @Override
            public void run() {
                flushEventsNotSent();
            }
        };

        EventUploadIntervalManager.getInstance().manageEventUploadInterval(eventSendingTask, packageQueue);
    }

    public void flushSingleEventPackageImmediately(EventPackage eventPackage) {

        eventSender.sendEvent(eventPackage, new IEventSender.IEventSendActionListener() {
            @Override
            public void onSuccess(EventPackage successfulEventPackage) {
                successPackageIdTable.add(successfulEventPackage.packageId);
            }

            @Override
            public void onFail(EventPackage failedEventPackage) {
                EventPackageContainer.this.addEventPackage(failedEventPackage);
            }

            @Override
            public void onComplete() {

            }
        });
    }

    public void flushEventsNotSent() {

        if (CoreUtils.getDRState(dataRegistry) == DRState.INIT_RESTART_NOT_SYNCED) {
            AbxLog.d("Initializing restart not synced! Cannot send events.", true);
            return;
        }

        //fail된 eventPackage는 다음에 보내기 위한 장치
        int currentQueueSize = packageQueue.size();

        for (int i = 0; i < currentQueueSize; i++) {
            final EventPackage eventPackage = packageQueue.poll();

            flushSingleEventPackageImmediately(eventPackage);
        }
    }

    public void clearQueue() {
        packageQueue.clear();
    }
}
