package io.adbrix.sdk.domain.model;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.Arrays;
import java.util.Map;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.CoreUtils;

/**
 * 이벤트 데이터
 */
public final class EventModel implements IDataModel{

    public final String last_firstopen_id;
    public final String last_deeplink_id;
    public final String last_open_id;
    public final String prev_id;
    public final String event_datetime;
    public final String group;
    public final String event_name;
    public final Map<String,Object> param;
    public final long session_interval;
    public final long session_length;
    public final String log_id;
    public final String session_id;
    public final Location location;
    public final String user_snapshot_id;
    public final JSONObject user_properties;
    private final JSONObject ci;

    public EventModel(
            String last_firstopen_id,
            String last_deeplink_id,
            String last_open_id,
            String prev_id,
            String event_datetime,
            String group,
            String event_name,
            Map<String,Object> param,
            long session_interval,
            long session_length,
            String log_id,
            String session_id,
            Location location,
            String user_snapshot_id,
            JSONObject user_properties,
            JSONObject ci
    ){
        this.last_firstopen_id = last_firstopen_id;
        this.last_deeplink_id = last_deeplink_id;
        this.last_open_id = last_open_id;
        this.prev_id = prev_id;
        this.event_datetime = event_datetime;
        this.group = group;
        this.event_name = event_name;
        this.param = param;
        this.session_interval = session_interval;
        this.session_length = session_length;
        this.log_id = log_id;
        this.session_id = session_id;
        this.location = location;
        this.user_snapshot_id = user_snapshot_id;
        this.user_properties = user_properties;
        this.ci = ci;
    }

    public static class Location{
        public final String lwid;
        public final Double lat;
        public final Double lng;
        public Location(
                String lwid,
                Double lat,
                Double lng)
        {
            this.lwid = lwid;
            this.lat = lat;
            this.lng = lng;
        }
    }


    /**
     * "evt": {  //고객이 입력한 이벤트
     *     "event_datetime": "20151106094112",    //이벤트의 시간을 yyyyMMddHHmmss 형태의 문자열로 변환
     *     "event_name": "open_menu",              //고객이 입력한 이벤트명
     *     "location": {                      //optional (위치 추적을 할 수 없다면 null)
     *       "lat": 37.7673973149,            //위도
     *       "lng": -122.4293583478           //경도
     *     },
     *     "param":                           //고객이 입력한 파라미터 key:value 페어로 이루어져있다.
     *     {
     *         "menu_name" : "menu1",
     *         "menu_id" : 30
     *     },
     *     "user_properties": {               //고객이 입력한 프로퍼티 key:value 페어로 이루어져 있다.
     *       "birthyear": 1982,
     *       "gender": "m",
     *       "level": 37,
     *       "character_class": "knight",
     *       "gold": 300
     *     }
     *   }
     * @return
     */
    public JSONObject getJson() throws JSONException {
        JSONObject root = new JSONObject();
        {
            root.put("last_firstopen_id",   this.last_firstopen_id);
            root.put("last_deeplink_id",    CoreUtils.convertNullableToJSONNullable(this.last_deeplink_id));
            root.put("last_open_id",        this.last_open_id);
            root.put("prev_id",             CoreUtils.convertNullableToJSONNullable(this.prev_id));
            //"event_datetime": "20151106094112"
            root.put("event_datetime",      this.event_datetime);
            root.put("group",               this.group);
            //"event_name": "open_menu"
            root.put("event_name",          this.event_name);

            /*
            "param":                           //고객이 입력한 파라미터 key:value 페어로 이루어져있다.
             {
                "menu_name" : "menu1",
                "menu_id" : 30
             }
            */
            if(this.param != null)
            {
                JSONObject paramJson = new JSONObject();
                {
                    for(String key : this.param.keySet())
                    {
                        paramJson.put(key,this.param.get(key));
                    }
                }
                root.put("param",paramJson);
            }
            else
            {
                root.put("param", JSONObject.NULL);
            }

            root.put("session_interval",    this.session_interval);
            root.put("session_length",      this.session_length);
            root.put("log_id",              this.log_id);
            root.put("session_id",          CoreUtils.convertNullableToJSONNullable(this.session_id));

            /*
            "location": {                      //optional (위치 추적을 할 수 없다면 null)
                "lat": 37.7673973149,            //위도
                "lng": -122.4293583478           //경도
            }
            */
            if(this.location != null)
            {
                JSONObject locationJson = new JSONObject();

                if (this.location.lwid == null) {
                    root.put("location", JSONObject.NULL);
                }
                else {
                    locationJson.put("lwid", this.location.lwid);
                    locationJson.put("lat", this.location.lat);
                    locationJson.put("lng", this.location.lng);
                    
                    root.put("location", locationJson);
                }
            }
            else
            {
                root.put("location", JSONObject.NULL);
            }

            root.put("user_snapshot_id", CoreUtils.convertNullableToJSONNullable(this.user_snapshot_id));

            /*
            "user_properties": {               //고객이 입력한 프로퍼티 key:value 페어로 이루어져 있다.
                "birthyear": 1982,
                "gender": "m",
                "level": 37,
                "character_class": "knight",
                "gold": 300
            }
            */

            root.put("user_properties", user_properties);
            root.put("ci", ci);
        }

        return root;
    }

    public static EventModel fromJson(JSONObject jsonObject) throws JSONException {

        Location location = null;
        Map<String, Object> param = null;

        try {
            if (jsonObject.get("location") instanceof JSONObject) {
                location = new Location(
                        CommonUtils.convertNullStringToNull(
                                jsonObject.getJSONObject("location").getString("lwid")),
                        jsonObject.getJSONObject("location").getDouble("lat"),
                        jsonObject.getJSONObject("location").getDouble("lng")
                );
            }
        } catch (JSONException e){
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
        }

        try {
            if (jsonObject.get("param") instanceof JSONObject) {
                param = CommonUtils.getMapFromJSONObject(jsonObject.getJSONObject("param"));
            }
        } catch (JSONException e){
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
        }

        return new EventModel(
                CommonUtils.convertNullStringToNull(jsonObject.getString("last_firstopen_id")),
                CommonUtils.convertNullStringToNull(jsonObject.getString("last_deeplink_id")),
                CommonUtils.convertNullStringToNull(jsonObject.getString("last_open_id")),
                CommonUtils.convertNullStringToNull(jsonObject.getString("prev_id")),
                CommonUtils.convertNullStringToNull(jsonObject.getString("event_datetime")),
                CommonUtils.convertNullStringToNull(jsonObject.getString("group")),
                CommonUtils.convertNullStringToNull(jsonObject.getString("event_name")),
                param,
                jsonObject.getLong("session_interval"),
                jsonObject.getLong("session_length"),
                CommonUtils.convertNullStringToNull(jsonObject.getString("log_id")),
                CommonUtils.convertNullStringToNull(jsonObject.getString("session_id")),
                location,
                CommonUtils.convertNullStringToNull(jsonObject.getString("user_snapshot_id")),
                jsonObject.getJSONObject("user_properties"),
                jsonObject.getJSONObject("ci")
        );
    }
}
