package io.adbrix.sdk.data.repository;

import android.app.Activity;
import android.content.Context;
import android.content.res.Configuration;
import android.net.ConnectivityManager;
import android.net.Network;
import android.net.NetworkCapabilities;
import android.net.NetworkRequest;
import android.os.Build;

import org.json.JSONException;

import java.util.Arrays;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.Map;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.EventBuffer;
import io.adbrix.sdk.component.EventPackageContainer;
import io.adbrix.sdk.component.EventPackagingFacade;
import io.adbrix.sdk.component.IABXComponentsFactory;
import io.adbrix.sdk.component.ILogger;
import io.adbrix.sdk.component.SKDataSender;
import io.adbrix.sdk.component.TimerGroup;
import io.adbrix.sdk.configuration.DefaultABXContextController;
import io.adbrix.sdk.data.ABXBooleanState;
import io.adbrix.sdk.data.S3ConfigHandler;
import io.adbrix.sdk.data.dataprovider.DefaultNowaitEventNameProvider;
import io.adbrix.sdk.data.dataprovider.DeviceRealtimeDataProvider;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.inAppMessage.AbxDialog;
import io.adbrix.sdk.data.inAppMessage.IInAppMessageViewFactory;
import io.adbrix.sdk.data.inAppMessage.InAppMessage;
import io.adbrix.sdk.data.inAppMessage.InAppMessageDAO;
import io.adbrix.sdk.data.inAppMessage.InAppMessageFactoryContainer;
import io.adbrix.sdk.data.modelprovider.DRModelProvider;
import io.adbrix.sdk.data.modelprovider.EventModelProvider;
import io.adbrix.sdk.data.modelprovider.GDPRModelProvider;
import io.adbrix.sdk.data.modelprovider.InAppMessageModelProvider;
import io.adbrix.sdk.data.net.ApiConnection;
import io.adbrix.sdk.data.net.ApiConnectionManager;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.Repository;
import io.adbrix.sdk.domain.model.DRState;
import io.adbrix.sdk.domain.model.DRModel;
import io.adbrix.sdk.domain.model.EventModel;
import io.adbrix.sdk.domain.model.IApiModel;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.domain.model.UserPropertyCommand;
import io.adbrix.sdk.domain.model.UserPropertyModel;
import io.adbrix.sdk.utils.CoreUtils;

public class RepositoryImpl implements Repository {
    private IABXComponentsFactory factory;
    private UserPropertyManager userPropertyManager;
    private DataRegistry dataRegistry;
    private Context androidContext;
    private ILogger logger;
    private EventPackagingFacade eventPackagingFacade;
    private EventPackageContainer eventPackageContainer;
    private EventBuffer eventBuffer;
    private SKDataSender skDataSender;
    private ConnectivityManager connectivityManager;
    private ConnectivityManager.NetworkCallback networkCallback;
    private InAppMessageDAO inAppMessageDAO;
    private ExecutorService inAppMessageExecutorService;
    private InAppMessageFactoryContainer inAppMessageFactoryContainer;
    private Activity currentActivity;
    private AbxDialog abxDialog;
    private DeviceRealtimeDataProvider deviceRealtimeDataProvider;

    public RepositoryImpl(IABXComponentsFactory factory) {
        this.factory = factory;
        this.logger = factory.createOrGetLogger();

        try {
            this.userPropertyManager = factory.createOrGetUserPropertyManager();
            this.dataRegistry = factory.createOrGetDataRegistry();
            this.androidContext = factory.getAndroidContext();
            this.eventPackageContainer = new EventPackageContainer(factory.createOrGetEventSender(), dataRegistry);
            this.eventBuffer = new EventBuffer(CoreConstants.EVENT_BUFFER_COUNT, eventPackageContainer);
            this.eventPackagingFacade = new EventPackagingFacade(
                    new DefaultNowaitEventNameProvider(),
                    eventBuffer
            );
            this.inAppMessageDAO = new InAppMessageDAO(androidContext, dataRegistry);
            this.inAppMessageExecutorService = Executors.newSingleThreadExecutor();
            this.inAppMessageFactoryContainer = new InAppMessageFactoryContainer(inAppMessageDAO);
            this.deviceRealtimeDataProvider = factory.createOrGetDeviceRealtimeDataProvider();
        } catch (IABXComponentsFactory.ComponentsCanNotCreateException e) {
            logger.error("RepositoryImpl : 컴포넌트를 만들지 못했습니다.");
        } catch (EventBuffer.BufferCapacityLimitsException e){
            //capacity 를 5이상 설정했기 때문에 여기올 가능성 없으니 아무처리도 하지 않습니다.
        }
    }

    @Override
    public EventModel getEventModel(LogEventParameter logEventParameter) {
        EventModelProvider eventModelProvider = new EventModelProvider(
                this.userPropertyManager,
                this.dataRegistry);

        eventModelProvider.setLogEventParameter(logEventParameter);

        return (EventModel) eventModelProvider.provide();
    }

    @Override
    public Boolean saveUserPropertyWithoutEvent(UserPropertyCommand userPropertyCommand) {
        return this.userPropertyManager.merge(userPropertyCommand);
    }

    @Override
    public void initDefaultConfigValue() {
        S3ConfigHandler.getInstance().initDefaultConfigValue(dataRegistry);
    }

    @Override
    public void logEvent(LogEventParameter logEventParameter) {
        EventModel eventModel = getEventModel(logEventParameter);

        this.eventPackagingFacade.addEventModel(eventModel);

        // prev_id를 저장해준다.
        dataRegistry.putDataRegistry(
                new DataUnit(DataRegistryKey.STRING_PREV_ID,
                        logEventParameter.eventId,
                        5,
                        this.getClass().getName(),
                        true));

        if (!CoreUtils.isOnline(androidContext) && !ABXBooleanState.getInstance().inForeground()
                && logEventParameter.eventName.equals(CoreConstants.EVENT_END_SESSION))
            saveUnsentEvents();
    }

    @Override
    public void flushAllNow() {
        this.eventPackagingFacade.flushAllNow();
    }

    @Override
    public void flushAtIntervals() {
        this.eventPackagingFacade.flushAtIntervals();
    }

    @Override
    public void saveUnsentEvents() {
        eventBuffer.saveUnsentEvents();
    }

    @Override
    public void syncApplicationConfigure() {
        if (ABXBooleanState.getInstance().isGdprForgetMe()) return;
        if (ABXBooleanState.getInstance().isAdbrixAllStop()) {
            AbxLog.d("ADBRIX ALL STOP :: don't synchronize with server", false);
            return;
        }

        AbxLog.d("Trying to get application settings..", false);
        AbxLog.d("requestAppConfig API called!", true);
        try {
            ApiConnectionManager apiConnectionManager = new ApiConnectionManager(new ApiConnectionManager.Result() {
                @Override
                public void connectSuccess(String responseString, int responseCode) {
                    AbxLog.d("App-Config : connectSuccess ! : " + responseString, true);
                    S3ConfigHandler.getInstance().parseResponseOfS3Config(responseString, dataRegistry);
                    S3ConfigHandler.getInstance().initDefaultConfigValue(dataRegistry);
                }

                @Override
                public void connectFail(int responseCode) {
                    AbxLog.w("AppSyncConfig Failed : Can not get S3 config.json file....  responseCode : " + responseCode, true);
                }
            });

            String appKey = dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY,null);
            String urlString = String.format(CoreConstants.CONFIG_REQUEST_URL, appKey);

            ApiConnection apiConnection = new ApiConnection(dataRegistry, androidContext)
                    .GET(urlString);

            apiConnectionManager.ExecuteWithRetry(apiConnection);

        } catch (Exception e) {
            AbxLog.e("API Error: " + e.getMessage(), false);
        }
    }

    @Override
    public void gdprForgetMe() {
        try {
            ApiConnectionManager apiConnectionManager = new ApiConnectionManager(new ApiConnectionManager.Result() {
                @Override
                public void connectSuccess(String responseString, int responseCode) {
                    AbxLog.d("GDPR connect success!!", true);
                    dataRegistry.putDataRegistry(
                            new DataUnit(
                                    DataRegistryKey.LONG_GDPR_FORGETME,
                                    1L,
                                    5,
                                    this.getClass().getName(),
                                    true
                            )
                    );

                    dataRegistry.putDataRegistry(
                            new DataUnit(
                                    DataRegistryKey.LONG_GDPR_FORGETME_SERVER_SYNC,
                                    1L,
                                    5,
                                    this.getClass().getName(),
                                    true
                            )
                    );

                    ABXBooleanState.getInstance().isGdprForgetMeSync.getAndSet(true);
                }

                @Override
                public void connectFail(int responseCode) {
                    AbxLog.d("GDPR connect failed!! responseCode : " + responseCode, true);
                    dataRegistry.putDataRegistry(
                            new DataUnit(
                                    DataRegistryKey.LONG_GDPR_FORGETME,
                                    1L,
                                    5,
                                    this.getClass().getName(),
                                    true
                            )
                    );

                    dataRegistry.putDataRegistry(
                            new DataUnit(
                                    DataRegistryKey.LONG_GDPR_FORGETME_SERVER_SYNC,
                                    0L,
                                    5,
                                    this.getClass().getName(),
                                    true
                            )
                    );
                }
            });

            IApiModel gdprModel = new GDPRModelProvider(dataRegistry).provide();

            ApiConnection apiConnection = new ApiConnection(dataRegistry, androidContext)
                    .POST(gdprModel);

            apiConnectionManager.ExecuteWithRetry(apiConnection);
        } catch (Exception e) {
            AbxLog.e("API Error: " + e.getMessage(), false);
            AbxLog.e("GDPR Request Error: " + e.getMessage(), true);
        }

        ABXBooleanState.getInstance().isGdprForgetMe.getAndSet(true);
        dataRegistry.saveRegistry();
    }

//    @Override
//    public void updateFavoriteApplication() {
//        Timer timer = new Timer();
//
//        timer.schedule(new TimerTask() {
//            @Override
//            public void run() {
//                try {
//                    if (ABXBooleanState.getInstance().isAdbrixAllStop() ||
//                            ABXBooleanState.getInstance().isAdbrixError()){
//                        return;
//                    }
//
//                    AbxLog.i("Application Scanning... per " + ABXConstants.BATCHING_INTERVAL / 1000 + " Seconds", false);
//
//                    AbxAppScanHandler appScanHandler = new AbxAppScanHandler(dataRegistry);
//                    if (!ABXBooleanState.getInstance().isAdbrixPause()) {
//                        if (appScanHandler.getAppScanStatus()) {
//                            if (appScanHandler.checkIsTimeToAppScan())
//                                appScanHandler.abxApplicationScan(androidContext);
//                        } else {
//                            boolean userFlag = dataRegistry.safeGetLong(DataRegistryKey.LONG_APP_SCAN_ON_OFF_USER, -1) == 1;
//                            AbxLog.d("Application Scanning Failed : Server Flag = " + !S3ConfigHandler.config_appScanStop, true);
//                            AbxLog.d("Application Scanning Failed : User flag = " + userFlag, true);
//                        }
//                    }
//                } catch (Exception e){
//                    e.printStackTrace();
//                    Log.e(ABXConstants.LOGTAG, e.getMessage());
//                    timer.cancel();
//                }
//            }
//        }, 60_000, 60_000);
//    }

    @Override
    public void registerNetworkCallback() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP){
            skDataSender = new SKDataSender(dataRegistry);

            connectivityManager = (ConnectivityManager) androidContext.getSystemService(Context.CONNECTIVITY_SERVICE);
            networkCallback = null;

            NetworkRequest.Builder builder = new NetworkRequest.Builder();
            builder.addTransportType(NetworkCapabilities.TRANSPORT_WIFI);

            NetworkRequest networkRequest = builder.build();

            networkCallback = new ConnectivityManager.NetworkCallback(){
                @Override
                public void onAvailable(Network network) {
                    super.onAvailable(network);
                    skDataSender.wifiConnectAction();
                }

                @Override
                public void onLost(Network network) {
                    super.onLost(network);
                    skDataSender.wifiDisconnectAction();
                }
            };

            connectivityManager.registerNetworkCallback(networkRequest, networkCallback);
        }
    }

    @Override
    public void resetInAppMessageFrequencySession() {
        if (inAppMessageExecutorService.isShutdown())
            inAppMessageExecutorService = Executors.newSingleThreadExecutor();

        inAppMessageExecutorService.submit(() -> inAppMessageDAO.resetFrequencyCapPerSession());
    }

    @Override
    public void triggerInAppMessage(String eventName) {
        if(dataRegistry.safeGetLong(DataRegistryKey.LONG_S3_CONFIG_IN_APP_MESSAGE_ACTIVE, -1) != 1){
            return;
        }

        if (inAppMessageExecutorService.isShutdown())
            inAppMessageExecutorService = Executors.newSingleThreadExecutor();

        if(currentActivity == null || currentActivity.isFinishing()) {
            AbxLog.d("InAppMessage:: Can't use current activity!", true);
            return;
        }

        inAppMessageExecutorService.submit(() -> {
            if(currentActivity == null || currentActivity.isFinishing()) {
                AbxLog.d("InAppMessage:: Can't use current activity!", true);
                return;
            }else if(ABXBooleanState.getInstance().isInAppMessagePresented()){
                AbxLog.d("InAppMessage already shown", true);
                return;
            }

            InAppMessage inAppMessage;

            try {
                boolean isPortrait = currentActivity.getResources().getConfiguration().orientation == Configuration.ORIENTATION_PORTRAIT;
                inAppMessage = inAppMessageDAO.getInAppMessageByEventName(eventName, isPortrait);
                if (inAppMessage != null) {
                    IInAppMessageViewFactory inAppMessageViewFactory = inAppMessageFactoryContainer.getInAppMessageViewFactory(inAppMessage);
                    inAppMessageViewFactory.createInAppMessage(currentActivity, inAppMessage, dialog -> abxDialog = dialog);
                }
            }catch (NullPointerException nullPointerException){
                AbxLog.e("triggerInAppMessage: nullpointerException" + Arrays.toString(nullPointerException.getStackTrace()),true);
            }
        });
    }

    @Override
    public void dismissInAppMessageDialog(){
        if(abxDialog != null){
            if(abxDialog.isShowing())
                abxDialog.dismiss();
            abxDialog = null;
        }
    }
  
    @Override
    public void shutDownInAppMessageExecutor() {
        if(inAppMessageExecutorService != null)
            inAppMessageExecutorService.shutdown();
        inAppMessageDAO.closeDataBase();
    }

    @Override
    public void updateCurrentActivity(Activity activity) {
        if(inAppMessageExecutorService == null)
            inAppMessageExecutorService = Executors.newSingleThreadExecutor();

        inAppMessageExecutorService.submit(() -> {
            currentActivity = activity;
        });
    }

    @Override
    public void deleteCurrentActivity(Activity activity) {
        if(currentActivity == activity){
            currentActivity = null;
        }
    }

    @Override
    public void connectToInAppMessageApi(){
        if(dataRegistry.safeGetLong(DataRegistryKey.LONG_S3_CONFIG_IN_APP_MESSAGE_ACTIVE, -1) != 1){
            return;
        }

        AbxLog.d("inAppMessageApiConnection start ::: ", true);
        try {
            ApiConnectionManager apiConnectionManager = new ApiConnectionManager(new ApiConnectionManager.Result() {
                @Override
                public void connectSuccess(String responseString, int responseCode) {
                    AbxLog.d("Terminated inAppMessageApiConnection",true);
                    AbxLog.d("inAppMessageApiConnection connectSuccess !! response: " + responseString, true);

                    inAppMessageExecutorService.submit(() -> {
                        try {
                            inAppMessageDAO.insertApiResponse2Table(responseString);
                        } catch (JSONException e) {
                            AbxLog.d("inserApiResponse2Table jsonerror : " + e, true);
                        }
                    });
                }

                @Override
                public void connectFail(int responseCode) {
                    AbxLog.d("Terminated inAppMessageApiConnection", true);
                    AbxLog.d("inAppMessageApiConnection connectFail !! ", true);
                }
            });

            IApiModel inAppMessageModel = new InAppMessageModelProvider(dataRegistry, androidContext, deviceRealtimeDataProvider).provide();


            ApiConnection apiConnection = new ApiConnection(dataRegistry, androidContext)
                    .POST(inAppMessageModel);

            if(inAppMessageExecutorService == null)
                inAppMessageExecutorService = Executors.newSingleThreadExecutor();

            inAppMessageExecutorService.submit(() ->
                    apiConnectionManager.ExecuteWithRetry(apiConnection)
            );

        } catch (JSONException e) {
            AbxLog.e("inAppMessageApiConnection Request Error: " + e.getMessage(), true);
        }
    }
  
    @Override
    public void deleteUserData(String userId, ApiConnectionManager.Result result) {

        //cancel all timer.
        TimerGroup.getInstance().cancelAll();

        //clear user properties
        UserPropertyModel userPropertyModel = userPropertyManager.getCurrentUserPropertyModel();
        UserPropertyCommand command = new UserPropertyCommand();

        for (Map.Entry<String, Object> entry : userPropertyModel.properties.entrySet()){
            command.unset(entry.getKey());
        }

        saveUserPropertyWithoutEvent(command);

        eventPackageContainer.clearQueue();

        inAppMessageDAO.deleteDB();
        if(inAppMessageExecutorService != null)
            inAppMessageExecutorService.shutdownNow();
        currentActivity = null;

        try {
            ApiConnectionManager apiConnectionManager = new ApiConnectionManager(result);

            IApiModel deleteUserDataModel = new DRModelProvider(factory,
                    userId, DRModel.OperationType.DELETE)
                    .provide();

            ApiConnection apiConnection = new ApiConnection(dataRegistry, androidContext)
                    .POST(deleteUserDataModel);

            apiConnectionManager.ExecuteWithRetry(apiConnection);
        } catch (Exception e) {
            AbxLog.e("Restart API Request Error: " + e.getMessage(), true);
        }
    }

    @Override
    public void restartSDK(String userId, ApiConnectionManager.Result result) {

        ApiConnectionManager apiConnectionManager = new ApiConnectionManager(result);
        IApiModel restartDataModel = null;

        inAppMessageDAO.restartInAppMessage();

        if (DefaultABXContextController.getInstance().getDRState() != DRState.INIT_RESTART_SYNCED) {
            DRModelProvider drModelProvider = new DRModelProvider(factory, userId, DRModel.OperationType.INITIALIZE);

            //restart - initialize
            if (DefaultABXContextController.getInstance().getDRState() == DRState.INIT_RESTART_NOT_SYNCED) {
                restartDataModel = drModelProvider.getNotSyncedInitRestartModel();
            }
            else {
                restartDataModel = drModelProvider.provide();
            }
        }
        else {
            //duplicated restart
            restartDataModel = new DRModelProvider(factory,
                    userId, DRModel.OperationType.RESTART)
                    .provide();
        }

        try {
            ApiConnection apiConnection = new ApiConnection(dataRegistry, androidContext)
                    .POST(restartDataModel);

            apiConnectionManager.Execute(apiConnection);
        } catch (Exception e) {
            AbxLog.e("Restart API Request Error: " + e.getMessage(), true);
        }
    }
}
