package io.adbrix.sdk.data.dataprovider;

import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteException;
import android.database.sqlite.SQLiteOpenHelper;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.Arrays;
import java.util.Iterator;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.data.S3ConfigHandler;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.data.repository.UserPropertyManager;
import io.adbrix.sdk.domain.model.UserPropertyCommand;
import io.adbrix.sdk.domain.model.UserPropertyModel;
import io.adbrix.sdk.utils.CommonUtils;

public class V1DataProvider extends AbstractDataProvider {
    public V1DatabaseOpenHelper v1DatabaseOpenHelper;
    private UserPropertyManager userPropertyManager;

    private static final String UUID = "uuid";
    private static final String LAST_FIRSTOPEN_ID = "last_firstopen_id";
    private static final String LAST_DEEPLINK_ID = "last_deeplink_id";
    private static final String LAST_OPEN_ID = "last_open_id";
    private static final String DAILY_FIRST_OPEN_PREV_DATE = "daily_first_open_prev_date";
    private static final String REGISTRATION_ID = "registration_id";
    private static final String PUSH_ENABLE = "push_enable";
    private static final String PUSH_ENABLE_OS = "push_enable_os";
    private static final String USER_PROFILE_SNAPSHOT_ID = "user_snapshot_id";
    private static final String USER_PROFILE = "abx:userProfile";
    private static final String USER_ID = "user_id";

    public V1DataProvider(Context context, DataRegistry dataRegistry, UserPropertyManager userPropertyManager) {
        super(dataRegistry);
        this.v1DatabaseOpenHelper = V1DatabaseOpenHelper.createInstance(context);
        this.userPropertyManager = userPropertyManager;
    }

    private static class V1DatabaseOpenHelper extends SQLiteOpenHelper {
        private static final String DATABASE_NAME = "abrix.v2.sql";
        protected static final String STRING_DATA_STORE_TABLE_NAME = "string_store";
        private static final String KEY_FIELD = "key";
        private static final String VALUE_FIELD = "value";
        private final SQLiteDatabase db;

        public static V1DatabaseOpenHelper createInstance(Context context) {
            return new V1DatabaseOpenHelper(context, DATABASE_NAME, null, 1);
        }

        private V1DatabaseOpenHelper(Context context, String name, SQLiteDatabase.CursorFactory factory, int version) {
            super(context, name, factory, version);
            this.db = getReadableDatabase();
        }

        @Override
        public void onCreate(SQLiteDatabase db) {

        }

        @Override
        public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {

        }

        public String getValue(String key) {
            return (String) getValueFromTable(STRING_DATA_STORE_TABLE_NAME, key);
        }

        private Object getValueFromTable(String table, String key) {
            Object value = null;
            try (Cursor cursor = queryDb(db, table, new String[]{KEY_FIELD, VALUE_FIELD}, KEY_FIELD + " = ?",
                    new String[]{key}, null, null, null, null)) {
                if (cursor.moveToFirst()) {
                    value = cursor.getString(1);
                }
            } catch (SQLiteException e) {
                AbxLog.w(String.format("getValue from %s failed: %s", table, e.getMessage()),true);
            } catch (StackOverflowError e) {
                AbxLog.w(String.format("getValue from %s failed: %s", table, e.getMessage()),true);
            } catch (RuntimeException e) {
                AbxLog.w(String.format("getValue from %s failed: %s", table, e.getMessage()),true);
            }
            return value;
        }

        private Cursor queryDb(
                SQLiteDatabase db, String table, String[] columns, String selection,
                String[] selectionArgs, String groupBy, String having, String orderBy, String limit
        ) {
            return db.query(table, columns, selection, selectionArgs, groupBy, having, orderBy, limit);
        }

        private boolean isTableDetected() {
            String query = "select DISTINCT tbl_name from sqlite_master where tbl_name = '"+STRING_DATA_STORE_TABLE_NAME+"'";
            try (Cursor cursor = db.rawQuery(query, null)) {
                if (cursor != null) {
                    return cursor.getCount() > 0;
                }
                return false;
            }
        }
    }

    private boolean isMigrated() {
        boolean isMigrated = safeGetBoolean(DataRegistryKey.BOOLEAN_IS_MIGRATED, false);

        return isMigrated;
    }
    
    public boolean isTableDetected() {
        return v1DatabaseOpenHelper.isTableDetected();
    }

    @Override
    public void provideDefaultValues() {
        if (isMigrated()) {
            return;
        }

        if (!isTableDetected()) {
            return;
        }

        AbxLog.d("Migration V1 to V2 is available. Start migrating!", true);

        String uuid = v1DatabaseOpenHelper.getValue(UUID);
        String lastFirstOpenId = v1DatabaseOpenHelper.getValue(LAST_FIRSTOPEN_ID);
        String lastDeeplinkId = v1DatabaseOpenHelper.getValue(LAST_DEEPLINK_ID);
        String lastOpenId = v1DatabaseOpenHelper.getValue(LAST_OPEN_ID);
        String dailyFirstOpenPrevDate = v1DatabaseOpenHelper.getValue(DAILY_FIRST_OPEN_PREV_DATE);
        String registrationId = v1DatabaseOpenHelper.getValue(REGISTRATION_ID);
        boolean isPushEnable = Boolean.parseBoolean(v1DatabaseOpenHelper.getValue(PUSH_ENABLE));
        boolean isPushEnableOs = Boolean.parseBoolean(v1DatabaseOpenHelper.getValue(PUSH_ENABLE_OS));
        String userProperties = v1DatabaseOpenHelper.getValue(USER_PROFILE);

        String prevUUID = safeGetString(DataRegistryKey.STRING_UUID, null);
        String prevLastFirstOpenId = safeGetString(DataRegistryKey.STRING_LAST_FIRSTOPEN_ID, null);
        String prevLastDeeplinkId = safeGetString(DataRegistryKey.STRING_LAST_DEEPLINK_ID, null);
        String prevLastOpenId = safeGetString(DataRegistryKey.STRING_LAST_OPEN_ID, null);
        String prevDailyFirstOpenPrevDate = safeGetString(DataRegistryKey.STRING_DAILY_FIRST_OPEN_PREV_DATE, null);
        String prevRegistrationId = safeGetString(DataRegistryKey.STRING_REGISTRATION_ID, null);
        Boolean prevIsPushEnable = getBoolean(DataRegistryKey.BOOLEAN_IS_PUSH_ENABLE);
        Boolean prevIsPushEnableOs = getBoolean(DataRegistryKey.BOOLEAN_IS_PUSH_ENABLE_OS);

        if (prevUUID == null) {
            putString(DataRegistryKey.STRING_UUID, uuid, 5, true);
            AbxLog.d("Migrated uuid = " + uuid, true);
        }

        if (prevLastFirstOpenId == null) {
            putString(DataRegistryKey.STRING_LAST_FIRSTOPEN_ID, lastFirstOpenId, 5, true);
            AbxLog.d("Migrated last_firstopen_id = " + lastFirstOpenId, true);
        }

        if (prevLastDeeplinkId == null) {
            putString(DataRegistryKey.STRING_LAST_DEEPLINK_ID, lastDeeplinkId, 5, true);
            AbxLog.d("Migrated last_deeplink_id = " + lastDeeplinkId, true);
        }

        if (prevLastOpenId == null) {
            putString(DataRegistryKey.STRING_LAST_OPEN_ID, lastOpenId, 5, true);
            AbxLog.d("Migrated last_open_id = " + lastOpenId, true);
        }

        if (prevDailyFirstOpenPrevDate == null) {
            putString(DataRegistryKey.STRING_DAILY_FIRST_OPEN_PREV_DATE, dailyFirstOpenPrevDate, 5, true);
            AbxLog.d("Migrated daily_first_open_prev_date = " + dailyFirstOpenPrevDate, true);
        }

        if (prevRegistrationId == null) {
            putString(DataRegistryKey.STRING_REGISTRATION_ID, registrationId, 5, true);
            AbxLog.d("Migrated registration_id = " + registrationId, true);
        }

        if (prevIsPushEnable == null) {
            putBoolean(DataRegistryKey.BOOLEAN_IS_PUSH_ENABLE, isPushEnable, 5, true);
            AbxLog.d("Migrated is_push_enable = " + isPushEnable, true);
        }

        if (prevIsPushEnableOs == null) {
            putBoolean(DataRegistryKey.BOOLEAN_IS_PUSH_ENABLE_OS, isPushEnableOs, 5,true);
            AbxLog.d("Migrated is_push_enable = " + isPushEnableOs, true);
        }

        if (userProperties != null && !userProperties.equals("")) {
            saveUserProperties(userProperties);

            UserPropertyModel finalUserPropertyModel = userPropertyManager.getCurrentUserPropertyModel();

            try {
                AbxLog.d("UserProperties after migration = " +
                        finalUserPropertyModel.getJson().toString(4), true);
            } catch (JSONException e) {
                AbxLog.e(Arrays.toString(e.getStackTrace()), true);
            }
        }

        putBoolean(DataRegistryKey.BOOLEAN_IS_MIGRATED, true, 5, true);
    }

    public void saveUserProperties(String userProperties) {
        try {
            UserPropertyModel userPropertyModel = userPropertyManager.getCurrentUserPropertyModel();
            JSONObject userPropertiesJSON = new JSONObject(userProperties);

            if (userPropertyModel.properties.isEmpty()) {
                updateEveryUserProperties(userPropertiesJSON);

                String userSnapshotId = v1DatabaseOpenHelper.getValue(USER_PROFILE_SNAPSHOT_ID);
                putString(DataRegistryKey.STRING_USER_SNAPSHOT_ID, userSnapshotId, 5,true);
                return;
            }

            String currentUserId = null;
            String prevUserId = null;

            if (userPropertyModel.properties.containsKey(USER_ID))
                currentUserId = (String) userPropertyModel.properties.get(USER_ID);

            currentUserId = currentUserId.replace("string:", "");

            if (userPropertiesJSON.has(USER_ID))
                prevUserId = userPropertiesJSON.getString(USER_ID);

            if (currentUserId != null && currentUserId.equals(prevUserId))
                updateCurrentUserNotContainedData(userPropertiesJSON);

        } catch (JSONException e) {
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
        }
    }

    private void updateCurrentUserNotContainedData(JSONObject userPropertiesJSON) {
        try {
            UserPropertyCommand command = new UserPropertyCommand();
            UserPropertyModel userPropertyModel = userPropertyManager.getCurrentUserPropertyModel();

            JSONObject truncatedUserPropertiesJson = CommonUtils.truncate(userPropertiesJSON);
            JSONObject parsedUserPropertiesJson = CommonUtils.parseValueWithDataType(truncatedUserPropertiesJson, CommonUtils.FIX_TYPE.PREFIX);

            Iterator<?> keys = parsedUserPropertiesJson.keys();
            int currentSizeOfUserPropertyModel = userPropertyModel.properties.size();

            while(keys.hasNext()){
                String key = (String) keys.next();

                if (currentSizeOfUserPropertyModel < S3ConfigHandler.config_propertyMaxSize) {

                    if (!userPropertyModel.properties.containsKey(key)) {
                        command.set(key, parsedUserPropertiesJson.get(key));
                        currentSizeOfUserPropertyModel++;
                    }

                } else {
                    AbxLog.d("UserProperties reaches MAX_PROPERTY_KEYS: " + S3ConfigHandler.config_propertyMaxSize, true);
                    break;
                }
            }

            userPropertyManager.merge(command);
        } catch (JSONException e) {
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
        }
    }

    private void updateEveryUserProperties(JSONObject userPropertiesJSON) {
        try {
            UserPropertyCommand command = new UserPropertyCommand();
            UserPropertyModel userPropertyModel = userPropertyManager.getCurrentUserPropertyModel();

            JSONObject truncatedUserPropertiesJson = CommonUtils.truncate(userPropertiesJSON);
            JSONObject parsedUserPropertiesJson = CommonUtils.parseValueWithDataType(truncatedUserPropertiesJson, CommonUtils.FIX_TYPE.PREFIX);

            Iterator<?> keys = parsedUserPropertiesJson.keys();
            int currentSizeOfUserPropertyModel = userPropertyModel.properties.size();

            while(keys.hasNext()){
                String key = (String) keys.next();

                if (currentSizeOfUserPropertyModel < S3ConfigHandler.config_propertyMaxSize) {
                    command.set(key, parsedUserPropertiesJson.get(key));
                    currentSizeOfUserPropertyModel++;
                } else {
                    AbxLog.d("UserProperties reaches MAX_PROPERTY_KEYS: " + S3ConfigHandler.config_propertyMaxSize, true);
                    break;
                }
            }

            userPropertyManager.merge(command);
        } catch (JSONException e) {
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
        }
    }

    @Override
    public void asyncProvide(IDataProviderResultListener dataProviderResultListener) {

    }

    @Override
    public void refreshData() {

    }
}
