package io.adbrix.sdk.configuration;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;

import java.util.ArrayList;
import java.util.List;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.IABXComponentsFactory;
import io.adbrix.sdk.data.SdkVersion;
import io.adbrix.sdk.data.dataprovider.AbstractDataProvider;
import io.adbrix.sdk.data.dataprovider.IDataProviderResultListener;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.net.ApiConnectionManager;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.Repository;
import io.adbrix.sdk.domain.exception.BeforeInitializeException;
import io.adbrix.sdk.domain.interactor.InitDefaultConfigValueUseCase;
import io.adbrix.sdk.domain.interactor.SyncApplicationConfigureUseCase;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.domain.model.UserPropertyCommand;
import io.adbrix.sdk.utils.CommonUtils;

public class InitializingABXContext implements IABXContext {
    private IABXContextController main_controller;
    private IABXComponentsFactory componentsFactory;
    private DataRegistry dataRegistry;
    private Repository repository;

    public InitializingABXContext(IABXContextController main_controller, Repository repository) {
        this.main_controller = main_controller;
        this.componentsFactory = main_controller.getComponentsFactory();
        this.repository = repository;
    }

    @Override
    public void initialize(Context context, String appkey, String secretKey) {
        AbxLog.d( "Trying to initialize Adbrix Remastered SDK", true);
        AbxLog.d("SDK Version :: " + SdkVersion.SDK_VERSION, true);

        if(componentsFactory == null){
            this.abort(new Exception("InitializngABXContext.initialize is called before startcontroller"));
        }

        //1. 컴포넌트 팩토리 android Context 설정
        componentsFactory.setAndroidContext(context);

        //2. 앱키 시크릿키 설정
        this.setupAppkeyAndSecretKey(appkey, secretKey);

        //3.앱 설정 초기화
        new InitDefaultConfigValueUseCase(repository).execute();
        new SyncApplicationConfigureUseCase(repository).execute();

        //4.데이터 레지스트리 저장
        main_controller.setDataRegistry(dataRegistry);

        if(DefaultABXContextController.getInstance().isGdprForgetMe()){
            this.abort(new Exception("GDPR FORGET ME 상태입니다."));
        }
        else {
            //5. 프로바이더 분류 후 초기값 설정 및 실행
            this.initializeProviders();

            try {
                //final. Active 상태로 전환합니다.
                main_controller.changeABXContext(
                        new ActiveABXContext(
                                componentsFactory,
                                repository
                        )
                );
                AbxLog.d("Result of initializing Adbrix SDK : Success", true);
            } catch (IABXComponentsFactory.ComponentsCanNotCreateException e)
            {
                AbxLog.e("컴포넌트를 만들지 못했습니다.", true);
                InitializingABXContext.this.abort(e);
            }
        }
    }

    private void setupAppkeyAndSecretKey(String appkey, String secretKey) {
        // AppKey, SecretKey가 22자 이상인지

        if(CommonUtils.isNullOrEmpty(appkey) || appkey.length() != 22)
        {
            AbxLog.e("AppKey가 길이 22의 문자열이 아닙니다.", true);
            this.abort(new Exception("Invalid AppKey"));
            return;
        }

        if(CommonUtils.isNullOrEmpty(secretKey) || secretKey.length() != 22)
        {
            AbxLog.e("SecretKey가 길이 22의 문자열이 아닙니다.", true);
            this.abort(new Exception("Invalid SecretKey"));
            return;
        }

        AbxLog.d("Both AppKey and SecretKey are valid", true);
        //1. 데이터 레지스트리 가져오기
        dataRegistry = null;
        try {
            dataRegistry = componentsFactory.createOrGetDataRegistry();
        } catch (IABXComponentsFactory.ComponentsCanNotCreateException e) { //이런일이 발생해서는 안된다.
            AbxLog.e(e.toString(), true);
        }

        //2. 앱키 설정
        dataRegistry.putDataRegistry(
            new DataUnit(
                DataRegistryKey.STRING_APPKEY,
                appkey,
                5,
                "Initializing",
                false
            )
        );

        //3. 시크릿키 설정
        dataRegistry.putDataRegistry(
            new DataUnit(
                DataRegistryKey.STRING_SECRETKEY,
                secretKey,
                5,
                "Initializing",
                false
            )
        );

        dataRegistry.saveRegistry();
    }

    /**
     * 초기화중에도 이벤트는 받을 수 있어야 합니다.
     */
    @Override
    public void logEvent(LogEventParameter logEventParameter) {
    }

    /**
     * 초기화 중에도 사용자 속성은 설정할 수 있어야 합니다.
     */
    @Override
    public void saveUserProperty(UserPropertyCommand userPropertyCommand) {
    }

    @Override
    public Boolean saveUserPropertyWithoutEvent(UserPropertyCommand userPropertyCommand) {
        return null;
    }

    /**
     * 작업을 취소하고 SDK 상태를 사용불가로 전환
     */
    private void abort(Exception e){
        this.main_controller.changeABXContext(
                new DisabledABXContext(
                        e.toString(),
                        repository
                )
        );
    }

    private void initializeProviders() {
        try {
            List<AbstractDataProvider> requiredProviderList = new ArrayList<AbstractDataProvider>();

            //필수적 프로바이더
            requiredProviderList.add(componentsFactory.createOrGetV1DataProvider());
            requiredProviderList.add(componentsFactory.createOrGetDeviceStaticDataProvider());
            requiredProviderList.add(componentsFactory.createOrGetDeviceRealtimeDataProvider());

            //4. 프로바이더 최초값 설정
            for (AbstractDataProvider provider : requiredProviderList)
                provider.provideDefaultValues();

            //5. 필수 프로바이더 동기적 실행
            for (final AbstractDataProvider provider : requiredProviderList) {
                provider.asyncProvide(new IDataProviderResultListener() {
                    @Override
                    public void onSuccess() {
                        AbxLog.d("프로바이더 초기화에 성공했습니다.-" + provider.getClass().getName(), true);
                    }

                    @Override
                    public void onFail(Exception e) {
                        AbxLog.e("프로바이더 초기화에 실패했습니다.-" + provider.getClass().getName(), true);
                        InitializingABXContext.this.abort(e);
                    }
                });
            }
        } catch (IABXComponentsFactory.ComponentsCanNotCreateException e)
        {
            AbxLog.e("컴포넌트를 만들지 못했습니다.", true);
            InitializingABXContext.this.abort(e);
        }
    }

    @Override
    public void runOnIdleTime() {
        //무시
    }

    @Override
    public void onResume(Activity activity) {
        throw new BeforeInitializeException();
    }

    @Override
    public void onPause() {
        throw new BeforeInitializeException();
    }

    @Override
    public void deeplink(Activity deeplinkActivity) {
        throw new BeforeInitializeException();
    }

    @Override
    public void deeplinkWithIntent(Intent deeplinkIntent) {
        throw new BeforeInitializeException();
    }

    @Override
    public void putDataRegistry(DataUnit dataUnit) {
        throw new BeforeInitializeException();
    }

    @Override
    public void registerNetworkCallback() {
        throw new BeforeInitializeException();
    }

    @Override
    public void deleteUserData(String userId, ApiConnectionManager.Result result) {
        throw new BeforeInitializeException();
    }

    @Override
    public void restartSDK(String userId, ApiConnectionManager.Result result) {
        throw new BeforeInitializeException();
    }
}