package io.adbrix.sdk.configuration;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.DefaultComponentsFactory;
import io.adbrix.sdk.component.IABXComponentsFactory;
import io.adbrix.sdk.component.ILogger;
import io.adbrix.sdk.component.IObserver;
import io.adbrix.sdk.data.ABXBooleanState;
import io.adbrix.sdk.data.dataprovider.V1DataProvider;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.net.ApiConnectionManager;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.data.repository.RepositoryImpl;
import io.adbrix.sdk.data.repository.UserPropertyManager;
import io.adbrix.sdk.domain.LogMessageFormat;
import io.adbrix.sdk.domain.Repository;
import io.adbrix.sdk.domain.exception.BeforeInitializeException;
import io.adbrix.sdk.domain.interactor.GdprForgetMeUseCase;
import io.adbrix.sdk.domain.model.DRState;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.domain.model.UserPropertyCommand;
import io.adbrix.sdk.domain.model.UserPropertyModel;
import io.adbrix.sdk.utils.CommonUtils;

public class DefaultABXContextController implements IABXContextController, IObserver<LogEventParameter> {
    private IABXComponentsFactory componentsFactory;
    private ILogger logger;
    private IABXContext abxContext;
    private DataRegistry dataRegistry;
    private ExecutorService executorService;
    private boolean isInitSubmitted = false;
    private boolean isOnResumeSubmitted = false;
    private boolean isFirstOpenSubmitted = false;
    private boolean isSDKFirstOpened = false;
    private Queue<MessageInvokeRunnable> initQueue;
    private Queue<MessageInvokeRunnable> onResumeQueue;
    private Queue<MessageInvokeRunnable> firstOpenQueue;
    private Queue<MessageInvokeRunnable> etcQueue;
    private Repository repository;
    private boolean isInitRestartProcessingNow = false;
    private boolean isDeleteProcessingNow = false;

    private static class Singleton{
        private static final DefaultABXContextController instance = new DefaultABXContextController();
    }

    private DefaultABXContextController(){}

    public static DefaultABXContextController getInstance(){
        return Singleton.instance;
    }

    public void startController(Context appContext){
        this.componentsFactory = new DefaultComponentsFactory();
        this.componentsFactory.setAndroidContext(appContext);
        this.repository = new RepositoryImpl(componentsFactory);
        this.abxContext = new InitializingABXContext(this, repository);
        this.componentsFactory.setABXContext(abxContext);
        this.logger = this.componentsFactory.createOrGetLogger();
        executorService = Executors.newSingleThreadExecutor();
        initQueue = new ConcurrentLinkedQueue<>();
        onResumeQueue = new ConcurrentLinkedQueue<>();
        firstOpenQueue = new ConcurrentLinkedQueue<>();
        etcQueue = new ConcurrentLinkedQueue<>();
        V1DataProvider v1DataProvider = null;
        try {
            dataRegistry = componentsFactory.createOrGetDataRegistry();
            v1DataProvider = componentsFactory.createOrGetV1DataProvider();
        } catch (IABXComponentsFactory.ComponentsCanNotCreateException e) { //이런일이 발생해서는 안된다.
            this.logger.error(e.toString());
        }

        isSDKFirstOpened = !v1DataProvider.isTableDetected() && dataRegistry.safeGetString(DataRegistryKey.STRING_LAST_FIRSTOPEN_ID, null) == null;
    }

    public void stopController(){
        executorService.shutdown();
        repository.shutDownInAppMessageExecutor();
    }

    private static class Message{
        private MessageType messageType;
        private Object[] args;
        public Message(MessageType messageType,Object...args)
        {
            this.messageType = messageType;
            this.args = args;
        }
    }

    private enum MessageType
    {
        CHANGE_ABX_CONTEXT,
        INITIALIZE,
        SAVE_USER_PROPERTY,
        SAVE_USER_PROPERTY_WITHOUT_EVENT,
        LOG_EVENT,
        ON_RESUME,
        ON_PAUSE,
        DEEPLINK,
        DEEPLINK_WITH_INTENT,
        GDPR_FORGET_ME,
        PUT_DATA_REGISTRY,
        RUN_IN_BACKGROUND_IN_ORDER,
        RUN_IN_BACKGROUND_WITHOUT_ORDER,
        REGISTER_NETWORK_CALLBACK,
        FIRST_OPEN_START_SESSION,
        DELETE_USER_DATA_AND_STOP_SDK,
        RESTART_SDK
    }

    private class MessageInvokeRunnable implements Runnable {
        private Message message;

        public MessageInvokeRunnable(MessageType messageType, Object... args){
            this.message = new Message(messageType, args);
        }

        @Override
        public void run(){
            //메시지 처리
            try {
                if (message == null) {
                    idleTime();
                } else {
                    try {
                        processMessage(message);
                    }
                    catch(BeforeInitializeException be) {
                        AbxLog.w( message.messageType + "\n" +
                                String.format(
                                        LogMessageFormat.DEFAULT_ABX_CONTEXT_BEFORE_INITIALIZE_EXCEPTION,
                                        be.toString()
                                ), true
                        );
                    }
                }
            }catch (Exception e) {
                AbxLog.e(
                        String.format(
                                LogMessageFormat.DEFAULT_ABX_CONTEXT_CONTROLLER_QUEUE_EXCEPTION,
                                message.messageType.toString(),
                                e.toString()
                        ), true
                );
                AbxLog.e(Arrays.toString(e.getStackTrace()), true);
            }
        }

        /**
         * message가 없을때 백그라운드 작업을 수행한다.
         */
        private void idleTime(){
            DefaultABXContextController.this.abxContext.runOnIdleTime();
        }

        /**
         * message를 처리한다.
         * @param message
         */
        private void processMessage(Message message)
        {
            switch (message.messageType)
            {
                case CHANGE_ABX_CONTEXT:
                    {
                        IABXContext newContext = (IABXContext)message.args[0];
                        DefaultABXContextController.this.abxContext = newContext;
                    }
                    break;
                case INITIALIZE:
                    {
                        Context context = (Context)message.args[0];
                        String appkey = (String)message.args[1];
                        String secretkey = (String)message.args[2];
                        DefaultABXContextController.this.abxContext.initialize(context,appkey,secretkey);

                        ABXBooleanState.getInstance().sdkInitializedFlag.getAndSet(true);
                    }
                    break;
                case SAVE_USER_PROPERTY:
                    {
                        UserPropertyCommand userPropertyCommand = (UserPropertyCommand)message.args[0];
                        DefaultABXContextController.this.abxContext.saveUserProperty(userPropertyCommand);
                    }
                    break;
                case SAVE_USER_PROPERTY_WITHOUT_EVENT:
                    {
                        UserPropertyCommand userPropertyCommand = (UserPropertyCommand)message.args[0];
                        DefaultABXContextController.this.abxContext.saveUserPropertyWithoutEvent(userPropertyCommand);
                    }
                    break;
                case LOG_EVENT:
                    {
                        LogEventParameter logEventParameter = (LogEventParameter)message.args[0];

                        //coreWrapper에서 들어오는 logevent의 eventId앞 시간과 event_datetime을 일치시킴
                       if(logEventParameter.event_datetime == null) {
                            logEventParameter.event_datetime = CommonUtils.getCurrentUTCInDBFormat();
                            logEventParameter.eventId = CommonUtils.randomUUIDWithCurrentTime();
                        }

                        DefaultABXContextController.this.abxContext.logEvent(logEventParameter);
                    }
                    break;
                case ON_RESUME:
                    {
                        Activity activity = (Activity)message.args[0];
                        DefaultABXContextController.this.abxContext.onResume(activity);
                    }
                    break;
                case ON_PAUSE:
                    {
                        DefaultABXContextController.this.abxContext.onPause();
                    }
                    break;
                case DEEPLINK:
                    {
                        Activity deeplinkActivity = (Activity)message.args[0];
                        DefaultABXContextController.this.abxContext.deeplink(deeplinkActivity);
                    }
                    break;
                case DEEPLINK_WITH_INTENT:
                    {
                        Intent deeplinkIntent = (Intent) message.args[0];
                        DefaultABXContextController.this.abxContext.deeplinkWithIntent(deeplinkIntent);
                    }
                break;
                case GDPR_FORGET_ME:
                    {
                        new GdprForgetMeUseCase(repository).execute();

                        disableSDK("GDPR FORGET ME");
                    }
                    break;
                case PUT_DATA_REGISTRY:
                    {
                        DataUnit dataUnit = (DataUnit) message.args[0];
                        DefaultABXContextController.this.abxContext.putDataRegistry(dataUnit);
                    }
                    break;
                case RUN_IN_BACKGROUND_IN_ORDER:
                case RUN_IN_BACKGROUND_WITHOUT_ORDER:
                case FIRST_OPEN_START_SESSION: {
                        Runnable runnable = (Runnable) message.args[0];
                        runnable.run();
                    }
                    break;
                case REGISTER_NETWORK_CALLBACK:
                    {
                        DefaultABXContextController.this.abxContext.registerNetworkCallback();
                    }
                    break;
                case DELETE_USER_DATA_AND_STOP_SDK:
                    {

                        if (getDRState() == DRState.DELETE_SYNCED) {
                            AbxLog.d("Delete user data already synced!", true);
                            return;
                        }

                        if (isDeleteProcessingNow) {
                            AbxLog.d("Delete API is already processing!", true);
                        }

                        isDeleteProcessingNow = true;

                        String userId = (String) message.args[0];
                        Runnable onSuccess = (Runnable) message.args[1];
                        Runnable onFail = (Runnable) message.args[2];

                        DefaultABXContextController.this.abxContext.deleteUserData(userId, new ApiConnectionManager.Result() {
                            @Override
                            public void connectSuccess(String responseString, int responseCode) {
                                AbxLog.d("Delete User Data API connect success!!", true);
                                deleteLocalDataAndStopSDK();
                                if(onSuccess != null)
                                    onSuccess.run();
                            }

                            @Override
                            public void connectFail(int responseCode) {
                                AbxLog.w("Delete User Data API connect failed!! responseCode : " + responseCode, true);
                                saveDeleteFailDataAndStopSDK();
                                if(onFail != null)
                                    onFail.run();
                            }
                        });
                    }
                break;
                case RESTART_SDK:
                    {
                        if (isInitRestartProcessingNow) {
                            AbxLog.d("InitRestart API is already processing!", true);
                        }

                        if (getDRState() != DRState.INIT_RESTART_SYNCED)
                            isInitRestartProcessingNow = true;

                        String userId = (String) message.args[0];
                        Runnable onSuccess = (Runnable) message.args[1];
                        Runnable onFail = (Runnable) message.args[2];

                        //ActiveABXContext에서 restart 요청이 된다. -> InitRestart 실패 시 DisableABXContext 적용.
                        DefaultABXContextController.this.abxContext.restartSDK(userId, new ApiConnectionManager.Result() {
                            @Override
                            public void connectSuccess(String responseString, int responseCode) {
                                AbxLog.d("Restart SDK API connect success!!", true);
                                saveRestartSuccessData();
                                if(onSuccess != null)
                                    onSuccess.run();
                            }

                            @Override
                            public void connectFail(int responseCode) {
                                AbxLog.w("Restart SDK API connect failed!! responseCode : " + responseCode, true);
                                saveRestartFailData();
                                if(onFail != null)
                                    onFail.run();
                            }
                        });
                        break;
                    }
                default:
                    throw new IllegalStateException("Unexpected value: " + message.messageType);
            }

            //메시지 처리 이후 백그라운드 작업을 수행한다.
            DefaultABXContextController.this.abxContext.runOnIdleTime();
        }
    }

    @Override
    public IABXComponentsFactory getComponentsFactory(){
        return this.componentsFactory;
    }

    @Override
    public void setDataRegistry(DataRegistry dataRegistry) {
        this.dataRegistry = dataRegistry;
        ABXBooleanState.getInstance().getGdprData(dataRegistry);
    }

    @Override
    public DRState getDRState() {

        if (dataRegistry == null) {
            AbxLog.d("DRSTATE :: Cannot get DRState before initialized!", true);
            return null;
        }

        int zerothBit = dataRegistry.safeGetBoolean(DataRegistryKey.BOOLEAN_IS_SDK_STOPPED_SERVER_SYNC, false) ? 1 : 0;
        int firstBit =  dataRegistry.safeGetBoolean(DataRegistryKey.BOOLEAN_IS_SDK_STOPPED, false) ? 1 : 0;

        int result = 2 * firstBit + zerothBit;

        switch (result) {
            case 0:
                AbxLog.d("SDK STOPPED STATE :: SDK normal state!", true);
                break;
            case 1:
                AbxLog.d("SDK STOPPED STATE :: SDK restarted by user, but not synced!", true);
                break;
            case 2:
                AbxLog.d("SDK STOPPED STATE :: Deleting user data requested, but not synced!", true);
                break;
            case 3:
                AbxLog.d("SDK STOPPED STATE :: Deleting user data requested, also synced!", true);
                break;
        }

        return DRState.fromInteger(result);
    }

    @Override
    public void changeABXContext(IABXContext newContext) {
        if (abxContext.getClass() == newContext.getClass()) {
            AbxLog.d("Same ABXContext! : " + abxContext.getClass().getName() + ", Cannot change ABXContext!", true);
            return;
        }
        this.abxContext = newContext;
        this.componentsFactory.setABXContext(newContext);
    }

    public void initialize(Context context, String appkey, String secretkey) {
        submit(MessageType.INITIALIZE, context, appkey, secretkey);
    }

    public void saveUserProperty(UserPropertyCommand userPropertyCommand) {
        submit(MessageType.SAVE_USER_PROPERTY, userPropertyCommand);
    }

    public void saveUserPropertyWithoutEvent(UserPropertyCommand userPropertyCommand) {
        submit(MessageType.SAVE_USER_PROPERTY_WITHOUT_EVENT, userPropertyCommand);
    }

    public void logEvent(LogEventParameter logEventParameter) {
        submit(MessageType.LOG_EVENT, logEventParameter);
    }

    public void onResume(Activity activity) {
        ABXBooleanState.getInstance().inForeground.getAndSet(true);
        submit(MessageType.ON_RESUME, activity);
    }

    public void onPause() {
        ABXBooleanState.getInstance().inForeground.getAndSet(false);
        submit(MessageType.ON_PAUSE);
    }

    public void onDestroy(Activity activity) {
        repository.dismissInAppMessageDialog();
        repository.deleteCurrentActivity(activity);
    }

    public void deeplink(Activity deeplinkActivity) {
        submit(MessageType.DEEPLINK, deeplinkActivity);
    }

    public void deeplinkWithIntent(Intent deeplinkIntent) {
        submit(MessageType.DEEPLINK_WITH_INTENT, deeplinkIntent);
    }

    public void gdprForgetMe(){
        submit(MessageType.GDPR_FORGET_ME);
    }

    public void putDataRegistry(DataUnit dataUnit){
        submit(MessageType.PUT_DATA_REGISTRY, dataUnit);
    }

    public void runInBackGroundInOrder(Runnable runnable){
        submit(MessageType.RUN_IN_BACKGROUND_IN_ORDER, runnable);
    }

    public void runInBackGroundWithoutOrder(Runnable runnable){
        submit(MessageType.RUN_IN_BACKGROUND_WITHOUT_ORDER, runnable);
    }

    public void processFirstOpenStartSession(Runnable runnable) {
        submit(MessageType.FIRST_OPEN_START_SESSION, runnable);
    }

    public void registerNetworkCallback(){
        submit(MessageType.REGISTER_NETWORK_CALLBACK);
    }

    public void deleteUserDataAndStopSDK(String userId, Runnable onSuccess, Runnable onFail){
        submit(MessageType.DELETE_USER_DATA_AND_STOP_SDK, userId, onSuccess, onFail);
    }

    public void restartSDK(String userId, Runnable onSuccess, Runnable onFail){
        submit(MessageType.RESTART_SDK, userId, onSuccess, onFail);
    }

    public DataRegistry getDataRegistry() throws NullPointerException{
        if(dataRegistry == null){
            throw new NullPointerException();
        }

        return dataRegistry;
    }

    public UserPropertyModel getCurrentUserPropertyModel(){
        try {
            UserPropertyManager userPropertyManager = this.componentsFactory.createOrGetUserPropertyManager();
            return userPropertyManager.getCurrentUserPropertyModel();
        } catch (IABXComponentsFactory.ComponentsCanNotCreateException e) {
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
            return new UserPropertyModel(null, new HashMap<String, Object>());
        }
    }

    public void triggerInAppMessage(String eventName){
        repository.triggerInAppMessage(eventName);
    }

    private void submit(MessageType messageType, Object... args){
        if(executorService == null){
            AbxLog.d("controller.submit is called before controller.startcontroller", true);
            return;
        }

        if (executorService.isShutdown())
            executorService = Executors.newSingleThreadExecutor();

        if (isInitSubmitted && messageType == MessageType.RUN_IN_BACKGROUND_WITHOUT_ORDER) {
            executorService.submit(new MessageInvokeRunnable(messageType, args));
            return;
        }

        if (isInitSubmitted && messageType == MessageType.RESTART_SDK) {
            executorService.submit(new MessageInvokeRunnable(messageType, args));
            return;
        }

        if (isInitSubmitted && messageType == MessageType.DELETE_USER_DATA_AND_STOP_SDK) {
            executorService.submit(new MessageInvokeRunnable(messageType, args));
            return;
        }

        if (!isSDKFirstOpened && isOnResumeSubmitted) {
            executorService.submit(new MessageInvokeRunnable(messageType, args));
            return;
        }
        else if (isSDKFirstOpened && isFirstOpenSubmitted) {
            executorService.submit(new MessageInvokeRunnable(messageType, args));
            return;
        }

        if (messageType == MessageType.INITIALIZE)
            initQueue.offer(new MessageInvokeRunnable(messageType, args));
        else if (!isOnResumeSubmitted && messageType == MessageType.ON_RESUME)
            onResumeQueue.offer(new MessageInvokeRunnable(messageType, args));
        else if (isSDKFirstOpened && messageType == MessageType.FIRST_OPEN_START_SESSION)
            firstOpenQueue.offer(new MessageInvokeRunnable(messageType, args));
        else etcQueue.offer(new MessageInvokeRunnable(messageType, args));

        if (!initQueue.isEmpty()) {
            executorService.submit(initQueue.poll());
            isInitSubmitted = true;
        }

        if (isInitSubmitted && !onResumeQueue.isEmpty() && !isOnResumeSubmitted) {
            executorService.submit(onResumeQueue.poll());
            isOnResumeSubmitted = true;
        }

        if (isOnResumeSubmitted && isSDKFirstOpened && !firstOpenQueue.isEmpty()) {
            executorService.submit(firstOpenQueue.poll());
            isFirstOpenSubmitted = true;
        }

        if (isSDKFirstOpened && isFirstOpenSubmitted) {
            while (!etcQueue.isEmpty())
                executorService.submit(etcQueue.poll());
        }
        else if (!isSDKFirstOpened && isOnResumeSubmitted) {
            while (!etcQueue.isEmpty())
                executorService.submit(etcQueue.poll());
        }
    }
  
    private void deleteLocalDataAndStopSDK() {
        //Restart 시 appKey, secretKey 가 필요하다.
        String appKey = dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY, null);
        String secretKey = dataRegistry.safeGetString(DataRegistryKey.STRING_SECRETKEY, null);

        //Delete API 연결애 성공했으므로 local data 삭제.
        dataRegistry.clearRegistry();

        //Delete 종료
        isDeleteProcessingNow = false;

        //다음 SDK 시작시에 stop state를 판단하기 위한 flag 저장.
        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.BOOLEAN_IS_SDK_STOPPED,
                true,
                5,
                this.getClass().getName(),
                true
        ));

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.BOOLEAN_IS_SDK_STOPPED_SERVER_SYNC,
                true,
                5,
                this.getClass().getName(),
                true
        ));

        //appKey, secretKey 저장
        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_APPKEY,
                appKey,
                5,
                this.getClass().getName(),
                true
        ));

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_SECRETKEY,
                secretKey,
                5,
                this.getClass().getName(),
                true
        ));

        dataRegistry.saveRegistry();

        //stop sdk.
        changeABXContext(new DisabledABXContext("SDK STOPPED", repository));
    }

    private void saveDeleteFailDataAndStopSDK() {

        //Restart 시 appKey, secretKey 가 필요하다.
        String appKey = dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY, null);
        String secretKey = dataRegistry.safeGetString(DataRegistryKey.STRING_SECRETKEY, null);

        dataRegistry.clearRegistry();

        //Delete 종료
        isDeleteProcessingNow = false;

        //다음 SDK 시작시에 stop state를 판단하기 위한 flag 저장.
        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.BOOLEAN_IS_SDK_STOPPED,
                true,
                5,
                this.getClass().getName(),
                true
        ));

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.BOOLEAN_IS_SDK_STOPPED_SERVER_SYNC,
                false,
                5,
                this.getClass().getName(),
                true
        ));

        //appKey, secretKey 저장
        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_APPKEY,
                appKey,
                5,
                this.getClass().getName(),
                true
        ));

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_SECRETKEY,
                secretKey,
                5,
                this.getClass().getName(),
                true
        ));

        dataRegistry.saveRegistry();

        //stop sdk.
       changeABXContext(new DisabledABXContext("SDK STOPPED", repository));
    }

    private void saveRestartSuccessData() {

        if (getDRState() != DRState.INIT_RESTART_SYNCED) {
            AbxLog.d("Initialize restart succeeded", true);

            isInitRestartProcessingNow = false;

            dataRegistry.putDataRegistry(new DataUnit(
                    DataRegistryKey.BOOLEAN_IS_SDK_STOPPED,
                    false,
                    5,
                    this.getClass().getName(),
                    true
            ));

            dataRegistry.putDataRegistry(new DataUnit(
                    DataRegistryKey.BOOLEAN_IS_SDK_STOPPED_SERVER_SYNC,
                    false,
                    5,
                    this.getClass().getName(),
                    true
            ));

            //Init Restart 성공했으므로 Init Restart Event Datetime 삭제.
            dataRegistry.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.STRING_INIT_RESTART_EVENT_DATETIME,
                            null,
                            5,
                            this.getClass().getName(),
                            true
                    )
            );

            //executor 가 정상동작하도록 flag 값 수정
            this.isSDKFirstOpened = false;

            if (DefaultABXContextController.this.abxContext instanceof DisabledABXContext) {
                changeABXContext(
                        new InitializingABXContext(this, repository));

                Context context = null;
                try {
                    context = componentsFactory.getAndroidContext();
                } catch (IABXComponentsFactory.ComponentsCanNotCreateException e) {
                    AbxLog.d(Arrays.toString(e.getStackTrace()), true);
                }
                String appKey = dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY, null);
                String secretKey = dataRegistry.safeGetString(DataRegistryKey.STRING_SECRETKEY, null);

                this.initialize(context, appKey, secretKey);

                //session 정보 초기화
                ABXBooleanState.getInstance().isSessionStarted.getAndSet(false);
                //이때 abxContext는 무조건 Active 이다. 이벤트 처리 전 바로 start_session 이벤트를 날리기 위함.
                this.onResume(null);
            }
        }
        else {
            AbxLog.d("Duplicated restart succeeded", true);
            //do nothing
        }

    }

    private void saveRestartFailData() {
        //initialize restart
        if (getDRState() != DRState.INIT_RESTART_SYNCED) {
            AbxLog.d("Initialize restart failed", true);

            isInitRestartProcessingNow = false;

            dataRegistry.putDataRegistry(new DataUnit(
                    DataRegistryKey.BOOLEAN_IS_SDK_STOPPED,
                    false,
                    5,
                    this.getClass().getName(),
                    true
            ));

            dataRegistry.putDataRegistry(new DataUnit(
                    DataRegistryKey.BOOLEAN_IS_SDK_STOPPED_SERVER_SYNC,
                    true,
                    5,
                    this.getClass().getName(),
                    true
            ));

            dataRegistry.saveRegistry();

            //executor 가 정상동작하도록 flag 값 수정
            this.isSDKFirstOpened = false;

            if (DefaultABXContextController.this.abxContext instanceof DisabledABXContext) {
                changeABXContext(
                        new InitializingABXContext(this, repository));

                Context context = null;
                try {
                    context = componentsFactory.getAndroidContext();
                } catch (IABXComponentsFactory.ComponentsCanNotCreateException e) {
                    AbxLog.e(Arrays.toString(e.getStackTrace()),true);
                }
                String appKey = dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY, null);
                String secretKey = dataRegistry.safeGetString(DataRegistryKey.STRING_SECRETKEY, null);

                this.initialize(context, appKey, secretKey);

                //session 정보 초기화
                ABXBooleanState.getInstance().isSessionStarted.getAndSet(false);
                //이때 abxContext는 무조건 Active 이다. 이벤트 처리 전 바로 start_session 이벤트를 날리기 위함.
                this.onResume(null);
            }
        }
        else {
            AbxLog.d("Duplicated restart failed", true);
            //do nothing
        }

    }

    public boolean isGdprForgetMe() {
        if (ABXBooleanState.getInstance().isGdprForgetMe()) {
            if (!ABXBooleanState.getInstance().isGdprForgetMeSync()) {
                new GdprForgetMeUseCase(repository).execute();
            }
            return true;
        }

        DRState currentState = getDRState();

        if (currentState == DRState.DELETE_NOT_SYNCED) {
            return true;
        }
        else if (currentState == DRState.DELETE_SYNCED) {
            return true;
        }

        return ABXBooleanState.getInstance().isGdprForgetMe();
    }

    public void disableSDK(String disabledReason) {
        changeABXContext(new DisabledABXContext(disabledReason, repository));
    }
  
    @Override
    public void update(LogEventParameter logEventParameter) {
        submit(MessageType.LOG_EVENT, logEventParameter);
    }
}