package io.adbrix.sdk.component;

import java.util.ArrayList;
import java.util.Queue;
import java.util.Timer;
import java.util.TimerTask;

import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.EventPackage;

public class EventUploadIntervalManager implements IObservable<Void>{
    private int COUNT_INTERVAL = 30;
    private int TIME_INTERVAL = 60_000;
    private Timer eventBusTimer;
    private ArrayList<IObserver<Void>> observerArrayList = new ArrayList<>();

    private EventUploadIntervalManager() {}

    private static class Singleton{
        private static final EventUploadIntervalManager instance = new EventUploadIntervalManager();
    }

    public static EventUploadIntervalManager getInstance() {
        return Singleton.instance;
    }

    public void setCountInterval(int countInterval) {
        this.COUNT_INTERVAL = countInterval;
    }

    public void setTimeInterval(int timeInterval) {
        this.TIME_INTERVAL = timeInterval;
        if(eventBusTimer != null) {
            eventBusTimer.cancel();
        }
        scheduleEventSendingJob();
    }

    public void manageEventUploadInterval(Runnable eventSendingTask, Queue<EventPackage> packageQueue) {
        AbxLog.d("EventQueue size : " + packageQueue.size(), true);
        scheduleEventSendingJob();
        countEventSendingJob(eventSendingTask, packageQueue);
    }

    private void scheduleEventSendingJob(){
        if (eventBusTimer == null) {
            eventBusTimer = new Timer();
            TimerGroup.getInstance().setEventBusTimer(eventBusTimer);
        }
        else return;

        eventBusTimer.schedule(new TimerTask() {
            @Override
            public void run() {
                AbxLog.d("Trying to upload events every " + TIME_INTERVAL / 1000 + " seconds.", true);
                notifyObserver(null);
            }
        }, this.TIME_INTERVAL, this.TIME_INTERVAL);
    }

    private void countEventSendingJob(Runnable eventSendingTask, Queue<EventPackage> packageQueue) {
        if (packageQueue.peek() == null)
            return;

        if (packageQueue.size() >= (COUNT_INTERVAL / CoreConstants.EVENT_BUFFER_COUNT)){
            eventSendingTask.run();
        }
    }

    @Override
    public void add(IObserver<Void> observer) {
        observerArrayList.add(observer);
    }

    @Override
    public void delete(IObserver<Void> observer) {
        observerArrayList.remove(observer);
    }

    @Override
    public void notifyObserver(Void v) {
        for (IObserver<Void> observer : observerArrayList){
            observer.update(v);
        }
    }
}
