package io.adbrix.sdk.component;

import io.adbrix.sdk.domain.model.EventModel;
import io.adbrix.sdk.domain.model.EventPackage;

import java.util.ArrayList;


public class EventBuffer implements IObserver<Void>{

    private int bufferCapacity;
    private ArrayList<EventModel> currentBuffer = null;
    private EventPackageContainer eventPackageContainer;

    public static class BufferCapacityLimitsException extends Exception {
        BufferCapacityLimitsException(String message){
            super(message);
        }
    }

    public EventBuffer(int bufferCapacity, EventPackageContainer eventPackageContainer) throws BufferCapacityLimitsException {
        this.bufferCapacity = bufferCapacity;
        this.eventPackageContainer = eventPackageContainer;

        if(bufferCapacity < 5)
        {
            throw new BufferCapacityLimitsException("버퍼크기는 5이상 설정해야 합니다.");
        }
    }

    public synchronized void addEventModel(EventModel model, boolean noWait) {
        ArrayList<EventModel> buffer = getCurrentBuffer();
        buffer.add(model);

        if (noWait)
            clearBufferAndFlushContainer();
        else
            addEventPackageToContainerIfBufferFull();
    }

    public void flushAllNow() {
        clearBufferAndFlushContainer();
    }

    public void flushContainerAtIntervals(){
        this.eventPackageContainer.flushAtIntervals();
    }

    @Override
    public void update(Void v) {
        flushAllNow();
    }

    public void saveUnsentEvents(){
        if (this.currentBuffer == null) {
            this.currentBuffer = new ArrayList<>(bufferCapacity);
        }

        if (currentBuffer.size() > 0){
            eventPackageContainer.addEventPackage(new EventPackage(currentBuffer));
            eventPackageContainer.updateUnsentEventPackagesInDataRegistry();
            this.currentBuffer = new ArrayList<>(bufferCapacity);
        }
        else {
            eventPackageContainer.updateUnsentEventPackagesInDataRegistry();
        }
    }

    private ArrayList<EventModel> getCurrentBuffer() {
        if (this.currentBuffer == null) {
            this.currentBuffer = new ArrayList<>(bufferCapacity);
            return currentBuffer;
        }

        addEventPackageToContainerIfBufferFull();

        return this.currentBuffer;
    }

    private synchronized void clearBufferAndFlushContainer(){
        if (currentBuffer == null)
            return;

        synchronized (currentBuffer) {
            //onNoWaitEvent
            if (currentBuffer.size() > 0) {
                eventPackageContainer.addEventPackage(new EventPackage(currentBuffer));
                eventPackageContainer.flushEventsNotSent();
                this.currentBuffer = new ArrayList<>(bufferCapacity);
            } else {
                eventPackageContainer.flushEventsNotSent();
            }
        }
    }

    private void addEventPackageToContainerIfBufferFull(){
        if (currentBuffer == null)
            return;

        synchronized (currentBuffer) {
           //onBufferFull
            if (currentBuffer.size() >= bufferCapacity) {
                eventPackageContainer.addEventPackage(new EventPackage(currentBuffer));
                this.currentBuffer = new ArrayList<>(bufferCapacity);
            }
        }
    }
}
