package io.adbrix.sdk.data.net;

import android.content.Context;
import android.os.Build;

import com.google.android.gms.common.GooglePlayServicesNotAvailableException;
import com.google.android.gms.common.GooglePlayServicesRepairableException;
import com.google.android.gms.security.ProviderInstaller;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.HttpURLConnection;
import java.net.URL;
import java.security.GeneralSecurityException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.data.SdkVersion;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.IApiModel;
import io.adbrix.sdk.utils.Base64;
import io.adbrix.sdk.utils.CoreUtils;

public class ApiConnection {

    private DataRegistry dataRegistry;
    private Context androidContext;
    private int responseCode;
    private StringBuilder responseStringBuilder;
    private String dynamicHashKey;
    private HostnameVerifier TRUSTED_VERIFIER;
    private SSLSocketFactory TRUSTED_FACTORY;
    private JSONObject body;
    private String urlString;
    private RequestMethod requestMethod;

    public ApiConnection(DataRegistry dataRegistry, Context androidContext){
        this.dataRegistry = dataRegistry;
        this.androidContext = androidContext;
    }

    private enum RequestMethod {
        POST, GET
    }

    public int getResponseCode() {
        return responseCode;
    }

    public String getResponseString() {
        return responseStringBuilder.toString();
    }

    public ApiConnection GET(String urlString){
        this.urlString = urlString;
        this.requestMethod = RequestMethod.GET;
        return this;
    }

    public ApiConnection POST(IApiModel apiModel) throws JSONException {
        String appKey = dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY, null);

        this.body = apiModel.getJson();
        this.urlString = String.format(apiModel.getUrlString(), appKey);
        this.requestMethod = RequestMethod.POST;

        return this;
    }

    public String request(){
        if (this.requestMethod == RequestMethod.GET)
            get(urlString);
        else post(body, urlString);

        if (responseStringBuilder != null) {
            return responseStringBuilder.toString();
        }
        else return null;
    }

    private void get(String urlString) {
        HttpURLConnection connection = null;
        InputStream inputStream;
        BufferedReader bufferedReader;

        try {
            connection = this.createConnection(urlString);
            connection.setRequestMethod("GET");

            connection.setInstanceFollowRedirects(false);

            responseStringBuilder = new StringBuilder();
            responseCode = connection.getResponseCode();

            if (200 <= responseCode && responseCode <= 299){
                inputStream = connection.getInputStream();
            }
            else {
                inputStream = connection.getErrorStream();
            }

            bufferedReader = new BufferedReader(new InputStreamReader(inputStream));

            String currentLine;

            while ((currentLine = bufferedReader.readLine()) != null)
                responseStringBuilder.append(currentLine);

            bufferedReader.close();

        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (connection != null) {
                connection.disconnect();
            }
        }
    }

    private void post(JSONObject body, String urlString){
        HttpURLConnection connection = null;
        InputStream inputStream;
        BufferedWriter bufferedWriter;
        BufferedReader bufferedReader;

        try {
            if (urlString.contains("drworks")) {
                AbxLog.d(body.toString(4), true);
                AbxLog.d(urlString, true);
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }

        try {
            connection = this.createConnection(urlString);
            connection.setRequestMethod("POST");

            String userAgent = String.format("IGAWorks/%s (Android %s; U; %s Build/%s)",
                    SdkVersion.SDK_VERSION,
                    dataRegistry.safeGetString(DataRegistryKey.STRING_OS, null),
                    dataRegistry.safeGetString(DataRegistryKey.STRING_MODEL, null),
                    dataRegistry.safeGetString(DataRegistryKey.STRING_BUILD_ID, null)
            );

            connection.setRequestProperty("User-Agent", userAgent);

            if (urlString.contains("opengdpr_requests")){
                connection.setRequestProperty("Content-Type", "application/json; charset=utf-8");
            }
            else {
                connection.setRequestProperty("Content-Type", "application/octet-stream; charset=utf-8");
            }
            connection.setInstanceFollowRedirects(false);

            connection.setRequestProperty("abx-auth-time", getAbxAuthTime(body));
            connection.setRequestProperty("abx-auth-cs", getAbxAuthCs(body));
            connection.setDoInput(true);
            connection.setDoOutput(true);
            connection.setRequestProperty("Accept-Charset", "UTF-8");

            bufferedWriter = new BufferedWriter(new OutputStreamWriter(connection.getOutputStream(),
                    "UTF-8"));

            if (urlString.contains("opengdpr_requests")){
                bufferedWriter.write(body.toString());
            }
            else {
                bufferedWriter.write(Base64.encode(body.toString()));
            }
            bufferedWriter.flush();
            bufferedWriter.close();

            responseStringBuilder = new StringBuilder();
            responseCode = connection.getResponseCode();

            if (200 <= responseCode && responseCode <= 299){
                inputStream = connection.getInputStream();
            }
            else {
                inputStream = connection.getErrorStream();
            }

            bufferedReader = new BufferedReader(new InputStreamReader(inputStream));

            String currentLine;

            while ((currentLine = bufferedReader.readLine()) != null)
                responseStringBuilder.append(currentLine);

            bufferedReader.close();

        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (connection != null) {
                connection.disconnect();
            }
        }
    }

    public boolean isHttpOK(){
        return HttpsURLConnection.HTTP_OK <= responseCode && responseCode < 300;
    }

    public boolean isWrongAppkey(){
        return responseCode == 404 || (responseCode > 500 && responseCode < 600);
    }

    public boolean isInvalidAppkey(){
        return responseCode == 400;
    }

    private HttpURLConnection createConnection(String urlString) throws IOException {
        URL url = new URL(urlString);

        final HttpURLConnection connection = (HttpURLConnection) url.openConnection();
        connection.setReadTimeout(CoreConstants.NETWORK_TIMEOUT);
        connection.setConnectTimeout(CoreConstants.NETWORK_TIMEOUT);

        if (urlString.startsWith("https")) {
            configureHttpsConnection((HttpsURLConnection) connection);
        }
        return connection;
    }

    private void configureHttpsConnection(HttpsURLConnection connection) {
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP)
        {
            try {
                SSLContext.getInstance("TLSv1.2");
            } catch (NoSuchAlgorithmException e) {
                e.printStackTrace();
            }

            try {
                ProviderInstaller.installIfNeeded(androidContext.getApplicationContext());
            } catch (GooglePlayServicesRepairableException | GooglePlayServicesNotAvailableException e) {
                e.printStackTrace();
            }
        }

        /*
        2020-09-01
        리마스터는 모두 https를 사용하고 있으므로 우회하는 코드를 사용하 않는다.
        google xtrust509 issue report를 해결하기 위한 주석처리.
         */
//        connection.setHostnameVerifier(getTrustedVerifier());
//        connection.setSSLSocketFactory(getTrustedFactory());
    }

    /**
     * Configure HTTPS connection to trust all hosts using a custom
     * {@link HostnameVerifier} that always returns <code>true</code> for each
     * host verified
     * <p>
     * This method does nothing if the current request is not a HTTPS request
     *
     * @return this request
     */
    private HostnameVerifier getTrustedVerifier() {
        if (TRUSTED_VERIFIER == null)
        {
            TRUSTED_VERIFIER = new HostnameVerifier() {
                @Override
                public boolean verify(String hostname, SSLSession session) {
                    return true;
                }
            };
        }

        return TRUSTED_VERIFIER;
    }

    /**
     * Configure HTTPS connection to trust all certificates
     * <p>
     * This method does nothing if the current request is not a HTTPS request
     *
     * @return this request
     * @throws 'HttpRequestException
     */
    private SSLSocketFactory getTrustedFactory() {
        if (TRUSTED_FACTORY == null)
        {
            final TrustManager[] trustAllCerts = new TrustManager[]{ new X509TrustManager() {

                public X509Certificate[] getAcceptedIssuers() {
                    return new X509Certificate[0];
                }

                public void checkClientTrusted(X509Certificate[] chain, String authType) {
                    // Intentionally left blank
                }

                public void checkServerTrusted(X509Certificate[] chain, String authType) throws CertificateException {
                    try {
                        chain[0].checkValidity();
                    } catch (Exception e) {
                        throw new CertificateException("Certificate not valid or trusted.");
                    }
                }
            }};

            try {
                SSLContext context = SSLContext.getInstance("TLS");
                context.init(null, trustAllCerts, new SecureRandom());
                TRUSTED_FACTORY = context.getSocketFactory();
            } catch (GeneralSecurityException e) {
                e.printStackTrace();
            }
        }

        return TRUSTED_FACTORY;
    }

    private String getAbxAuthTime(JSONObject jsonObject) {
        Date currentTime = null;
        SimpleDateFormat sdfKST = CoreUtils.createDateFormat(CoreConstants.DB_DATE_FORMAT);

        try {
            if (jsonObject.has("common"))
            {
                JSONObject common = jsonObject.getJSONObject("common");
                if (common.has("request_datetime"))
                {
                    Object request_datetime = common.get("request_datetime");
                    if (request_datetime instanceof String)
                    {
                        currentTime = sdfKST.parse((String) request_datetime);
                    }
                }
            }
        } catch (JSONException | ParseException e) {
            e.printStackTrace();
        } finally {
            if (currentTime == null) {
                currentTime = new Date();
            }
        }

        setDynamicHashKey(currentTime);
        return sdfKST.format(currentTime);
    }

    private void setDynamicHashKey(Date currentTime) {
        DateFormat h = CoreUtils.createDateFormat("HH");
        String secretKey = dataRegistry.safeGetString(DataRegistryKey.STRING_SECRETKEY, "");
        char[] epK = {
                (char) 99, (char) 50, (char) 100, (char) 108,
                (char) 97, (char) 51, (char) 74, (char) 122,
                (char) 98, (char) 88, (char) 82, (char) 105,
                (char) 89, (char) 81, (char) 61, (char) 61
        };
        int index = (Integer.parseInt(h.format(currentTime)) * 174 + 8123 * 13) % 32;
        String fullHashKey = secretKey + Base64.decode(String.valueOf(epK));
        dynamicHashKey = fullHashKey.substring(index) + fullHashKey.substring(0, index);
    }

    private String getAbxAuthCs(JSONObject jsonObject) {
        // first encrypt by hashkey, second encrypt by dsk
        return CoreUtils.HMAC_SHA256(dynamicHashKey, getDynamicHashKeyMD5(Base64.encode(jsonObject.toString())));
    }

    private String getDynamicHashKeyMD5(String string) {
        try {
            // Create MD5 Hash
            MessageDigest digest = java.security.MessageDigest.getInstance("MD5");
            digest.update(string.getBytes());
            byte[] messageDigest = digest.digest();

            // Create Hex String
            StringBuffer hexString = new StringBuffer();
            for (int i = 0; i < messageDigest.length; i++)
            {
                hexString.append(String.format("%02X", 0xFF & messageDigest[i]));
            }

            return hexString.toString();
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }
        return string;
    }
}
