package io.adbrix.sdk.configuration;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.DeeplinkPostingObservable;
import io.adbrix.sdk.component.DeferredDeeplinkPostingObservable;
import io.adbrix.sdk.component.EventUploadIntervalManager;
import io.adbrix.sdk.component.IObserver;
import io.adbrix.sdk.component.OsPushEnableObservable;
import io.adbrix.sdk.component.PushOpenMonitor;
import io.adbrix.sdk.data.ABXBooleanState;
import io.adbrix.sdk.data.S3ConfigHandler;
import io.adbrix.sdk.data.SdkVersion;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.domain.model.UserPropertyCommand;
import io.adbrix.sdk.domain.model.UserPropertyModel;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.CoreUtils;

public class CoreWrapper {
    private static DefaultABXContextController controller= DefaultABXContextController.getInstance();
    private static final String ADBRIXRM_CLASS_NAME = "com.igaworks.v2.core.AdBrixRm";
    private static final String ABX_PUSH_COMMON_DAO_CLASS_NAME = "com.igaworks.v2.core.push.notification.AbxPushCommonDAO";
    private static final String ABX_POP_UP_COMMON_DAO_CLASS_NAME = "com.igaworks.v2.core.push.popup.AbxPopUpCommonDAO";
    private static final String PUSH_CONTROLLER_CLASS_NAME = "com.igaworks.v2.core.push.PushController";

    private CoreWrapper(){}

    public static void startController(Context appContext){
        controller.startController(appContext);
    }

    public static void stopController(){
        controller.stopController();
    }

    public static void initialize(Context appContext, String appKey, String secretKey){
        controller.initialize(appContext, appKey, secretKey);
    }

    public static void registerNetworkCallback(){
        controller.registerNetworkCallback();
    }

    public static void gdprForgetMe(){
        controller.gdprForgetMe();
    }

    public static void postAbxEvent(String eventName){
        LogEventParameter eventParameter = new LogEventParameter(
                CoreConstants.GROUP_ABX,
                eventName,
                null,
                0,
                0
        );
        controller.logEvent(eventParameter);
    }

    public static void postAbxEvent(String eventName, JSONObject eventParam){
        LogEventParameter eventParameter = new LogEventParameter(
                CoreConstants.GROUP_ABX,
                eventName,
                CommonUtils.getMapFromJSONObject(eventParam),
                0,
                0
        );
        controller.logEvent(eventParameter);
    }

    public static void postSameAbxEvent(String eventName, List<JSONObject> eventParamList){
        String eventId = CommonUtils.randomUUIDWithCurrentTime();
        String prevId;
        try{
            prevId = controller.getDataRegistry().safeGetString(DataRegistryKey.STRING_PREV_ID, null);
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return;
        }

        String eventDateTime = CommonUtils.getCurrentUTCInDBFormat();

        int count = 1;
        for (JSONObject eventParamJson: eventParamList) {
            try {
                JSONObject temp = new JSONObject();
                temp.put(CoreConstants.PAGE_ABX_KEY, count);
                temp = CommonUtils.parseValueWithDataType(temp, CommonUtils.FIX_TYPE.PREFIX);

                eventParamJson.put(CoreConstants.PAGE_ABX_KEY, temp.getString(CoreConstants.PAGE_ABX_KEY));
            } catch (JSONException e) {
                AbxLog.e(Arrays.toString(e.getStackTrace()), true);
            }
            LogEventParameter eventParameter = new LogEventParameter(
                    eventId,
                    prevId,
                    CoreConstants.GROUP_ABX,
                    eventName,
                    CommonUtils.getMapFromJSONObject(eventParamJson),
                    0,
                    0,
                    eventDateTime
            );
            controller.logEvent(eventParameter);
            count++;
        }
    }

    public static void saveUserProperty(JSONObject userPropertiesJson){
        controller.saveUserProperty(makeUserPropertyCommand(userPropertiesJson));
    }

    public static void saveUserPropertyWithoutEvent(JSONObject userPropertiesJson){
        controller.saveUserPropertyWithoutEvent(makeUserPropertyCommand(userPropertiesJson));
    }

    public static void clearUserProperties(){
        UserPropertyModel userPropertyModel = getCurrentUserPropertyModel();
        UserPropertyCommand command = new UserPropertyCommand();

        for (Map.Entry<String, Object> entry : userPropertyModel.properties.entrySet()){
            command.unset(entry.getKey());
        }

        controller.saveUserPropertyWithoutEvent(command);
    }

    public static void saveCi(JSONObject ciPropertiesJson){
        controller.saveUserProperty(makeCiCommand(ciPropertiesJson));
    }

    private static UserPropertyCommand makeUserPropertyCommand(JSONObject userPropertiesJson){
        UserPropertyCommand command = new UserPropertyCommand();
        UserPropertyModel userPropertyModel = getCurrentUserPropertyModel();

        JSONObject truncatedUserPropertiesJson = CommonUtils.truncate(userPropertiesJson);
        JSONObject parsedUserPropertiesJson = CommonUtils.parseValueWithDataType(truncatedUserPropertiesJson, CommonUtils.FIX_TYPE.PREFIX);


        Iterator<?> keys = parsedUserPropertiesJson.keys();
        int currentSizeOfUserPropertyModel = userPropertyModel.properties.size();

        while(keys.hasNext()){
            String key = (String) keys.next();

            try {
                if (currentSizeOfUserPropertyModel < S3ConfigHandler.config_propertyMaxSize) {
                    command.set(key, parsedUserPropertiesJson.get(key));
                    currentSizeOfUserPropertyModel++;
                } else {
                    AbxLog.d("UserProperties reaches MAX_PROPERTY_KEYS: " + S3ConfigHandler.config_propertyMaxSize, true);
                    break;
                }
            } catch (JSONException e) {
                AbxLog.e("updateLocalUserProperties Error: " + e.toString(), true);
            }
        }

        return command;
    }

    private static UserPropertyCommand makeCiCommand(JSONObject ciPropertiesJson){
        UserPropertyCommand command = new UserPropertyCommand();
        UserPropertyModel userPropertyModel = getCurrentUserPropertyModel();

        JSONObject truncatedUserCiJson = CommonUtils.truncate(ciPropertiesJson);
        JSONObject parsedUserPropertiesJson = CommonUtils.parseValueWithDataType(truncatedUserCiJson, CommonUtils.FIX_TYPE.PREFIX);

        Iterator<?> keys = parsedUserPropertiesJson.keys();
        int currentSizeOfUserPropertyModel = userPropertyModel.properties.size();

        while(keys.hasNext()){
            String key = (String) keys.next();

            try {
                if (currentSizeOfUserPropertyModel < S3ConfigHandler.config_propertyMaxSize) {
                    command.set(CoreConstants.CI_KEY + key, parsedUserPropertiesJson.get(key));
                    currentSizeOfUserPropertyModel++;
                } else {
                    AbxLog.d("UserProperties reaches MAX_PROPERTY_KEYS: " + S3ConfigHandler.config_propertyMaxSize, true);
                    break;
                }
            } catch (JSONException e) {
                AbxLog.e("updateLocalUserProperties Error: " + e.toString(), true);
            }
        }

        return command;
    }

    public static UserPropertyModel getCurrentUserPropertyModel(){
        return controller.getCurrentUserPropertyModel();
    }

    public static boolean isLoginIdExist(String loginId){
        UserPropertyModel userPropertyModel = getCurrentUserPropertyModel();
        loginId = "string:" + loginId;

        for (Map.Entry<String, Object> entry : userPropertyModel.properties.entrySet()){
            if("user_id".equals(entry.getKey()) && loginId.equals(entry.getValue())){
                return true;
            }
        }

        return false;
    }

    public static void onPause(){
        controller.onPause();
    }

    public static void onResume(Activity activity){
        controller.onResume(activity);
    }

    public static void deeplink(Activity deeplinkActivity) {
        if (!isFirstOpenTriggered())
            controller.deeplink(deeplinkActivity);
    }

    public static void deeplinkWithIntent(Intent deeplinkIntent) {
        if (!isFirstOpenTriggered()) {
            controller.deeplinkWithIntent(deeplinkIntent);
        }
    }

    public static boolean isFirstOpenTriggered(){
        try {
            return controller.getDataRegistry().safeGetString(DataRegistryKey.STRING_LAST_FIRSTOPEN_ID, null) == null;
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return false;
        }
    }

    public static void setEnableLocationListening(Boolean isEnable) {
        controller.putDataRegistry(new DataUnit(
                DataRegistryKey.BOOLEAN_LOCATION_PERMISSION_GRANTED,
                isEnable,
                5,
                ADBRIXRM_CLASS_NAME,
                true
        ));
    }

    public static void setLocation(double latitude, double longitude) {

        controller.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_LOCATION_LWID,
                UUID.randomUUID().toString(),
                5,
                ADBRIXRM_CLASS_NAME,
                true
        ));

        controller.putDataRegistry(new DataUnit(
                DataRegistryKey.DOUBLE_LOCATION_LAT,
                latitude,
                5,
                ADBRIXRM_CLASS_NAME,
                true
        ));

        controller.putDataRegistry(new DataUnit(
                DataRegistryKey.DOUBLE_LOCATION_LNG,
                longitude,
                5,
                ADBRIXRM_CLASS_NAME,
                true
        ));
    }

    public static void event(String eventName){
        LogEventParameter eventParameter = new LogEventParameter(
                CoreConstants.GROUP_CUSTOM,
                eventName,
                null,
                0,
                0
        );
        controller.logEvent(eventParameter);
    }

    public static void event(String eventName, JSONObject eventParam){
        LogEventParameter eventParameter = new LogEventParameter(
                CoreConstants.GROUP_CUSTOM,
                eventName,
                CommonUtils.getMapFromJSONObject(eventParam),
                0,
                0
        );
        controller.logEvent(eventParameter);
    }

    public static void setCountInterval(int countInterval){
        EventUploadIntervalManager.getInstance().setCountInterval(countInterval);
        AbxLog.v("set EventUploadCountInterval : " + countInterval, true);
    }

    public static void setTimeInterval(int timeInterval){
        EventUploadIntervalManager.getInstance().setTimeInterval(timeInterval);
        AbxLog.v("set EventUploadTimeInterval : " + timeInterval, true);
    }

    public static void setAppScanEnable(boolean enable) {
        AbxLog.d("AbxApplicationScan:: DEPRECATED!!!", true);
//        if (enable){
//            controller.putDataRegistry(new DataUnit(
//                    DataRegistryKey.LONG_APP_SCAN_ON_OFF_USER,
//                    1L,
//                    5,
//                    ADBRIXRM_CLASS_NAME,
//                    true
//            ));
//            Log.d(ABXConstants.LOGTAG, "Application Scanning set to true ( CLIENT )");
//        }
//        else {
//            controller.putDataRegistry(new DataUnit(
//                    DataRegistryKey.LONG_APP_SCAN_ON_OFF_USER,
//                    0L,
//                    5,
//                    ADBRIXRM_CLASS_NAME,
//                    true
//            ));
//            Log.d(ABXConstants.LOGTAG, "Application Scanning set to false ( CLIENT )");
//        }
    }

    public static void setPushIconStyle(String smallIconName, String largeIconName, int argb) {
        try {
            controller.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.STRING_PUSH_DB_SMALL_ICON_NAME,
                            smallIconName,
                            5,
                            ABX_PUSH_COMMON_DAO_CLASS_NAME,
                            true
                    ));
            controller.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.STRING_PUSH_DB_LARGE_ICON_NAME,
                            largeIconName,
                            5,
                            ABX_PUSH_COMMON_DAO_CLASS_NAME,
                            true
                    ));
            if(argb != -1) controller.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.LONG_PUSH_DB_ARGB,
                            (long) argb,
                            5,
                            ABX_PUSH_COMMON_DAO_CLASS_NAME,
                            true
                    ));
        } catch (Exception e) {
            AbxLog.d("ERROR :: can't set push properties", true);
        }
    }

    public static void setNotificationOption(int priority, int visibility) {
        try {
            controller.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.LONG_PUSH_DB_PRIORITY,
                            (long) priority,
                            5,
                            ABX_PUSH_COMMON_DAO_CLASS_NAME,
                            true
                    ));
            controller.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.LONG_PUSH_DB_VISIBILITY,
                            (long) visibility,
                            5,
                            ABX_PUSH_COMMON_DAO_CLASS_NAME,
                            true
                    ));
        } catch (Exception e) {
            AbxLog.d("ERROR :: can't set push properties", true);
        }
    }

    public static String getSmallIconName() {
        String result;
        try {
            result = controller.getDataRegistry().safeGetString(DataRegistryKey.STRING_PUSH_DB_SMALL_ICON_NAME, null);
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return "";
        }

        if (result != null) return result;
        else {
            AbxLog.d("ERROR :: can't get small_icon_value", true);
            return "";
        }
    }

    public static String getLargeiconName() {
        String result;
        try {
            result = controller.getDataRegistry().safeGetString(DataRegistryKey.STRING_PUSH_DB_LARGE_ICON_NAME, null);
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return null;
        }

        if (result != null) return result;
        else {
            AbxLog.d("ERROR :: can't get large_icon_value", true);
            return null;
        }
    }

    public static int getARGB() {
        int result;
        try {
            result = (int) controller.getDataRegistry().safeGetLong(DataRegistryKey.LONG_PUSH_DB_ARGB, -1);
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return -1;
        }

        return result;
    }

    public static int getPriority() {
        int result;

        try {
            result = (int) controller.getDataRegistry().safeGetLong(DataRegistryKey.LONG_PUSH_DB_PRIORITY, 0);
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return 0;
        }

        return result;
    }

    public static int getVisibility() {
        int result;
        try {
            result = (int) controller.getDataRegistry().safeGetLong(DataRegistryKey.LONG_PUSH_DB_VISIBILITY, 0);
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return 0;
        }

        return result;
    }

    public static int getImportance() {
        int result;
        try {
            result = (int) controller.getDataRegistry().safeGetLong(DataRegistryKey.LONG_PUSH_DB_IMPORTANCE, 0);
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return 0;
        }

        return result;
    }

    public static void setStackingNotificationOption(boolean useStacking, boolean useTitleForStacking, String contentTitle, String contentText, String bigContentTitle, String bigContentSummaryText) {
        try {
            controller.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.LONG_PUSH_DB_USE_STACKING,
                            (long) (useStacking ? 1 : 0),
                            5,
                            ABX_PUSH_COMMON_DAO_CLASS_NAME,
                            true
                    ));
            controller.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.LONG_PUSH_DB_USE_TITLE_FOR_STACKING,
                            (long) (useTitleForStacking ? 1 : 0),
                            5,
                            ABX_PUSH_COMMON_DAO_CLASS_NAME,
                            true
                    ));
        } catch (Exception e) {
            AbxLog.d("ERROR :: can't set stacking option properties", true);
        }
    }

    public static boolean getUseStacking() {
        try {
            return controller.getDataRegistry().safeGetLong(DataRegistryKey.LONG_PUSH_DB_USE_STACKING, 0) == 1;
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return false;
        }
    }

    public static boolean getUseTitleforStacking() {
        try {
            return controller.getDataRegistry().safeGetLong(DataRegistryKey.LONG_PUSH_DB_USE_TITLE_FOR_STACKING, 0) == 1;
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return false;
        }
    }

    public static String getStackingContentTitle() {
        String result;
        try {
            result = controller.getDataRegistry().safeGetString(DataRegistryKey.STRING_PUSH_DB_STACKING_CONTENT_TITLE, null);
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return null;
        }

        return result;
    }

    public static String getStackingContentText() {
        String result;

        try {
            result = controller.getDataRegistry().safeGetString(DataRegistryKey.STRING_PUSH_DB_STACKING_CONTENT_TEXT, null);
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return null;
        }

        return result;
    }

    public static String getStackingBigContentTitle() {
        String result;
        try{
            result = controller.getDataRegistry().safeGetString(DataRegistryKey.STRING_PUSH_DB_STACKING_BIG_CONTENT_TITLE, null);
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return null;
        }

        return result;
    }

    public static String getStackingBigContentSummaryText() {
        String result;

        try {
            result = controller.getDataRegistry().safeGetString(DataRegistryKey.STRING_PUSH_DB_STACKING_BIG_CONTENT_SUMMARY_TEXT, null);
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return null;
        }

        return result;
    }

    public static void setNotificationChannel(String channelName, String channelDescription,
                                              int importance, boolean vibrateEnable) {
        long tempBoolean;

        controller.putDataRegistry(
                new DataUnit(
                        DataRegistryKey.STRING_PUSH_DB_NOTIFICATION_CHANNEL_NAME,
                        channelName,
                        5,
                        ABX_PUSH_COMMON_DAO_CLASS_NAME,
                        true
                ));
        controller.putDataRegistry(
                new DataUnit(
                        DataRegistryKey.STRING_PUSH_DB_NOTIFICATION_CHANNEL_DESCRIPTION,
                        channelDescription,
                        5,
                        ABX_PUSH_COMMON_DAO_CLASS_NAME,
                        true
                ));

        controller.putDataRegistry(
                new DataUnit(
                        DataRegistryKey.LONG_PUSH_DB_IMPORTANCE,
                        (long) importance,
                        5,
                        ABX_PUSH_COMMON_DAO_CLASS_NAME,
                        true
                ));

        tempBoolean = vibrateEnable ? 1 : 0;
        controller.putDataRegistry(
                new DataUnit(
                        DataRegistryKey.LONG_PUSH_DB_NOTIFICATION_CHANNEL_VIBRATE,
                        tempBoolean,
                        5,
                        ABX_PUSH_COMMON_DAO_CLASS_NAME,
                        true
                ));
    }

    public static String getNotificationChannelName() {
        try {
            return controller.getDataRegistry().safeGetString(DataRegistryKey.STRING_PUSH_DB_NOTIFICATION_CHANNEL_NAME, null);
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return null;
        }
    }

    public static String getNotificationChannelDescription() {
        try{
            return controller.getDataRegistry().safeGetString(DataRegistryKey.STRING_PUSH_DB_NOTIFICATION_CHANNEL_DESCRIPTION, null);
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return null;
        }
    }

    public static int getNotificationChannelLight() {
        try {
            return (int) controller.getDataRegistry().safeGetLong(DataRegistryKey.LONG_PUSH_DB_NOTIFICATION_CHANNEL_LIGHT, 1);
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return 1;
        }
    }

    public static int getNotificationChannelVibrate() {
        try {
            return (int) controller.getDataRegistry().safeGetLong(DataRegistryKey.LONG_PUSH_DB_NOTIFICATION_CHANNEL_VIBRATE, 1);
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return 1;
        }
    }

    public static void pushEventTracking(JSONObject pushOpenParam) {
        if (ABXBooleanState.getInstance().isSessionStarted()) {
            LogEventParameter eventParameter = new LogEventParameter(
                    CoreConstants.GROUP_ABX,
                    CoreConstants.EVENT_OPEN_PUSH,
                    CommonUtils.getMapFromJSONObject(CommonUtils.parseValueWithDataType(pushOpenParam, CommonUtils.FIX_TYPE.PREFIX)),
                    0,
                    0
            );

            controller.logEvent(eventParameter);
        }
        else {
            PushOpenMonitor.getInstance().setOpenedByPush(true);
            PushOpenMonitor.getInstance().setPushOpenEventParam(
                    CommonUtils.getMapFromJSONObject(
                            pushOpenParam
                    )
            );
        }
    }

    public static void runInBackGroundInOrder(Runnable runnable){
        controller.runInBackGroundInOrder(runnable);
    }

    public static void runInBackGroundWithoutOrder(Runnable runnable){
        controller.runInBackGroundWithoutOrder(runnable);
    }

    public static void setStopPopUpFlag(boolean stop) {
        if (stop)
            controller.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.LONG_POP_UP_STOP_FLAG,
                            (long) 1,
                            5,
                            ABX_POP_UP_COMMON_DAO_CLASS_NAME,
                            true
                    ));
        else
            controller.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.LONG_POP_UP_STOP_FLAG,
                            (long) 0,
                            5,
                            ABX_POP_UP_COMMON_DAO_CLASS_NAME,
                            true
                    ));
    }

    public static boolean getStopPopUpFlag(){
        long stopPopUpFlag;
        try {
            stopPopUpFlag = controller.getDataRegistry().safeGetLong(DataRegistryKey.LONG_POP_UP_STOP_FLAG, -1);
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return false;
        }
        return stopPopUpFlag == 1;
    }

    public static boolean getPushEnable() {
        try {
            return controller.getDataRegistry().safeGetBoolean(DataRegistryKey.BOOLEAN_IS_PUSH_ENABLE, false);
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return false;
        }
    }

    public static boolean getOsPushEnable() {
        try{
            return controller.getDataRegistry().safeGetBoolean(DataRegistryKey.BOOLEAN_IS_PUSH_ENABLE_OS, false);
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return false;
        }
    }

    public static void updatePushEnable(boolean userPushEnable) {

        controller.putDataRegistry(new DataUnit(
                DataRegistryKey.BOOLEAN_IS_PUSH_ENABLE,
                userPushEnable,
                5,
                PUSH_CONTROLLER_CLASS_NAME,
                true
        ));

        LogEventParameter eventParameter = new LogEventParameter(
                CoreConstants.GROUP_ABX,
                CoreConstants.EVENT_SET_PUSH,
                null,
                0,
                0
        );

        controller.logEvent(eventParameter);
    }

    public static void updateOsPushEnable(boolean userOsPushEnable) {
        try {
            controller.getDataRegistry().putDataRegistry(new DataUnit(
                    DataRegistryKey.BOOLEAN_IS_PUSH_ENABLE_OS,
                    userOsPushEnable,
                    5,
                    PUSH_CONTROLLER_CLASS_NAME,
                    true
            ));
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return;
        }

        LogEventParameter eventParameter = new LogEventParameter(
                CoreConstants.GROUP_ABX,
                CoreConstants.EVENT_SET_PUSH,
                null,
                0,
                0
        );
        controller.logEvent(eventParameter);
        AbxLog.d("CoreWrapper.updateOsPushEnable is called", true);
    }

    public static void setRegistrationID(String userRegistrationId) {
        controller.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_REGISTRATION_ID,
                userRegistrationId,
                5,
                PUSH_CONTROLLER_CLASS_NAME,
                true
        ));

        LogEventParameter eventParameter = new LogEventParameter(
                CoreConstants.GROUP_ABX,
                CoreConstants.EVENT_SET_PUSH,
                null,
                0,
                0
        );

        controller.logEvent(eventParameter);
    }

    public static String getRegistrationID() {
        try {
            return controller.getDataRegistry().safeGetString(DataRegistryKey.STRING_REGISTRATION_ID, null);
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return null;
        }
    }

    public static boolean checkDuplicatedNotification(int notificationId){
        JSONArray storedIds = new JSONArray();
        String IDs;
        try{
            IDs = controller.getDataRegistry().safeGetString(DataRegistryKey.STRING_ABX_NOTIFICATION_ID, null);
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return false;
        }

        if(!CommonUtils.isNullOrEmpty(IDs)) {
            try {
                storedIds = new JSONArray(IDs);
                for(int i=0; i<storedIds.length(); i++){
                    int id = storedIds.getInt(i);
                    if (id != 0 && id == notificationId){
                        return true;
                    }
                }
            } catch (JSONException e) {
                AbxLog.d("Invalid json array", true);
                AbxLog.d(Arrays.toString(e.getStackTrace()), true);
                return false;
            }
        }

        storedIds.put(notificationId);
        controller.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_ABX_NOTIFICATION_ID,
                storedIds.toString(),
                5,
                PUSH_CONTROLLER_CLASS_NAME,
                true
        ));
        return false;
    }

    public static void addObserverToDeferredDeeplinkPostingObservable(IObserver<String> observer){
        DeferredDeeplinkPostingObservable.getInstance().add(observer);
    }

    public static void addObserverToDeeplinkPostingObservable(IObserver<String> observer){
        DeeplinkPostingObservable.getInstance().add(observer);
    }

    public static void addObserverToOsPushEnableObservable(IObserver<Boolean> observer){
        OsPushEnableObservable.getInstance().add(observer);
    }

    public static boolean isGdprForgetMe(){
        return controller.isGdprForgetMe();
    }

    public static boolean isGdprForgetMeSync(){
        return ABXBooleanState.getInstance().isGdprForgetMeSync();
    }

    public static boolean isAdbrixPause(){
        return ABXBooleanState.getInstance().isAdbrixPause();
    }

    public static boolean isSdkInitialized(){
        return ABXBooleanState.getInstance().isSdkInitialized();
    }

    public static boolean inForeground(){
        return ABXBooleanState.getInstance().inForeground();
    }

    public static void setDevState(Context context){
        CoreConstants.setDevState(CoreUtils.isDevState(context));
    }

    public static void setStageState(Context context) {
        CoreConstants.setStageState(CoreUtils.isStageState(context));
    }

    public static int getPropertyMaxSize(){
        return S3ConfigHandler.config_propertyMaxSize;
    }

    public static Uri getDeeplinkUriExceptAdbrixParameter(Uri uri) {
        try {
            return CoreUtils.getDeeplinkUriExceptAdbrixParameter(uri, controller.getDataRegistry());
        }catch (NullPointerException nullPointerException){
            AbxLog.d("getDataRegistry is called before controller.startcontroller", true);
            return null;
        }
    }

    public static void deleteUserDataAndStopSDK(String userId, Runnable onSuccess, Runnable onFail) {
        controller.deleteUserDataAndStopSDK(userId, onSuccess, onFail);
    }

    public static void restartSDK(String userId, Runnable onSuccess, Runnable onFail) {
        controller.restartSDK(userId, onSuccess, onFail);
    }

    public static String getSDKVersion() {
        return SdkVersion.SDK_VERSION;
    }
}
