package io.adbrix.sdk.configuration;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.IABXComponentsFactory;
import io.adbrix.sdk.component.autoEvent.DailyFirstOpenAutoEventGenerator;
import io.adbrix.sdk.component.autoEvent.DeeplinkAutoEventGenerator;
import io.adbrix.sdk.component.autoEvent.FirstOpenAutoEventGenerator;
import io.adbrix.sdk.component.autoEvent.SessionAutoEventGenerator;
import io.adbrix.sdk.data.ABXBooleanState;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.net.ApiConnectionManager;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.LogMessageFormat;
import io.adbrix.sdk.domain.Repository;
import io.adbrix.sdk.domain.interactor.DeleteUserDataUseCase;
import io.adbrix.sdk.domain.interactor.FlushAllNowUseCase;
import io.adbrix.sdk.domain.interactor.FlushAtIntervalsUseCase;
import io.adbrix.sdk.domain.interactor.LogEventUseCase;
import io.adbrix.sdk.domain.interactor.RegisterNetworkCallbackUseCase;
import io.adbrix.sdk.domain.interactor.RestartSDKUseCase;
import io.adbrix.sdk.domain.interactor.SaveUnsentEventsUseCase;
import io.adbrix.sdk.domain.interactor.SaveUserPropertyUseCaseWithoutEvent;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.domain.model.UserPropertyCommand;
import io.adbrix.sdk.utils.CoreUtils;

public class ActiveABXContext implements IABXContext {
    private DataRegistry dataRegistry;
    private SessionAutoEventGenerator sessionAutoEventGenerator;
    private DeeplinkAutoEventGenerator deeplinkAutoEventGenerator;
    private FirstOpenAutoEventGenerator firstOpenAutoEventGenerator;
    private DailyFirstOpenAutoEventGenerator dailyFirstOpenAutoEventGenerator;
    private Context androidContext;
    private Repository repository;


    public ActiveABXContext(
            IABXComponentsFactory componentsFactory,
            Repository repository
    ) throws IABXComponentsFactory.ComponentsCanNotCreateException {
        this.dataRegistry = componentsFactory.createOrGetDataRegistry();
        this.androidContext = componentsFactory.getAndroidContext();
        this.repository = repository;

        this.firstOpenAutoEventGenerator = new FirstOpenAutoEventGenerator(dataRegistry ,androidContext);
        this.sessionAutoEventGenerator = new SessionAutoEventGenerator(this,dataRegistry,firstOpenAutoEventGenerator);
        this.deeplinkAutoEventGenerator = new DeeplinkAutoEventGenerator(this, dataRegistry);
        this.dailyFirstOpenAutoEventGenerator = new DailyFirstOpenAutoEventGenerator(this, dataRegistry, repository);

        if (CoreUtils.isOnline(androidContext)) {
            new FlushAllNowUseCase(repository).execute();
        }
    }

    @Override
    public void initialize(Context context, String appkey, String secretkey) {
        AbxLog.w(String.format(LogMessageFormat.INVALID_CONTEXT,"Active","initialize"), true);
    }

    @Override
    public void saveUserProperty(UserPropertyCommand userPropertyCommand) {

        if(saveUserPropertyWithoutEvent(userPropertyCommand)) {
            LogEventParameter eventParameter = new LogEventParameter(
                    CoreConstants.GROUP_ABX,
                    CoreConstants.EVENT_USER_PROPERTY_CHANGED,
                    null,
                    0,
                    0
            );

            logEvent(eventParameter);
        }
    }

    @Override
    public Boolean saveUserPropertyWithoutEvent(UserPropertyCommand userPropertyCommand) {
        return new SaveUserPropertyUseCaseWithoutEvent(repository).execute(userPropertyCommand);
    }

    @Override
    public void logEvent(LogEventParameter logEventParameter) {
        new LogEventUseCase(repository).execute(logEventParameter);
    }

    @Override
    public void runOnIdleTime() {
        new FlushAtIntervalsUseCase(repository).execute();
    }

    @Override
    public void onResume(Activity activity) {
        this.sessionAutoEventGenerator.onResume(activity);

        if(ABXBooleanState.getInstance().isFirstOnResume()){
            dailyFirstOpenAutoEventGenerator.process();
//            new UpdateFavoriteAppUseCase(repository).execute();
        }

        ABXBooleanState.getInstance().isFirstOnResume.getAndSet(false);
    }

    @Override
    public void onPause() {
        this.sessionAutoEventGenerator.onPause();

        if (CoreUtils.isOnline(androidContext)) {
            new FlushAllNowUseCase(repository).execute();
        }

        new SaveUnsentEventsUseCase(repository).execute();

        dataRegistry.saveRegistry();
    }

    @Override
    public void deeplink(Activity deeplinkActivity) {
        if(sessionAutoEventGenerator.isStartSessionByDeeplink(deeplinkActivity)){
            return;
        }
        this.deeplinkAutoEventGenerator.process(deeplinkActivity);
    }

    @Override
    public void deeplinkWithIntent(Intent deeplinkIntent) {
        if (sessionAutoEventGenerator.isStartSessionByDeeplinkIntent(deeplinkIntent)) {
            return;
        }
        this.deeplinkAutoEventGenerator.process(deeplinkIntent);
    }

    @Override
    public void putDataRegistry(DataUnit dataUnit) {
        dataRegistry.putDataRegistry(dataUnit);
    }

    @Override
    public void registerNetworkCallback() {
        new RegisterNetworkCallbackUseCase(repository).execute();
    }

    @Override
    public void deleteUserData(String userId, ApiConnectionManager.Result result) {
        AbxLog.d("Delete user data api called!", true);

        new DeleteUserDataUseCase(repository, result).execute(userId);
    }

    @Override
    public void restartSDK(String userId, ApiConnectionManager.Result result) {
        AbxLog.d("Restart SDK api called!", true);

        new RestartSDKUseCase(repository, result).execute(userId);
    }
}
