package io.adbrix.sdk.component.autoEvent;

import android.app.Activity;
import android.content.Intent;
import android.net.Uri;

import java.util.HashMap;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.DeeplinkPostingObservable;
import io.adbrix.sdk.component.PushOpenMonitor;
import io.adbrix.sdk.configuration.DefaultABXContextController;
import io.adbrix.sdk.configuration.IABXContext;
import io.adbrix.sdk.data.ABXBooleanState;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.CoreUtils;

public class SessionAutoEventGenerator {
    private IABXContext abxContext;
    private DataRegistry dataRegistry;
    private FirstOpenAutoEventGenerator firstOpenAutoEventGenerator;

    private String lastCreatedSessionId;
    private String lastEndSessionId;
    private long lastSessionCreatedTime;
    private long lastSessionEndTime;
    private boolean isEndSessionSentLast;

    private Timer timer;

    public SessionAutoEventGenerator(IABXContext abxContext, DataRegistry dataRegistry, FirstOpenAutoEventGenerator firstOpenAutoEventGenerator) {
        this.abxContext = abxContext;
        this.dataRegistry = dataRegistry;
        this.firstOpenAutoEventGenerator = firstOpenAutoEventGenerator;

        lastCreatedSessionId =  dataRegistry.safeGetString( DataRegistryKey.STRING_LAST_CREATED_SESSION_ID,     null);
        lastEndSessionId = dataRegistry.safeGetString(      DataRegistryKey.STRING_LAST_END_SESSION_ID,         null);
        lastSessionCreatedTime = dataRegistry.safeGetLong(  DataRegistryKey.LONG_LAST_SESSION_CREATED_TIME,     0);
        lastSessionEndTime = dataRegistry.safeGetLong(      DataRegistryKey.LONG_LAST_SESSION_END_TIME,         0);
        isEndSessionSentLast = dataRegistry.safeGetBoolean( DataRegistryKey.BOOLEAN_IS_END_SESSION_SENT_LAST,   false);

        timer = null;
    }

    public void onResume(final Activity activity) {
        if (timer != null){
            // Cancel the app background event from being sent,
            // it hasn't been enough time between previous Session end and this start
            cancelEndSessionSchedule();
            return;
        }

        if(ABXBooleanState.getInstance().isSessionStarted())
            return;

        if(CommonUtils.isNullOrEmpty(lastCreatedSessionId) == false && // First StartSession
                lastCreatedSessionId != lastEndSessionId) {
            LogEventParameter eventParameter = new LogEventParameter(
                    CoreConstants.GROUP_ABX,
                    CoreConstants.EVENT_END_SESSION,
                    null,
                    0,
                    getSessionLength()
            );

            processEndSession(eventParameter);
        }

        if (firstOpenAutoEventGenerator.isTrigger()){
            firstOpenAutoEventGenerator.getGooglePlayReferrerPropertiesOrNull(mapParam ->
                    DefaultABXContextController.getInstance().processFirstOpenStartSession(() -> {
                        AbxLog.d("FirstOpen start", true);

                        processStartSession("first_open", mapParam);

                        AbxLog.d("FirstOpen finished successfully", true);
                        firstOpenAutoEventGenerator.runDeferredDeeplinkProcess();

                        CoreUtils.clearAdbrixTrackingParamFromCurrentActivity(activity, dataRegistry);
            }));
        }
        else if (CoreUtils.isActivityStartedByDeeplink(activity, dataRegistry)){
            Uri uri = CoreUtils.getDeeplinkUriFromCurrentActivity(activity);

            if(uri == null){
                AbxLog.d( "deeplink uri is null", true);
                processStartSession("basic_open", null);
                return;
            }

            String deeplinkUrl = uri.toString();

            Map<String, Object> deeplinkEventParam = CoreUtils.getDeeplinkParameterFromUriString(deeplinkUrl);

            processStartSession("deeplink_open", deeplinkEventParam);

            DeeplinkPostingObservable.getInstance().notifyObserver(deeplinkUrl);

            CoreUtils.clearAdbrixTrackingParamFromCurrentActivity(activity, dataRegistry);
        }
        else if (PushOpenMonitor.getInstance().isOpenedByPush()){
            Map<String, Object> pushOpenEventParam = PushOpenMonitor.getInstance().getPushOpenEventParam();

            processStartSession("push_open", pushOpenEventParam);

            PushOpenMonitor.getInstance().refresh();
        }
        else {
            processStartSession("basic_open", null);
        }
    }

    // Schedule timer to send an end_session event after SESSION_TIMEOUT ms have passed
    public void onPause(){
        cancelEndSessionSchedule();

        updateLastSessionEndTime(System.currentTimeMillis());

        LogEventParameter eventParameter = new LogEventParameter(
                CoreConstants.GROUP_ABX,
                CoreConstants.EVENT_END_SESSION,
                null,
                0,
                getSessionLength(),
                CommonUtils.getCurrentUTCInDBFormat()
        );

        timer = new Timer();
        timer.schedule(new TimerTask() {
            @Override
            public void run() {
                // Null out the timer so we know it's finished running
                if(timer != null)
                {
                    timer = null;
                    processEndSession(eventParameter);
                    dataRegistry.saveRegistry();
                }
            }
        }, CoreConstants.SESSION_TIMEOUT);
    }

    public void ping() {
        // updateElapsedTimeInForeground(SystemClock.elapsedRealtime());
    }

    public boolean isStartSessionByDeeplink(Activity deeplinkActivity){
        return CoreUtils.isActivityStartedByDeeplink(deeplinkActivity, dataRegistry) && !ABXBooleanState.getInstance().isSessionStarted();
    }

    public boolean isStartSessionByDeeplinkIntent(Intent deeplinkIntent){
        return CoreUtils.isStartedByDeeplinkIntent(deeplinkIntent, dataRegistry) && !ABXBooleanState.getInstance().isSessionStarted();
    }

    //Used for Start Session Event only
    private long getSessionInterval() {
        return lastSessionEndTime == 0 ? 0 : System.currentTimeMillis() - lastSessionEndTime;
    }

    //Used for EndSession Event only
    private long getSessionLength() {
        return lastSessionEndTime - lastSessionCreatedTime;
    }

    private void processStartSession(String openType, Map<String, Object> eventParam) {

        if (ABXBooleanState.getInstance().isSessionStarted()) {
            return;
        }

        updateLastCreatedSessionId(CommonUtils.randomUUIDWithCurrentTime());
        updateLastSessionCreatedTime(System.currentTimeMillis());

        if (eventParam == null){
            eventParam = new HashMap<>();
        }
        eventParam.put("abx:open_type", openType);

        LogEventParameter eventParameter = new LogEventParameter(
                CoreConstants.GROUP_ABX,
                CoreConstants.EVENT_START_SESSION,
                CommonUtils.parseValueWithDataType(eventParam, CommonUtils.FIX_TYPE.PREFIX),
                getSessionInterval(),
                0
        );

        if (openType.equals("first_open")) {
            //DataRegistry에 last_first_open_event_id 설정
            abxContext.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.STRING_LAST_FIRSTOPEN_ID,
                            eventParameter.eventId,
                            5,
                            this.getClass().getName(),
                            true
                    ));

            //DataRegistry에 last_open_id 설정
            abxContext.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.STRING_LAST_OPEN_ID,
                            eventParameter.eventId,
                            5,
                            getClass().getName(),
                            true
                    ));
        }
        else if (openType.equals("deeplink_open")){
            abxContext.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.STRING_LAST_DEEPLINK_ID,
                            eventParameter.eventId,
                            5,
                            this.getClass().getName(),
                            true
                    ));

            abxContext.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.STRING_LAST_OPEN_ID,
                            eventParameter.eventId,
                            5,
                            this.getClass().getName(),
                            true
                    ));
        }

        abxContext.logEvent(eventParameter);
        ABXBooleanState.getInstance().isSessionStarted.getAndSet(true);

        isEndSessionSentLast = false;

        abxContext.putDataRegistry(new DataUnit(
                DataRegistryKey.BOOLEAN_IS_END_SESSION_SENT_LAST,
                false,
                5,
                this.getClass().getName(),
                true
        ));
    }

    private void processEndSession(LogEventParameter eventParameter) {
        if (isEndSessionSentLast)
            return;

        updateLastEndSessionId(lastCreatedSessionId);

        abxContext.logEvent(eventParameter);
        ABXBooleanState.getInstance().isSessionStarted.getAndSet(false);

        isEndSessionSentLast = true;

        abxContext.putDataRegistry(new DataUnit(
                DataRegistryKey.BOOLEAN_IS_END_SESSION_SENT_LAST,
                true,
                5,
                this.getClass().getName(),
                true
        ));
    }

    public void cancelEndSessionSchedule() {
        if(timer != null)
        {
            timer.cancel();
            timer = null;
        }
    }

    private void updateLastCreatedSessionId(String id) {
        lastCreatedSessionId = id;
        abxContext.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_LAST_CREATED_SESSION_ID,
                id,
                5,
                this.getClass().getName(),
                true));
    }

    private void updateLastEndSessionId(String id) {
        lastEndSessionId = id;
        abxContext.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_LAST_END_SESSION_ID,
                id,
                5,
                this.getClass().getName(),
                true));
    }

    private void updateLastSessionCreatedTime(long time) {
        lastSessionCreatedTime = time;
        abxContext.putDataRegistry(new DataUnit(
                DataRegistryKey.LONG_LAST_SESSION_CREATED_TIME,
                time,
                5,
                this.getClass().getName(),
                true));
    }

    private void updateLastSessionEndTime(long time) {
        lastSessionEndTime = time;
        abxContext.putDataRegistry(new DataUnit(
                DataRegistryKey.LONG_LAST_SESSION_END_TIME,
                time,
                5,
                this.getClass().getName(),
                true));
    }
}
