package io.adbrix.sdk.utils;

import android.app.Activity;
import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.net.ConnectivityManager;
import android.net.Network;
import android.net.NetworkCapabilities;
import android.net.NetworkInfo;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;

import org.json.JSONException;
import org.json.JSONObject;

import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.data.ABXBooleanState;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;

public class CoreUtils {

    private CoreUtils(){}

    public static SimpleDateFormat createDateFormat(String pattern) {
        SimpleDateFormat dateFormat = new SimpleDateFormat(pattern, Locale.KOREA);
        dateFormat.setTimeZone(TimeZone.getTimeZone("GMT+0"));
        return dateFormat;
    }

    public static String getDateViaGMTDateFormat(String format) {
        DateFormat date = new SimpleDateFormat("z", Locale.getDefault());
        String gmt = date.format(System.currentTimeMillis());
        SimpleDateFormat utc_df = new SimpleDateFormat(format);
        utc_df.setTimeZone(TimeZone.getTimeZone(gmt));
        return utc_df.format(new Date());
    }

    // HMAC-SHA256 Algorithm을 이용한 결과값을 반환한다. Exception 발생 시, Empty String을 반환한다.
    public static String HMAC_SHA256(String key, String message) {
        try {
            String algorithm = "HmacSHA256";
            Mac mac = Mac.getInstance(algorithm);
            SecretKeySpec secretKey = new SecretKeySpec(key.getBytes(), algorithm);
            mac.init(secretKey);
            byte[] hash = mac.doFinal(message.getBytes());

            StringBuffer buffer = new StringBuffer();
            for(int i = 0; i < hash.length; i++)
            {
                int d = hash[i];

                if(d < 0)
                {
                    d = d + 0x100;
                }

                if (d < 16)
                {
                    buffer.append("0");
                }

                buffer.append(Integer.toString(d, 0x10));
            }

            return buffer.toString();
        } catch (NoSuchAlgorithmException | InvalidKeyException e) {
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
        }

        return "";
    }

    public static String md5(String s) {
        String MD5 = "";

        try{

            MessageDigest md = MessageDigest.getInstance("MD5");
            md.update(s.getBytes());
            byte byteData[] = md.digest();

            StringBuffer sb = new StringBuffer();

            for(int i = 0 ; i < byteData.length ; i++){
                sb.append(Integer.toString((byteData[i]&0xff) + 0x100, 16).substring(1));
            }

            MD5 = sb.toString();
        }catch(NoSuchAlgorithmException e){
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
            MD5 = null;
        }
        return MD5;
    }

    public static Object convertNullableToJSONNullable(Object object) {
        return object == null ? JSONObject.NULL : object;
    }

    public static JSONObject getJSONObjectFromMap(Map<String, Object> map ) {
        JSONObject jsonObject = new JSONObject();
        try {
            for (Map.Entry<String, Object> entry : map.entrySet()) {
                String key = entry.getKey();
                Object value = entry.getValue();
                jsonObject.put(key, value);
            }
        }
        catch (JSONException e){
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
        }

        return jsonObject;
    }

    public static String getValueOfJsonObject(String uriStr, String key) {
        try {
            JSONObject myObj = new JSONObject(uriStr);
            if (CommonUtils.isNullOrEmpty(key) || myObj.isNull(key)) return null;
            else return String.valueOf(myObj.get(key));
        } catch (JSONException e) {
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
            return "JsonException error : " + e.getMessage();
        }
    }

    public static String get02XMD5(String s) {
        try {
            // Create MD5 Hash
            MessageDigest digest = java.security.MessageDigest.getInstance("MD5");
            digest.update(s.getBytes());
            byte messageDigest[] = digest.digest();

            // Create Hex String
            StringBuffer hexString = new StringBuffer();
            for (int i = 0; i < messageDigest.length; i++)
            {
                hexString.append(String.format("%02X", 0xFF & messageDigest[i]));
            }

            String temp = hexString.toString();
            return temp.toLowerCase();
        } catch (NoSuchAlgorithmException e) {
            AbxLog.w( "md5 error: " + e.getMessage(), true);
        }
        return s;
    }

    public static synchronized boolean isOnline(Context context) {
        ConnectivityManager connectivityManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);

        if(Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            Network network = connectivityManager.getActiveNetwork();
            if (network != null) {
                NetworkCapabilities networkCapabilities = connectivityManager.getNetworkCapabilities(network);
                if(networkCapabilities != null) {
                    return networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI) || networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR);
                }
            }
            return false;
        }
        else {
            NetworkInfo activeNetworkInfo = connectivityManager.getActiveNetworkInfo();
            return activeNetworkInfo != null && activeNetworkInfo.isConnected();
        }
    }

    public static boolean canConnectBackend(Context context) {

        boolean retBool = false;

        if (context == null) {
            throw new IllegalStateException("ConnectBackend: Can not use null context!");
        }

        if(isOnline(context)) {
            if(ABXBooleanState.getInstance().isSdkInitialized()) {
                retBool = true;
            }
            else {
                AbxLog.e( "canConnectBackend :: cannot use :: SDK is not initialized yet", true);
            }
        }
        else {
            AbxLog.e( "canConnectBackend :: Network Connection is not online", true);
        }

        return retBool;
    }

    // Get immutable URI
    public static Uri getDeeplinkUriFromCurrentActivity(Activity activity) {
        if(activity != null && activity.getIntent() != null && activity.getIntent().getData() != null){

            Uri iUri =  activity.getIntent().getData(); // Immutable URI reference
            iUri = Uri.parse(Uri.decode(iUri.toString()));

            return iUri;
        }
        else return null;
    }

    public static void clearAdbrixTrackingParamFromCurrentActivity(Activity activity, DataRegistry dataRegistry) {
        if(activity != null && activity.getIntent() != null && activity.getIntent().getData() != null) {

            Uri iUri = activity.getIntent().getData(); // Immutable URI reference
            iUri = Uri.parse(Uri.decode(iUri.toString()));

            if (iUri.toString().indexOf('#') != -1) iUri = Uri.parse(iUri.toString().replaceAll("\\#", "%32"));
            String uriString = activity.getIntent().getDataString();
            uriString = Uri.decode(uriString);

            if (iUri != null && iUri.isHierarchical()) {
                if (iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TID) != null) {
                    String paramString = CoreConstants.DEEPLINK_ABX_TID + "=" + iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TID);
                    if (iUri.getQuery().length() == paramString.length()) {
                        paramString = "\\?" + paramString;
                    } else if (uriString != null && (uriString.length() - paramString.length()) == uriString.indexOf(paramString)) {
                        paramString = "&" + paramString;
                    } else {
                        paramString = paramString + "&";
                    }
                    if (!CommonUtils.isNullOrEmpty(uriString)) {
                        uriString = uriString.replaceFirst(paramString, "");
                    }
                }
                else {
                    AbxLog.d( "abx_tid doesn't exist", true);
                }

                if (iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TRACKER_ID) != null) {
                    String paramString = CoreConstants.DEEPLINK_ABX_TRACKER_ID + "=" + iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TRACKER_ID);
                    if (iUri.getQuery().length() == paramString.length()) {
                        paramString = "\\?" + paramString;
                    } else if (uriString != null && (uriString.length() - paramString.length()) == uriString.indexOf(paramString)) {
                        paramString = "&" + paramString;
                    } else {
                        paramString = paramString + "&";
                    }
                    if (!CommonUtils.isNullOrEmpty(uriString)) {
                        uriString = uriString.replaceFirst(paramString, "");
                    }
                }
                else {
                    AbxLog.d("abx_tracker_id doesn't exist", true);
                }

                String appkey = dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY,null);

                if ((iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TID) != null) ||
                        (iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TRACKER_ID) != null) ||
                        (iUri.toString().toLowerCase().startsWith("https://" + appkey.toLowerCase() + CoreConstants.DEEPLINK_LIVE_URL)) ||
                        (iUri.toString().toLowerCase().startsWith("https://" + appkey.toLowerCase() + CoreConstants.DEEPLINK_STAGE_URL)) ||
                        (iUri.toString().toLowerCase().startsWith("https://" + appkey.toLowerCase() + CoreConstants.DEEPLINK_DEV_URL)))
                {
                    Uri newData = Uri.parse(uriString);
                    activity.getIntent().setData(newData);
                }
                else {
                    AbxLog.e( "Check Adbrix Parameters :: There is no Deeplink information in Activity!", true);
                }
            }
        }
    }

    public static Uri getDeeplinkUriExceptAdbrixParameter(Uri deeplinkUri, DataRegistry dataRegistry) {
        Uri iUri = Uri.parse(Uri.decode(deeplinkUri.toString()));

        if (iUri.toString().indexOf('#') != -1) iUri = Uri.parse(iUri.toString().replaceAll("\\#", "%32"));
        String uriString = deeplinkUri.toString();
        uriString = Uri.decode(uriString);

        if (iUri != null && iUri.isHierarchical()) {
            if (iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TID) != null) {
                String paramString = CoreConstants.DEEPLINK_ABX_TID + "=" + iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TID);
                if (iUri.getQuery().length() == paramString.length()) {
                    paramString = "\\?" + paramString;
                } else if (uriString != null && (uriString.length() - paramString.length()) == uriString.indexOf(paramString)) {
                    paramString = "&" + paramString;
                } else {
                    paramString = paramString + "&";
                }
                if (!CommonUtils.isNullOrEmpty(uriString)) {
                    uriString = uriString.replaceFirst(paramString, "");
                }
            }
            else {
                AbxLog.d("abx_tid doesn't exist", true);
            }

            if (iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TRACKER_ID) != null) {
                String paramString = CoreConstants.DEEPLINK_ABX_TRACKER_ID + "=" + iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TRACKER_ID);
                if (iUri.getQuery().length() == paramString.length()) {
                    paramString = "\\?" + paramString;
                } else if (uriString != null && (uriString.length() - paramString.length()) == uriString.indexOf(paramString)) {
                    paramString = "&" + paramString;
                } else {
                    paramString = paramString + "&";
                }
                if (!CommonUtils.isNullOrEmpty(uriString)) {
                    uriString = uriString.replaceFirst(paramString, "");
                }
            }
            else {
                AbxLog.d("abx_tracker_id doesn't exist", true);
            }

            String appkey = dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY,null);

            if ((iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TID) != null) ||
                    (iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TRACKER_ID) != null) ||
                    (iUri.toString().toLowerCase().startsWith("https://" + appkey.toLowerCase() + CoreConstants.DEEPLINK_LIVE_URL)) ||
                    (iUri.toString().toLowerCase().startsWith("https://" + appkey.toLowerCase() + CoreConstants.DEEPLINK_STAGE_URL)) ||
                    (iUri.toString().toLowerCase().startsWith("https://" + appkey.toLowerCase() + CoreConstants.DEEPLINK_DEV_URL)))
            {
                return Uri.parse(uriString);
            }
            else {
                AbxLog.e("Check Adbrix Parameters :: There is no Deeplink information in Activity!", true);
                return Uri.parse(uriString);
            }
        }
        return Uri.parse(uriString);
    }

    public static Map<String, Object> getDeeplinkParameterFromUriString(String encodedString) {
        String uriString = Uri.decode(encodedString);
        Uri uri = Uri.parse(uriString);

        if (uri == null || uri.isHierarchical() == false || uri.toString().contains("?") == false)
        {
            // ERROR :: Uri is null or parameter doesn't exist in ...
            AbxLog.e("ERROR :: Uri is null or parameter doesn't exist in ...", true);
            return new HashMap<>();
        }

        if (uri.toString().contains("#"))
        {
            uri = Uri.parse(uri.toString().replaceAll("\\#", "%32"));
        }

        Map<String, Object> parameter = new HashMap<>();
        parameter.put(CoreConstants.DEEPLINK_PAYLOAD, uri.toString());

        String abx_tid = uri.getQueryParameter("abx_tid");
        String abx_tracker_id = uri.getQueryParameter("abx_tracker_id");

        if (!CommonUtils.isNullOrEmpty(abx_tid))
        {
            parameter.put(CoreConstants.DEEPLINK_ABX_ADKEY, abx_tid);
        }
        if (!CommonUtils.isNullOrEmpty(abx_tracker_id))
        {
            parameter.put(CoreConstants.DEEPLINK_ABX_TRACKER_ID, abx_tracker_id);
        }

        return parameter;
    }

    public static boolean isActivityStartedByDeeplink(Activity activity, DataRegistry dataRegistry) {
        if(activity != null && activity.getIntent() != null && activity.getIntent().getData() != null){
            Uri iUri =  activity.getIntent().getData(); // Immutable URI reference
            if (iUri != null && iUri.isHierarchical()) {

                String appkey = dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY, null);

                return (iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TID) != null) ||
                        (iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TRACKER_ID) != null) ||
                        (iUri.toString().toLowerCase().startsWith("https://" + appkey.toLowerCase() + CoreConstants.DEEPLINK_LIVE_URL)) ||
                        (iUri.toString().toLowerCase().startsWith("https://" + appkey.toLowerCase() + CoreConstants.DEEPLINK_STAGE_URL)) ||
                        (iUri.toString().toLowerCase().startsWith("https://" + appkey.toLowerCase() + CoreConstants.DEEPLINK_DEV_URL));
            }
        }
        return false;
    }

    public static boolean isDevState(Context context) {
        try {
            ApplicationInfo appInfo;
            appInfo = context.getPackageManager().getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA);
            Bundle bundle = appInfo.metaData;
            return bundle.getBoolean("19327da43293b1dce56da3a83829ccaf434bb68ef83ab61a2dca61d3bb15f2be");
        } catch (PackageManager.NameNotFoundException e1) {
            return false;
        } catch (Exception e2) {
            return false;
        }
    }

    public static boolean isStageState(Context context) {
        try {
            ApplicationInfo appInfo;
            appInfo = context.getPackageManager().getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA);
            Bundle bundle = appInfo.metaData;
            return bundle.getBoolean("AdBrixRmStageMode");
        } catch (PackageManager.NameNotFoundException e1) {
            return false;
        } catch (Exception e2) {
            return false;
        }
    }
}
