package io.adbrix.sdk.data.modelprovider;

import android.content.Context;

import org.json.JSONException;
import org.json.JSONObject;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.IABXComponentsFactory;
import io.adbrix.sdk.data.dataprovider.DeviceRealtimeDataProvider;
import io.adbrix.sdk.data.dataprovider.DeviceStaticDataProvider;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.data.repository.UserPropertyManager;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.CommonModel;
import io.adbrix.sdk.domain.model.DRModel;
import io.adbrix.sdk.domain.model.EventModel;
import io.adbrix.sdk.domain.model.IApiModel;
import io.adbrix.sdk.domain.model.IdentityModel;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.utils.CommonUtils;

public class DRModelProvider implements IApiModelProvider {
    private Context context;
    private DeviceRealtimeDataProvider deviceRealtimeDataProvider;
    private DeviceStaticDataProvider deviceStaticDataProvider;
    private UserPropertyManager userPropertyManager;
    private DataRegistry dataRegistry;
    private String userId;
    private DRModel.OperationType operationType;

    public DRModelProvider(
            IABXComponentsFactory factory,
            String userId,
            DRModel.OperationType operationType
    ) {
        try {
            this.context = factory.getAndroidContext();
            this.deviceRealtimeDataProvider = factory.createOrGetDeviceRealtimeDataProvider();
            this.deviceStaticDataProvider = factory.createOrGetDeviceStaticDataProvider();
            this.userPropertyManager = factory.createOrGetUserPropertyManager();
            this.dataRegistry = factory.createOrGetDataRegistry();
        } catch (IABXComponentsFactory.ComponentsCanNotCreateException e) {
            AbxLog.e("DeleteRestartModelProvider:: ComponentsCanNotCreateException!", true);
        }

        this.userId = userId;
        this.operationType = operationType;
    }

    @Override
    public IApiModel provide() {
        deviceStaticDataProvider.provideDefaultValues();
        deviceRealtimeDataProvider.refreshData();

        IdentityModel identityModel = new IdentityModel(
                this.dataRegistry.safeGetString(    DataRegistryKey.STRING_ADID,                null),
                this.dataRegistry.safeGetString(    DataRegistryKey.STRING_GAID,                null),
                this.dataRegistry.safeGetString(    DataRegistryKey.STRING_UUID,                null),
                this.dataRegistry.safeGetString(    DataRegistryKey.STRING_IDFA,                null),
                this.dataRegistry.safeGetString(    DataRegistryKey.STRING_IDFV,                null),
                this.dataRegistry.safeGetString(    DataRegistryKey.STRING_IGAW_ID,             null),
                this.dataRegistry.safeGetBoolean(   DataRegistryKey.BOOLEAN_AD_ID_OPT_OUT,      false),
                this.dataRegistry.safeGetString(    DataRegistryKey.STRING_DEVICE_ID,           null),
                this.dataRegistry.safeGetString(    DataRegistryKey.STRING_REGISTRATION_ID,     null),
                this.dataRegistry.safeGetBoolean(   DataRegistryKey.BOOLEAN_IS_PUSH_ENABLE,     false),
                false
        );

        JSONObject req = new JSONObject();

        if (operationType == DRModel.OperationType.INITIALIZE) {
            try {
                JSONObject evt = new JSONObject();

                String eventDatetime = CommonUtils.getCurrentUTCInDBFormat();
                String lastFirstOpenId = CommonUtils.randomUUIDWithCurrentTime();

                //DataRegistry에 Init Restart Event Datetime 저장. Init Restart Not Synced 상황에서 사용 예정.
                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.STRING_INIT_RESTART_EVENT_DATETIME,
                                eventDatetime,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );

                //DataRegistry에 last_first_open_event_id 설정
                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.STRING_LAST_FIRSTOPEN_ID,
                                lastFirstOpenId,
                                5,
                                this.getClass().getName(),
                                true
                        ));

                //DataRegistry에 last_open_id 설정
                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.STRING_LAST_OPEN_ID,
                                lastFirstOpenId,
                                5,
                                getClass().getName(),
                                true
                        ));

                evt.put("log_id", lastFirstOpenId);
                evt.put("event_datetime", eventDatetime);

                LogEventParameter logEventParameter = new LogEventParameter(
                        lastFirstOpenId,
                        null,
                        CoreConstants.GROUP_ABX,
                        "",
                        null,
                        0,
                        0,
                        eventDatetime
                );

                CommonModelProvider commonModelProvider = new CommonModelProvider(dataRegistry, context, deviceRealtimeDataProvider);
                CommonModel commonModel = (CommonModel) commonModelProvider.provide();

                EventModelProvider eventModelProvider = new EventModelProvider(userPropertyManager, dataRegistry);
                eventModelProvider.setLogEventParameter(logEventParameter);
                EventModel eventModel = (EventModel) eventModelProvider.provide();

                req.put("common", commonModel.getJson());
                req.put("evt", eventModel.getJson());

                dataRegistry.putDataRegistry(
                        new DataUnit(DataRegistryKey.STRING_PREV_ID,
                                lastFirstOpenId,
                                5,
                                this.getClass().getName(),
                                true));

                return new DRModel(identityModel, userId, operationType, eventDatetime, req);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        return new DRModel(identityModel, userId, operationType, req);
    }

    public IApiModel getNotSyncedInitRestartModel() {
        deviceStaticDataProvider.provideDefaultValues();
        deviceRealtimeDataProvider.refreshData();

        IdentityModel identityModel = new IdentityModel(
                this.dataRegistry.safeGetString(    DataRegistryKey.STRING_ADID,                null),
                this.dataRegistry.safeGetString(    DataRegistryKey.STRING_GAID,                null),
                this.dataRegistry.safeGetString(    DataRegistryKey.STRING_UUID,                null),
                this.dataRegistry.safeGetString(    DataRegistryKey.STRING_IDFA,                null),
                this.dataRegistry.safeGetString(    DataRegistryKey.STRING_IDFV,                null),
                this.dataRegistry.safeGetString(    DataRegistryKey.STRING_IGAW_ID,             null),
                this.dataRegistry.safeGetBoolean(   DataRegistryKey.BOOLEAN_AD_ID_OPT_OUT,      false),
                this.dataRegistry.safeGetString(    DataRegistryKey.STRING_DEVICE_ID,           null),
                this.dataRegistry.safeGetString(    DataRegistryKey.STRING_REGISTRATION_ID,     null),
                this.dataRegistry.safeGetBoolean(   DataRegistryKey.BOOLEAN_IS_PUSH_ENABLE,     false),
                false
        );

        String eventDatetime = getNotSyncedInitRestartDatetime();
        String lastFirstOpenId = getNotSyncedInitRestartLastFirstOpenId();

        //reset prevId. If not reset prev id, last first open id equals prev id
        dataRegistry.putDataRegistry(
                new DataUnit(DataRegistryKey.STRING_PREV_ID,
                        null,
                        5,
                        this.getClass().getName(),
                        true));

        JSONObject req = new JSONObject();
        JSONObject evt = new JSONObject();

        try {
            evt.put("log_id", lastFirstOpenId);
            evt.put("event_datetime", eventDatetime);

            LogEventParameter logEventParameter = new LogEventParameter(
                    lastFirstOpenId,
                    null,
                    CoreConstants.GROUP_ABX,
                    "",
                    null,
                    0,
                    0,
                    eventDatetime
            );

            CommonModelProvider commonModelProvider = new CommonModelProvider(dataRegistry, context, deviceRealtimeDataProvider);
            CommonModel commonModel = (CommonModel) commonModelProvider.provide();

            EventModelProvider eventModelProvider = new EventModelProvider(userPropertyManager, dataRegistry);
            eventModelProvider.setLogEventParameter(logEventParameter);
            EventModel eventModel = (EventModel) eventModelProvider.provide();

            req.put("common", commonModel.getJson());
            req.put("evt", eventModel.getJson());

            dataRegistry.putDataRegistry(
                    new DataUnit(DataRegistryKey.STRING_PREV_ID,
                            lastFirstOpenId,
                            5,
                            this.getClass().getName(),
                            true));

        } catch (JSONException e) {
            e.printStackTrace();
        }
        return new DRModel(identityModel, userId, operationType, eventDatetime, req);
    }

    private String getNotSyncedInitRestartDatetime() {
        return dataRegistry.safeGetString(DataRegistryKey.STRING_INIT_RESTART_EVENT_DATETIME, null);
    }

    private String getNotSyncedInitRestartLastFirstOpenId() {
        return dataRegistry.safeGetString(DataRegistryKey.STRING_LAST_OPEN_ID, null);
    }
}
