package io.adbrix.sdk.data.dataprovider;

import android.content.ContentResolver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.res.Configuration;
import android.graphics.Point;
import android.location.Geocoder;
import android.net.ConnectivityManager;
import android.net.Network;
import android.net.NetworkCapabilities;
import android.net.NetworkInfo;
import android.os.BatteryManager;
import android.os.Build;
import android.provider.Settings;
import android.telephony.TelephonyManager;
import android.view.Display;
import android.view.WindowManager;

import com.google.android.gms.ads.identifier.AdvertisingIdClient;
import com.google.android.gms.common.GooglePlayServicesNotAvailableException;
import com.google.android.gms.common.GooglePlayServicesRepairableException;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;

import io.adbrix.sdk.component.IObservable;
import io.adbrix.sdk.component.IObserver;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.CoreUtils;

public class DeviceRealtimeDataProvider extends AbstractDataProvider implements IObservable<LogEventParameter> {
    private Context context;
    private List<IObserver<LogEventParameter>> observers = new ArrayList<>();

    public DeviceRealtimeDataProvider(DataRegistry dataRegistry, Context context) {
        super(dataRegistry);
        this.context = context;
        //TODO InAppMessage -> dev 머지 후 아래 주석 해제
        //add(DefaultABXContextController.getInstance());
    }

    public void refreshData() {
        AdInfo adInfo = getAdInfo();
        String adid;

        String prevGaid = safeGetString(DataRegistryKey.STRING_GAID, null);

        if (prevGaid != null && !prevGaid.equals(adInfo.advertisingId) ||
                prevGaid == null && adInfo.advertisingId != null
        ) {
            Map<String, Object> param = new HashMap<>();
            //TODO map의 value가 "" 또는 null 인 경우, parseValueWithDataType 에서 해당 key & value 가 삭제됨.
            //TODO 이 경우 null로 들어오는 prev_id 또는 curr_gaid 를 어떻게 표현하지?
            param.put("prev_gaid", prevGaid);
            param.put("curr_gaid", adInfo.advertisingId);

            notifyObserver(new LogEventParameter(
                    CoreConstants.GROUP_ABX,
                    CoreConstants.EVENT_ADID_CHANGED,
                    CommonUtils.parseValueWithDataType(param, CommonUtils.FIX_TYPE.PREFIX),
                    0,
                    0
            ));
        }

        if (adInfo.advertisingId != null)
            adid = adInfo.advertisingId;
        else {
            String uuid = safeGetString(DataRegistryKey.STRING_UUID, null);
            adid = uuid != null ? "uuid:" + uuid : null;
        }

        putString(DataRegistryKey.STRING_ADID,              adid,                           5, true);
        putString(DataRegistryKey.STRING_GAID,              adInfo.advertisingId,           5, true);
        putBoolean(DataRegistryKey.BOOLEAN_AD_ID_OPT_OUT,   adInfo.limitAdTracking,         5, true);
        putString(DataRegistryKey.STRING_IGAW_ID,           getIgawId(adInfo.advertisingId),5, true);
        putString(DataRegistryKey.STRING_RESOLUTION,        getResolution(),                5, false);
        putBoolean(DataRegistryKey.BOOLEAN_IS_PORTRAIT,     isPortrait(),                   5, false);
        putString(DataRegistryKey.STRING_NETWORK,           getNetwork(),                   5, false);
        putInt(DataRegistryKey.INT_BATT_L,                  getBatteryLevel(),              5, false);
        putBoolean(DataRegistryKey.BOOLEAN_BATT_C,          isBatteryCharging(),            5, false);
        putString(DataRegistryKey.STRING_LANGUAGE,          getLanguage(),                  5, false);
        putString(DataRegistryKey.STRING_COUNTRY,           getCountry(),                   5, true);
        putInt(DataRegistryKey.INT_TIME_ZONE_OFFSET,        getTimeZoneOffset(),            5, true);
        putInt(DataRegistryKey.INT_DEVICE_TIME_TYPE,        getDeviceTimeType(),            5, true);
    }

    @Override
    public void provideDefaultValues() {
        this.refreshData();
    }

    @Override
    public void asyncProvide(IDataProviderResultListener dataProviderResultListener) {
        this.refreshData();
        dataProviderResultListener.onSuccess();
    }

    private String getRequestDateTime() {
        return CommonUtils.getCurrentUTCInDBFormat();
    }

    private String getResolution() {
        WindowManager windowManager = ((WindowManager) context.getSystemService(Context.WINDOW_SERVICE));
        if(windowManager == null)
        {
            // Remaster에서는 windowManager가 null인 경우를 catch하지 않았다.
            return null;
        }
        Display defaultDisplay = windowManager.getDefaultDisplay();
        Point point = new Point();
        defaultDisplay.getSize(point);

        if(isPortrait())
        {
            return point.x + "x" + point.y;
        }
        else
        {
            return point.y + "x" + point.x;
        }
    }

    private boolean isPortrait() {
        return context.getResources().getConfiguration().orientation == Configuration.ORIENTATION_PORTRAIT;
    }

    private String getNetwork() {
        ConnectivityManager connectivityManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        if(connectivityManager == null)
            return "unknown";

        if(android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.M)
        {
            Network network = connectivityManager.getActiveNetwork();
            if (network != null)
            {
                NetworkCapabilities networkCapabilities = connectivityManager.getNetworkCapabilities(network);
                if(networkCapabilities != null)
                {
                    if (networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI))
                    {
                        return "wifi";
                    }
                    else if (networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR))
                    {
                        return "mobile";
                    }
                }
            }
        }
        else
        {
            try {
                NetworkInfo mobile = connectivityManager.getNetworkInfo(0);
                NetworkInfo wifi = connectivityManager.getNetworkInfo(1);

                if (mobile != null &&
                        (mobile.getState() == NetworkInfo.State.CONNECTED || mobile.getState() == NetworkInfo.State.CONNECTING))
                {
                    return "mobile";
                }
                else if (wifi != null &&
                        (wifi.getState() == NetworkInfo.State.CONNECTED || wifi.getState() == NetworkInfo.State.CONNECTING))
                {
                    return "wifi";
                }
            }
            catch (Exception e) {
                e.printStackTrace();
            }
        }

        return "unknown";
    }

    private String getLanguage() {
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.N) {
            try {
                if (context != null &&
                        context.getResources() != null &&
                        context.getResources().getConfiguration() != null &&
                        context.getResources().getConfiguration().getLocales().size() > 0)
                {
                    String language = context.getResources().getConfiguration().getLocales().get(0).getLanguage();
                    if(!CommonUtils.isNullOrEmpty(language))
                    {
                        return language.toUpperCase(Locale.US);
                    }
                }
            } catch (Exception e) {
                // 예외 처리
            }
        }

        String defaultLanguage = Locale.getDefault().getLanguage();
        if(CommonUtils.isNullOrEmpty(defaultLanguage))
        {
            return "unknown";
        }
        else
        {
            return defaultLanguage.toUpperCase(Locale.US);
        }
    }

    public int getBatteryLevel() {
        Intent intent = context.registerReceiver(null, new IntentFilter(Intent.ACTION_BATTERY_CHANGED));

        int level = -1;
        int scale = -1;
        if(intent != null)
        {
            level = intent.getIntExtra(BatteryManager.EXTRA_LEVEL, -1);
            scale = intent.getIntExtra(BatteryManager.EXTRA_SCALE, -1);
        }

        return (int) ((level / (float) scale) * 100);
    }

    private boolean isBatteryCharging() {
        Intent intent = context.registerReceiver(null, new IntentFilter(Intent.ACTION_BATTERY_CHANGED));

        int state = -1;
        if (intent != null) {
            state = intent.getIntExtra(BatteryManager.EXTRA_PLUGGED, -1);
        }

        return state == BatteryManager.BATTERY_PLUGGED_AC ||
               state == BatteryManager.BATTERY_PLUGGED_USB ||
                (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1 && state == BatteryManager.BATTERY_PLUGGED_WIRELESS);
    }

    /**
     * 2021.02.18 - walter
     * 개인 정보 이슈로 위치정보 수집 기능을 제거함.
     * 따라서 위치 기반 Country Code 수집도 제외함.
     * @return
     */
    private String getCountry() {
        // This should not be called on the main thread.

        // Prioritize reverse geocode, but until we have a result from that,
        // we try to grab the country from the network, and finally the locale

        try {
//            String country = getCountryFromLocation();
//            if (!CommonUtils.isNullOrEmpty(country))
//            {
//                return country;
//            }

            TelephonyManager telephonyManager = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
            if (telephonyManager.getSimState() == TelephonyManager.SIM_STATE_READY &&
                    telephonyManager.getPhoneType() != TelephonyManager.PHONE_TYPE_CDMA)
            {
                String countryCode = telephonyManager.getNetworkCountryIso();
                if (!CommonUtils.isNullOrEmpty(countryCode))
                {
                    return countryCode.toUpperCase(Locale.US);
                }
            }
        } catch (Exception e) {
        }

        return Locale.getDefault().getCountry();
    }

    /**
     * 2021.02.18 - walter
     * 개인 정보 이슈로 위치정보 수집 기능을 제거함.
     * @return
     */
//    private String getCountryFromLocation() {
//        if (!isLocationListening)
//        {
//            return null;
//        }
//
//        Location mostRecentLocation = getMostRecentLocation();
//        if (mostRecentLocation != null) {
//            try {
//                if (Geocoder.isPresent())
//                {
//                    Geocoder geocoder = getGeocoder();
//                    List<Address> addresses = geocoder.getFromLocation(mostRecentLocation.getLatitude(), mostRecentLocation.getLongitude(), 1);
//                    if (addresses != null)
//                    {
//                        for (Address address : addresses)
//                        {
//                            if (address != null)
//                            {
//                                return address.getCountryCode();
//                            }
//                        }
//                    }
//                }
//            } catch (IOException | NullPointerException | NoSuchMethodError | IllegalArgumentException | IllegalStateException e) {
//                // Failed to reverse geocode location
//                // Failed to reverse geocode location
//                // failed to fetch geocoder
//                // Bad lat / lon values can cause Geocoder to throw IllegalArgumentExceptions
//                // sometimes the location manager is unavailable
//            }
//        }
//        return null;
//    }

//    private String getLocationId()
//    {
//        return UUID.randomUUID().toString();
//    }
//
//    private Location getLocation() {
//        if(isLocationListening)
//        {
//            return getMostRecentLocation();
//        }
//        else
//        {
//            // 기존 코드에서는 set된 lat, lng를 사용했다.
//
//            return null;
//        }
//    }
//
//    public Location getMostRecentLocation() {
//        if(isLocationListening == false)
//        {
//            return null;
//        }
//
//        LocationManager locationManager = (LocationManager) context.getSystemService(Context.LOCATION_SERVICE);
//
//        // Don't crash if the device does not have location services.
//        if (locationManager == null)
//        {
//            return null;
//        }
//
//        // It's possible that the location service is running out of process
//        // and the remote getProviders call fails. Handle null provider lists.
//        List<String> providers = null;
//        try {
//            providers = locationManager.getProviders(true);
//        } catch (SecurityException e) {
//            // failed to get providers list
//        }
//        if (providers == null)
//        {
//            return null;
//        }
//
//        List<Location> locations = new ArrayList<Location>();
//        for (String provider : providers) {
//            Location location = null;
//            try {
//                location = locationManager.getLastKnownLocation(provider);
//            } catch (IllegalArgumentException e) {
//                // failed to get last known location from provider
//            } catch (SecurityException e) {
//                // failed to get last known location from provider
//            }
//            if (location != null)
//            {
//                locations.add(location);
//            }
//        }
//
//        long maximumTimestamp = -1;
//        Location bestLocation = null;
//        for (Location location : locations)
//        {
//            if (location.getTime() > maximumTimestamp)
//            {
//                maximumTimestamp = location.getTime();
//                bestLocation = location;
//            }
//        }
//
//        return bestLocation;
//    }

    // @VisibleForTesting
    protected Geocoder getGeocoder() {
        return new Geocoder(context, Locale.ENGLISH);
    }

    // TODO Realtime인지, Static인지
    private int getTimeZoneOffset() {
        try {
            TimeZone defaultTimeZone = TimeZone.getDefault();
            String GMT = defaultTimeZone.getDisplayName(
                    defaultTimeZone.inDaylightTime(new Date()),
                    TimeZone.SHORT
            );

            if (CommonUtils.isNullOrEmpty(GMT) == false && GMT.length() == 9)
            {
                String[] timeArr = GMT.substring(4).split(":");
                int absOffset = Integer.valueOf(timeArr[0]) * 60 + Integer.valueOf(timeArr[1]);
                return GMT.charAt(3) == '+' ? absOffset : absOffset * -1;
            }
        } catch (AssertionError | Exception e) {
            e.printStackTrace();
        }

        return 0;
    }

    private int getDeviceTimeType() {
        int useAutoTime;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1)
        {
            useAutoTime = Settings.Global.getInt(
                    context.getContentResolver(),
                    Settings.Global.AUTO_TIME,
                    -1);
        }
        else
        {
            useAutoTime = android.provider.Settings.System.getInt(
                    context.getContentResolver(),
                    Settings.System.AUTO_TIME,
                    -1);
        }

        if (useAutoTime == 1)
        {
            return 1;
        }
        else if (useAutoTime == 0)
        {
            return 2;
        }
        else
        {
            return 0;
        }
    }

    //googleServiceDataProvider
    private static class AdInfo {
        final String advertisingId;
        final boolean limitAdTracking;

        public AdInfo(String advertisingId, boolean limitAdTracking) {
            this.advertisingId = advertisingId;
            this.limitAdTracking = limitAdTracking;
        }
    }

    private AdInfo getAdInfo() {
        AdvertisingIdClient.Info info = null;

        try {
            info = AdvertisingIdClient.getAdvertisingIdInfo(context);
        } catch (IOException | GooglePlayServicesNotAvailableException | GooglePlayServicesRepairableException e) {
            e.printStackTrace();
        }

        if (info != null && !CommonUtils.isNullOrEmpty(info.getId()))
        {
            return new AdInfo(info.getId(), info.isLimitAdTrackingEnabled());
        }
        else if ("Amazon".equals(Build.MANUFACTURER))
        {
            ContentResolver contentResolver = context.getContentResolver();
            return new AdInfo(
                    Settings.Secure.getString(contentResolver, "advertising_id"),
                    Settings.Secure.getInt(contentResolver, "limit_ad_tracking", 0) == 1);
        }
        else
        {
            return new AdInfo(null, true);
        }
    }

    private String getIgawId(String adid){
        if (adid == null){
            String uuid = safeGetString(DataRegistryKey.STRING_UUID, null);
            return "iga:" + CoreUtils.md5(uuid + "!gaWrk");
        }
        return "iga:" + CoreUtils.md5(adid + "!gaWrk");
    }

    @Override
    public void add(IObserver<LogEventParameter> observer) {
        observers.add(observer);
    }

    @Override
    public void delete(IObserver<LogEventParameter> observer) {
        observers.remove(observer);
    }

    @Override
    public void notifyObserver(LogEventParameter logEventParameter) {
        for (IObserver<LogEventParameter> observer : observers) {
            observer.update(logEventParameter);
        }
    }
}