package io.adbrix.sdk.utils;

import android.util.Log;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;
import java.util.UUID;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.data.S3ConfigHandler;
import io.adbrix.sdk.domain.ABXConstants;
import io.adbrix.sdk.domain.CoreConstants;

public class CommonUtils {

    private CommonUtils(){}

    public static boolean isNullOrEmpty(String string) {
        return string == null || string.isEmpty();
    }

    public static String getUTC_DBFormat(long timeStamp) {
        SimpleDateFormat sdfKST = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", Locale.KOREA);
        sdfKST.setTimeZone(TimeZone.getTimeZone("GMT+0"));
        return sdfKST.format(new Date(timeStamp));
    }

    public static String convertNullStringToNull(String string) {
        return string == "null" ? null : string;
    }

    // (현재 시간):(UUID) 형식의 문자열을 만드는 Utility Method이다. Event Id 등을 만드는 데에 사용된다.
    public static String randomUUIDWithCurrentTime()
    {
        return System.currentTimeMillis() + ":" + UUID.randomUUID().toString();
    }

    public static String getCurrentUTCInDBFormat() {
        DateFormat dateFormat = CoreUtils.createDateFormat(CoreConstants.DB_DATE_FORMAT);
        Date date = new Date(System.currentTimeMillis());
        return dateFormat.format(date);
    }

    /**
     * Truncate values in a JSON object. Any string values longer than 1024 characters will be
     * truncated to 1024 characters.
     * Any dictionary with more than 1000 items will be ignored.
     *
     * @param object the object
     * @return the truncated JSON object
     */
    public static JSONObject truncate(JSONObject object) {
        if (object == null) {
            return new JSONObject();
        }
        Iterator<?> keys = object.keys();
        while (keys.hasNext()) {
            String key = (String) keys.next();
            if (key.length() > ABXConstants.MAX_KEY_LENGTH) {
                AbxLog.w(String.format("Validation CHECK :: Invalid key >> Maximum length is %s. Key %s is too long.", String.valueOf(ABXConstants.MAX_KEY_LENGTH), key.substring(0, ABXConstants.MAX_KEY_LENGTH)), true);
                object.remove(key);
            } else {
                try {
                    Object value = object.get(key);
                    // do not truncate important key
                    if (key.equals(CoreConstants.SPECIAL_KEY) || key.equals(CoreConstants.DEEPLINK_PAYLOAD) || key.equals(CoreConstants.COMMERCE_KEY)) {
                        object.put(key, value);
                    } else {

                        if (isNullOrEmpty(key) == false) {

                            if (value.getClass().equals(String.class)) {
                                object.put(key, truncate((String) value));
                            } else if (value.getClass().equals(JSONObject.class)) {
                                object.put(key, truncate((JSONObject) value));  //this method
                            } else if (value.getClass().equals(JSONArray.class)) {
                                object.put(key, truncate((JSONArray) value));
                            }
                        }
                    }
                    if (object.length() > S3ConfigHandler.config_propertyMaxSize) {
                        AbxLog.w("Validation CHECK :: Warning: properties over the limit " + S3ConfigHandler.config_propertyMaxSize + ", exceeded objects be ignored", true);
                        break;
                    }
                } catch (JSONException e) {
                    AbxLog.e(e.toString(), true);
                }
            }


        }

        return object;
    }

    /**
     * Truncate a string to 1024 characters.
     *
     * @param value the value
     * @return the truncated string
     */
    public static String truncate(String value) {
        return value.length() <= CoreConstants.MAX_STRING_LENGTH ? value : value.substring(0, CoreConstants.MAX_STRING_LENGTH);
    }

    /**
     * Truncate values in a JSON array. Any string values longer than 1024 characters will be
     * truncated to 1024 characters.
     *
     * @param array the array
     * @return the truncated JSON array
     * @throws JSONException the json exception
     */
    public static JSONArray truncate(JSONArray array) throws JSONException {
        if (array == null) {
            return new JSONArray();
        }

        for (int i = 0; i < array.length(); i++) {
            Object value = array.get(i);
            if (value.getClass().equals(String.class)) {
                array.put(i, truncate((String) value));
            } else if (value.getClass().equals(JSONObject.class)) {
                array.put(i, truncate((JSONObject) value));
            } else if (value.getClass().equals(JSONArray.class)) {
                array.put(i, truncate((JSONArray) value));
            }
        }
        return array;
    }

    public enum FIX_TYPE {
        PREFIX, SUFFIX
    }

    public static Map<String, Object> parseValueWithDataType(Map<String, Object> map, FIX_TYPE fixType){
        JSONObject jsonObject = getJSONObjectFromMap(map);
        jsonObject = parseValueWithDataType(jsonObject, fixType);
        map = getMapFromJSONObject(jsonObject);
        return map;
    }

    public static JSONObject getJSONObjectFromMap(Map<String, Object> map ) {
        JSONObject jsonObject = new JSONObject();
        try {
            for (Map.Entry<String, Object> entry : map.entrySet()) {
                String key = entry.getKey();
                Object value = entry.getValue();
                jsonObject.put(key, value);
            }
        }
        catch (JSONException e){
            e.printStackTrace();
        }

        return jsonObject;
    }

    public static Map<String, Object> getMapFromJSONObject(JSONObject object) {
        Map<String, Object> map = new HashMap<>();

        Iterator<String> keysItr = object.keys();
        try {
            while (keysItr.hasNext()) {
                String key = keysItr.next();

                Object value = object.get(key);

                if (value instanceof JSONObject) {
                    value = getMapFromJSONObject((JSONObject) value);
                }
                map.put(key, value);
            }
        }
        catch (JSONException e){
            e.printStackTrace();
        }
        return map;
    }

    public static JSONObject parseValueWithDataType(JSONObject json, FIX_TYPE fixType) {
        JSONObject retJsonObject = new JSONObject();
        try {
            if (json != null) {
                Iterator<?> keys = json.keys();
                while (keys.hasNext()) {
                    String key = (String) keys.next();

                    Object obj = json.get(key);
                    if(obj == null || obj == JSONObject.NULL || obj == ""){
                        continue;
                    }
                    StringBuilder sb = new StringBuilder();

                    if (obj instanceof String) {
                        sb.append("string");
                    } else if (obj instanceof Boolean) {
                        sb.append("boolean");
                    } else if (obj instanceof Integer) {
                        sb.append("long");
                    } else if (obj instanceof Long) {
                        sb.append("long");
                    } else if (obj instanceof Double) {
                        sb.append("double");
                    } else if (obj instanceof Float) {
                        sb.append("double");
                    } else {
                        sb.append("string");
                    }

                    if(FIX_TYPE.PREFIX == fixType) {
                        sb.append(":");
                        sb.append(obj);
                    }
                    else {
                        sb.insert(0, ":");
                        sb.insert(0, obj);
                    }
                    retJsonObject.put(key, sb.toString());
                }
            }
        }
        catch (JSONException e) {
            e.printStackTrace();
        }

        return retJsonObject;
    }

    public static Object replaceWithJSONNull(Object obj) {
        return obj == null ? JSONObject.NULL : obj;
    }

    public static <T> List<List<T>> split(List<T> list, int splitSize){
        List<List<T>> lists = new ArrayList<>();
        List<T> beginning;
        List<T> remaining;

        if (list.size() <= splitSize) {
            lists.add(list);
            return lists;
        }
        else {
            beginning = list.subList(0, splitSize);
            remaining = list.subList(splitSize, list.size());

            lists.add(beginning);
            lists.addAll(split(remaining, splitSize));

            return lists;
        }
    }
}