package io.adbrix.sdk.data.net;

import java.util.Timer;
import java.util.TimerTask;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.configuration.DefaultABXContextController;
import io.adbrix.sdk.configuration.DisabledABXContext;
import io.adbrix.sdk.data.ABXBooleanState;

public class ApiConnectionManager {
    public interface Result {
        void connectSuccess(String responseString, int responseCode);
        void connectFail(int responseCode);
    }

    private Result result;
    private int attempt = 0;
    private final int MAX_RETRY = 3;
    private Timer retryTimer;

    public ApiConnectionManager(Result result) {
        if (result == null)
            this.result = new NullResult();
        else this.result = result;
    }

    public void ExecuteWithRetry(ApiConnection apiConnection) {
        if (ABXBooleanState.getInstance().isAdbrixPause()){
            AbxLog.d("ApiConnectionManager :: Skip request due to pause state", true);
            result.connectFail(apiConnection.getResponseCode());
            return;
        }

        String response = apiConnection.request();

        if (apiConnection.isHttpOK()) {
            if (response != null && response.equals("{\"deeplink\":null}")) {
                retry(apiConnection);
            }
            else
                result.connectSuccess(response, apiConnection.getResponseCode());
        }
        else if(apiConnection.isWrongAppkey()){
            // HTTP 500: wrong appKey
            ABXBooleanState.getInstance().AdbrixError.getAndSet(true);
            AbxLog.w("Warning!! :: wrong appkey : " + response, true);

            retry(apiConnection);
        }
        else if(apiConnection.isInvalidAppkey()){
            //AppKey ERR
            ABXBooleanState.getInstance().AdbrixPause.getAndSet(true);
            ABXBooleanState.getInstance().AdbrixAllStop.getAndSet(true);
            ABXBooleanState.getInstance().AdbrixError.getAndSet(true);
            AbxLog.w("Warning!! :: Appkey is not valid. " + response, true);
            // not retry
            AbxLog.d("Broken connection to AdBrixRm. Skip retry uploading events. " + response, true);

            DefaultABXContextController.getInstance().changeABXContext(new DisabledABXContext(DefaultABXContextController.getInstance(), "AdbrixAllStop"));
            result.connectFail(apiConnection.getResponseCode());
        }
        else {
            // not retry
            AbxLog.d("Broken connection to AdBrixRm. Skip retry uploading events. " + response, true);
            result.connectFail(apiConnection.getResponseCode());
        }
    }

    public void Execute(ApiConnection apiConnection) {
        if (ABXBooleanState.getInstance().isAdbrixPause()){
            AbxLog.d("ApiConnectionManager :: Skip request due to pause state", true);
            result.connectFail(apiConnection.getResponseCode());
            return;
        }

        String response = apiConnection.request();

        if (apiConnection.isHttpOK()) {
            result.connectSuccess(response, apiConnection.getResponseCode());
        }
        else if(apiConnection.isWrongAppkey()){
            // HTTP 500: wrong appKey
            ABXBooleanState.getInstance().AdbrixError.getAndSet(true);
            AbxLog.w("Warning!! :: wrong appkey : " + response, true);
            result.connectFail(apiConnection.getResponseCode());
        }
        else if(apiConnection.isInvalidAppkey()){
            //AppKey ERR
            ABXBooleanState.getInstance().AdbrixPause.getAndSet(true);
            ABXBooleanState.getInstance().AdbrixAllStop.getAndSet(true);
            ABXBooleanState.getInstance().AdbrixError.getAndSet(true);
            AbxLog.w("Warning!! :: Appkey is not valid. " + response, true);
            // not retry
            AbxLog.d("Broken connection to AdBrixRm. Skip retry uploading events. " + response, true);

            DefaultABXContextController.getInstance().changeABXContext(new DisabledABXContext(DefaultABXContextController.getInstance(), "AdbrixAllStop"));
            result.connectFail(apiConnection.getResponseCode());
        }
        else {
            // not retry
            AbxLog.d("Broken connection to AdBrixRm. Skip retry uploading events. " + response, true);
            result.connectFail(apiConnection.getResponseCode());
        }
    }



    private static class NullResult implements Result{
        @Override
        public void connectSuccess(String responseString, int responseCode) {
            //do nothing
        }

        @Override
        public void connectFail(int responseCode) {
            //do nothing
        }
    }

    private void retry(ApiConnection apiConnection){
        if(attempt >= MAX_RETRY) {
            if(retryTimer!=null) {
                retryTimer.cancel();
            }
            AbxLog.d( "ApiConnectionManager retry failed. Skip retry uploading events.", true);
            result.connectFail(apiConnection.getResponseCode());
            return;
        }

        retryTimer = new Timer();

        retryTimer.schedule(new TimerTask() {
            @Override
            public void run() {
                AbxLog.d( "ApiConnectionManager retry : ("+(attempt + 1) +"/3)", true);
                attempt++;
                ExecuteWithRetry(apiConnection);
            }
        }, 10_000);
    }
}
