package io.adbrix.sdk.configuration;

import android.app.Activity;
import android.content.Context;

import java.util.HashMap;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.DefaultComponentsFactory;
import io.adbrix.sdk.component.IABXComponentsFactory;
import io.adbrix.sdk.component.ILogger;
import io.adbrix.sdk.data.ABXBooleanState;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.data.repository.RepositoryImpl;
import io.adbrix.sdk.data.repository.UserPropertyManager;
import io.adbrix.sdk.domain.LogMessageFormat;
import io.adbrix.sdk.domain.Repository;
import io.adbrix.sdk.domain.exception.BeforeInitializeException;
import io.adbrix.sdk.domain.interactor.GdprForgetMeUseCase;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.domain.model.UserPropertyCommand;
import io.adbrix.sdk.domain.model.UserPropertyModel;
import io.adbrix.sdk.utils.CommonUtils;

public class DefaultABXContextController implements IABXContextController {
    private IABXComponentsFactory componentsFactory;
    private ILogger logger;
    private IABXContext abxContext;
    private DataRegistry dataRegistry;
    private ExecutorService executorService;
    private boolean isInitSubmitted = false;
    private boolean isOnResumeSubmitted = false;
    private boolean isFirstOpenSubmitted = false;
    private boolean isSDKFirstOpened = false;
    private Queue<MessageInvokeRunnable> initQueue;
    private Queue<MessageInvokeRunnable> onResumeQueue;
    private Queue<MessageInvokeRunnable> firstOpenQueue;
    private Queue<MessageInvokeRunnable> etcQueue;
    private Repository repository;

    private static class Singleton{
        private static final DefaultABXContextController instance = new DefaultABXContextController();
    }

    private DefaultABXContextController(){}

    public static DefaultABXContextController getInstance(){
        return Singleton.instance;
    }

    public void startController(Context appContext){
        this.componentsFactory = new DefaultComponentsFactory();
        this.componentsFactory.setAndroidContext(appContext);
        this.repository = new RepositoryImpl(componentsFactory);
        this.abxContext = new InitializingABXContext(this, repository);
        this.componentsFactory.setABXContext(abxContext);
        this.logger = this.componentsFactory.createOrGetLogger();
        executorService = Executors.newSingleThreadExecutor();
        initQueue = new ConcurrentLinkedQueue<>();
        onResumeQueue = new ConcurrentLinkedQueue<>();
        firstOpenQueue = new ConcurrentLinkedQueue<>();
        etcQueue = new ConcurrentLinkedQueue<>();

        try {
            dataRegistry = componentsFactory.createOrGetDataRegistry();
        } catch (IABXComponentsFactory.ComponentsCanNotCreateException e) { //이런일이 발생해서는 안된다.
            this.logger.error(e.toString());
        }

        isSDKFirstOpened = dataRegistry.safeGetString(DataRegistryKey.STRING_LAST_FIRSTOPEN_ID, null) == null;
    }

    public void stopController(){
        executorService.shutdown();
    }

    private static class Message{
        private MessageType messageType;
        private Object[] args;
        public Message(MessageType messageType,Object...args)
        {
            this.messageType = messageType;
            this.args = args;
        }
    }

    private enum MessageType
    {
        CHANGE_ABX_CONTEXT,
        INITIALIZE,
        SAVE_USER_PROPERTY,
        SAVE_USER_PROPERTY_WITHOUT_EVENT,
        LOG_EVENT,
        ON_RESUME,
        ON_PAUSE,
        DEEPLINK,
        GDPR_FORGET_ME,
        PUT_DATA_REGISTRY,
        RUN_IN_BACKGROUND_IN_ORDER,
        RUN_IN_BACKGROUND_WITHOUT_ORDER,
        REGISTER_NETWORK_CALLBACK,
        FIRST_OPEN_START_SESSION
    }

    private class MessageInvokeRunnable implements Runnable {
        private Message message;

        public MessageInvokeRunnable(MessageType messageType, Object... args){
            this.message = new Message(messageType, args);
        }

        @Override
        public void run(){
            //메시지 처리
            try {
                if (message == null) {
                    idleTime();
                } else {
                    try {
                        processMessage(message);
                    }
                    catch(BeforeInitializeException be) {
                        AbxLog.w(
                                String.format(
                                        LogMessageFormat.DEFAULT_ABX_CONTEXT_BEFORE_INITIALIZE_EXCEPTION,
                                        be.toString()
                                ), true
                        );
                    }
                }
            }catch (Exception e) {
                AbxLog.e(
                        String.format(
                                LogMessageFormat.DEFAULT_ABX_CONTEXT_CONTROLLER_QUEUE_EXCEPTION,
                                message.messageType.toString(),
                                e.toString()
                        ), true
                );
                e.printStackTrace();
            }
        }

        /**
         * message가 없을때 백그라운드 작업을 수행한다.
         */
        private void idleTime(){
            DefaultABXContextController.this.abxContext.runOnIdleTime();
        }

        /**
         * message를 처리한다.
         * @param message
         */
        private void processMessage(Message message)
        {
            switch (message.messageType)
            {
                case CHANGE_ABX_CONTEXT:
                    {
                        IABXContext newContext = (IABXContext)message.args[0];
                        DefaultABXContextController.this.abxContext = newContext;
                    }
                    break;
                case INITIALIZE:
                    {
                        Context context = (Context)message.args[0];
                        String appkey = (String)message.args[1];
                        String secretkey = (String)message.args[2];
                        DefaultABXContextController.this.abxContext.initialize(context,appkey,secretkey);

                        ABXBooleanState.getInstance().sdkInitializedFlag.getAndSet(true);
                    }
                    break;
                case SAVE_USER_PROPERTY:
                    {
                        UserPropertyCommand userPropertyCommand = (UserPropertyCommand)message.args[0];
                        DefaultABXContextController.this.abxContext.saveUserProperty(userPropertyCommand);
                    }
                    break;
                case SAVE_USER_PROPERTY_WITHOUT_EVENT:
                    {
                        UserPropertyCommand userPropertyCommand = (UserPropertyCommand)message.args[0];
                        DefaultABXContextController.this.abxContext.saveUserPropertyWithoutEvent(userPropertyCommand);
                    }
                    break;
                case LOG_EVENT:
                    {
                        LogEventParameter logEventParameter = (LogEventParameter)message.args[0];

                        //coreWrapper에서 들어오는 logevent의 eventId앞 시간과 event_datetime을 일치시킴
                       if(logEventParameter.event_datetime == null) {
                            logEventParameter.event_datetime = CommonUtils.getCurrentUTCInDBFormat();
                            logEventParameter.eventId = CommonUtils.randomUUIDWithCurrentTime();
                        }

                        DefaultABXContextController.this.abxContext.logEvent(logEventParameter);
                    }
                    break;
                case ON_RESUME:
                    {
                        Activity activity = (Activity)message.args[0];
                        DefaultABXContextController.this.abxContext.onResume(activity);
                    }
                    break;
                case ON_PAUSE:
                    {
                        DefaultABXContextController.this.abxContext.onPause();
                    }
                    break;
                case DEEPLINK:
                    {
                        Activity deeplinkActivity = (Activity)message.args[0];
                        DefaultABXContextController.this.abxContext.deeplink(deeplinkActivity);
                    }
                    break;
                case GDPR_FORGET_ME:
                    {
                        new GdprForgetMeUseCase(repository).execute();

                        DefaultABXContextController.this.abxContext =
                                new DisabledABXContext(DefaultABXContextController.this,"GDPR FORGET ME");
                    }
                    break;
                case PUT_DATA_REGISTRY:
                    {
                        DataUnit dataUnit = (DataUnit) message.args[0];
                        DefaultABXContextController.this.abxContext.putDataRegistry(dataUnit);
                    }
                    break;
                case RUN_IN_BACKGROUND_IN_ORDER:
                case RUN_IN_BACKGROUND_WITHOUT_ORDER:
                case FIRST_OPEN_START_SESSION: {
                        Runnable runnable = (Runnable) message.args[0];
                        runnable.run();
                    }
                    break;
                case REGISTER_NETWORK_CALLBACK:
                    {
                        DefaultABXContextController.this.abxContext.registerNetworkCallback();
                    }
                    break;
                default:
                    throw new IllegalStateException("Unexpected value: " + message.messageType);
            }

            //메시지 처리 이후 백그라운드 작업을 수행한다.
            DefaultABXContextController.this.abxContext.runOnIdleTime();
        }
    }

    @Override
    public IABXComponentsFactory getComponentsFactory(){
        return this.componentsFactory;
    }

    @Override
    public void setDataRegistry(DataRegistry dataRegistry) {
        this.dataRegistry = dataRegistry;
        ABXBooleanState.getInstance().getGdprData(dataRegistry);
    }

    @Override
    public void changeABXContext(IABXContext newContext) {
        this.abxContext = newContext;
        this.componentsFactory.setABXContext(newContext);
    }

    public void initialize(Context context, String appkey, String secretkey) {
        submit(MessageType.INITIALIZE, context, appkey, secretkey);
    }

    public void saveUserProperty(UserPropertyCommand userPropertyCommand) {
        submit(MessageType.SAVE_USER_PROPERTY, userPropertyCommand);
    }

    public void saveUserPropertyWithoutEvent(UserPropertyCommand userPropertyCommand) {
        submit(MessageType.SAVE_USER_PROPERTY_WITHOUT_EVENT, userPropertyCommand);
    }

    public void logEvent(LogEventParameter logEventParameter) {
        submit(MessageType.LOG_EVENT, logEventParameter);
    }

    public void onResume(Activity activity) {
        ABXBooleanState.getInstance().inForeground.getAndSet(true);
        submit(MessageType.ON_RESUME, activity);
    }

    public void onPause() {
        ABXBooleanState.getInstance().inForeground.getAndSet(false);
        submit(MessageType.ON_PAUSE);
    }

    public void deeplink(Activity deeplinkActivity) {
        submit(MessageType.DEEPLINK, deeplinkActivity);
    }

    public void gdprForgetMe(){
        submit(MessageType.GDPR_FORGET_ME);
    }

    public void putDataRegistry(DataUnit dataUnit){
        submit(MessageType.PUT_DATA_REGISTRY, dataUnit);
    }

    public void runInBackGroundInOrder(Runnable runnable){
        submit(MessageType.RUN_IN_BACKGROUND_IN_ORDER, runnable);
    }

    public void runInBackGroundWithoutOrder(Runnable runnable){
        submit(MessageType.RUN_IN_BACKGROUND_WITHOUT_ORDER, runnable);
    }

    public void processFirstOpenStartSession(Runnable runnable) {
        submit(MessageType.FIRST_OPEN_START_SESSION, runnable);
    }

    public void registerNetworkCallback(){
        submit(MessageType.REGISTER_NETWORK_CALLBACK);
    }

    public DataRegistry getDataRegistry() {
        return dataRegistry;
    }

    public UserPropertyModel getCurrentUserPropertyModel(){
        try {
            UserPropertyManager userPropertyManager = this.componentsFactory.createOrGetUserPropertyManager();
            return userPropertyManager.getCurrentUserPropertyModel();
        } catch (IABXComponentsFactory.ComponentsCanNotCreateException e) {
            e.printStackTrace();
        }
        return new UserPropertyModel(null, new HashMap<String, Object>());
    }

    private void submit(MessageType messageType, Object... args){
        if (executorService.isShutdown())
            executorService = Executors.newSingleThreadExecutor();

        if (isInitSubmitted && messageType == MessageType.RUN_IN_BACKGROUND_WITHOUT_ORDER) {
            executorService.submit(new MessageInvokeRunnable(messageType, args));
            return;
        }

        if (!isSDKFirstOpened && isOnResumeSubmitted) {
            executorService.submit(new MessageInvokeRunnable(messageType, args));
            return;
        }
        else if (isSDKFirstOpened && isFirstOpenSubmitted) {
            executorService.submit(new MessageInvokeRunnable(messageType, args));
            return;
        }

        if (messageType == MessageType.INITIALIZE)
            initQueue.offer(new MessageInvokeRunnable(messageType, args));
        else if (!isOnResumeSubmitted && messageType == MessageType.ON_RESUME)
            onResumeQueue.offer(new MessageInvokeRunnable(messageType, args));
        else if (isSDKFirstOpened && messageType == MessageType.FIRST_OPEN_START_SESSION)
            firstOpenQueue.offer(new MessageInvokeRunnable(messageType, args));
        else etcQueue.offer(new MessageInvokeRunnable(messageType, args));

        if (!initQueue.isEmpty()) {
            executorService.submit(initQueue.poll());
            isInitSubmitted = true;
        }

        if (isInitSubmitted && !onResumeQueue.isEmpty() && !isOnResumeSubmitted) {
            executorService.submit(onResumeQueue.poll());
            isOnResumeSubmitted = true;
        }

        if (isOnResumeSubmitted && isSDKFirstOpened && !firstOpenQueue.isEmpty()) {
            executorService.submit(firstOpenQueue.poll());
            isFirstOpenSubmitted = true;
        }

        if (isSDKFirstOpened && isFirstOpenSubmitted) {
            while (!etcQueue.isEmpty())
                executorService.submit(etcQueue.poll());
        }
        else if (!isSDKFirstOpened && isOnResumeSubmitted) {
            while (!etcQueue.isEmpty())
                executorService.submit(etcQueue.poll());
        }
    }
}