package io.adbrix.sdk.component;

import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.EventPackage;

import java.util.ArrayList;
import java.util.Queue;
import java.util.Timer;
import java.util.TimerTask;

public class EventUploadIntervalManager implements IObservable<Void>{
    private int COUNT_INTERVAL = 0;
    private int TIME_INTERVAL = 0;
    private UploadMode currentUploadMode = UploadMode.NOT_SELECTED_MODE;
    private Timer eventBusTimer;
    private ArrayList<IObserver<Void>> observerArrayList = new ArrayList<>();

    public enum UploadMode {
        COUNT_MODE,
        TIME_MODE,
        NOT_SELECTED_MODE
    }

    private EventUploadIntervalManager() {}

    private static class Singleton{
        private static final EventUploadIntervalManager instance = new EventUploadIntervalManager();
    }

    public static EventUploadIntervalManager getInstance() {
        return Singleton.instance;
    }

    public void setCountInterval(int countInterval) {
        this.COUNT_INTERVAL = countInterval;
        currentUploadMode = UploadMode.COUNT_MODE;
    }

    public void setTimeInterval(int timeInterval) {
        this.TIME_INTERVAL = timeInterval;
        currentUploadMode = UploadMode.TIME_MODE;
    }

    public void manageEventUploadInterval(Runnable eventSendingTask, Queue<EventPackage> packageQueue) {
        switch (currentUploadMode){
            case TIME_MODE:
                scheduleEventSendingJob(packageQueue);
                break;
            case COUNT_MODE:
                countEventSendingJob(eventSendingTask, packageQueue);
                break;
            default:
                AbxLog.d("EventUploadMode is not selected.", true);
                break;
        }
    }

    private void scheduleEventSendingJob(Queue<EventPackage> packageQueue){
        AbxLog.d("scheduleEventSendingJob: queue size : " + packageQueue.size(), true);
        if (eventBusTimer == null) {
            eventBusTimer = new Timer();
        }
        else return;

        eventBusTimer.schedule(new TimerTask() {
            @Override
            public void run() {
                notifyObserver(null);
            }
        }, this.TIME_INTERVAL, this.TIME_INTERVAL);
    }

    private void countEventSendingJob(Runnable eventSendingTask, Queue<EventPackage> packageQueue) {
        AbxLog.d("countEventSendingJob: queue size : " + packageQueue.size(), true);
        if (packageQueue.peek() == null)
            return;

        if (packageQueue.size() >= (COUNT_INTERVAL / CoreConstants.EVENT_BUFFER_COUNT)){
            eventSendingTask.run();
        }
    }

    @Override
    public void add(IObserver<Void> observer) {
        observerArrayList.add(observer);
    }

    @Override
    public void delete(IObserver<Void> observer) {
        observerArrayList.remove(observer);
    }

    @Override
    public void notifyObserver(Void v) {
        for (IObserver<Void> observer : observerArrayList){
            observer.update(v);
        }
    }
}
