package io.adbrix.sdk.component;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Queue;
import java.util.UUID;
import java.util.regex.Pattern;

import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.data.repository.datasource.IDataContext;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.LogMessageFormat;
import io.adbrix.sdk.domain.model.UserPropertyCommand;
import io.adbrix.sdk.domain.model.UserPropertyModel;
import io.adbrix.sdk.utils.CommonUtils;

public class UserPropertyManager implements IUserPropertyManager {
    private ILogger logger;
    private HashMap<String, Object> userPropertyMap = new HashMap<>();
    private IDataContext dataContext;
    private DataRegistry dataRegistry;

    public UserPropertyManager(IDataContext dataContext, ILogger logger, DataRegistry dataRegistry) {
        this.dataContext = dataContext;
        this.logger = logger;
        this.dataRegistry = dataRegistry;

        //dataContext를 통해 디스크에 저장된 정보로 초기화 한다.
        String userPropertyString = dataContext.getValueOrNull(IDataContext.KEY_USER_PROPERTY);
        if (userPropertyString != null) {
            try {
                JSONObject jsonObject = new JSONObject(userPropertyString);
                Iterator itr = jsonObject.keys();
                while (itr.hasNext()) {
                    String key = itr.next().toString();
                    Object value = jsonObject.get(key);
                    this.userPropertyMap.put(key, value);
                }
            } catch (Exception e) {
                this.logger.error(e.toString());
            }
        }
    }

    @Override
    public Boolean merge(UserPropertyCommand command) {
        boolean isChanged = false;
        Queue<UserPropertyCommand.UserPropertyCommandUnit> commandQueue = command.getCommands();
        while (!commandQueue.isEmpty()) {
            UserPropertyCommand.UserPropertyCommandUnit commandUnit = commandQueue.poll();
            if(commandUnit == null){
                AbxLog.w("commandUnit is null", true);
                continue;
            }

            if (validationCheck(commandUnit)) {
                switch (commandUnit.commandType) {
                    case SET:
                        if (!userPropertyMap.containsKey(commandUnit.key) ||
                                !userPropertyMap.get(commandUnit.key).equals(commandUnit.value)) {
                            userPropertyMap.put(commandUnit.key, commandUnit.value);
                            isChanged = true;
                        }
                        break;

                    case UNSET:
                        if (userPropertyMap.containsKey(commandUnit.key)) {
                            userPropertyMap.remove(commandUnit.key);
                            isChanged = true;
                        }
                        break;

                    default:
                }
            }

            try {
                this.dataContext.setValue(
                        IDataContext.KEY_USER_PROPERTY,
                        this.getCurrentUserPropertyModel().getJson().toString()
                );
            } catch (JSONException e) {
                this.logger.error(e.toString());
            }
        }

        if (isChanged) {
            // 새로운 user snapshot id 발급
            refreshUserSnapShotId();
        }

        return isChanged;
    }

    @Override
    public void clear() {
        userPropertyMap.clear();
        dataContext.remove(IDataContext.KEY_USER_PROPERTY);
        refreshUserSnapShotId();
    }

    @Override
    public UserPropertyModel getCurrentUserPropertyModel() {
        //Deep Copy
        HashMap<String, Object> copyUserPropertyMap = new HashMap<>();
        for (String key : this.userPropertyMap.keySet()) {
            copyUserPropertyMap.put(key, this.userPropertyMap.get(key));
        }

        return new UserPropertyModel(
                dataRegistry.safeGetString(DataRegistryKey.STRING_USER_SNAPSHOT_ID, null),
                copyUserPropertyMap
        );
    }

    @Override
    public Object getUserPropertyValue(String key) {
        return this.userPropertyMap.get(key);
    }

    private boolean validationCheck(UserPropertyCommand.UserPropertyCommandUnit commandUnit) {
        return
                validation1(commandUnit) &&
                        validation2(commandUnit) &&
                        validation3(commandUnit) &&
                        validation4(commandUnit);
    }

    private boolean validation1(UserPropertyCommand.UserPropertyCommandUnit commandUnit) {
        //빈값은 사용할 수 없다.
        if (CommonUtils.isNullOrEmpty(commandUnit.key)) {
            this.logger.warn(String.format(LogMessageFormat.USER_PROPERTY_MANAGER_E1, commandUnit.key));
            return false;
        }

        return true;
    }

    private boolean validation2(UserPropertyCommand.UserPropertyCommandUnit commandUnit) {
        //"-"를 사용할 수 없다. 만약 있다면 "_"으로 대체해주자.
        //검증은 없고 대체만 있다.
        //대체를 해주는 이유는 기존 SDK에서 그렇게 해주었다. 갑자기 기준을 바꾸면 고객입장에서는 갑자기 이벤트 수집이 안되는것처럼 보인다 ;_;
        if (commandUnit.key.contains("-")) {
            this.logger.warn(String.format(LogMessageFormat.USER_PROPERTY_MANAGER_E2, commandUnit.key));
            commandUnit.key = commandUnit.key.replace('-', '_');
        }

        return true;
    }

    private boolean validation3(UserPropertyCommand.UserPropertyCommandUnit commandUnit) {
        //50자를 넘을 수 없다.
        if (commandUnit.key.length() > CoreConstants.USER_PROPERTY_MAX_KEY_LENGTH) {
            this.logger.warn(String.format(LogMessageFormat.USER_PROPERTY_MANAGER_E3, commandUnit.key));
            return false;
        }

        return true;
    }

    private boolean validation4(UserPropertyCommand.UserPropertyCommandUnit commandUnit) {
        //소문자 알파벳과 숫자와 언더바, 콜론만 사용할 수 있다.
        if (!Pattern.matches("^[a-z0-9_:]*$", commandUnit.key)) {
            this.logger.warn(String.format(LogMessageFormat.USER_PROPERTY_MANAGER_E4, commandUnit.key));
            return false;
        }

        return true;
    }

    private void refreshUserSnapShotId(){
        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_USER_SNAPSHOT_ID,
                UUID.randomUUID().toString(),
                5,
                this.getClass().getName(),
                true
        ));
    }

    public void deactivate(){
        clear();
    }
}
