package io.adbrix.sdk.ui.inappmessage;

import android.app.Activity;
import android.content.res.Configuration;

import org.json.JSONObject;

import java.lang.ref.SoftReference;
import java.lang.ref.WeakReference;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.IABXComponentsFactory;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.domain.function.Completion;
import io.adbrix.sdk.domain.model.DfnInAppMessage;
import io.adbrix.sdk.domain.model.Empty;
import io.adbrix.sdk.domain.model.Error;
import io.adbrix.sdk.domain.model.IAMEnums;
import io.adbrix.sdk.domain.model.Result;
import io.adbrix.sdk.domain.model.Success;
import io.adbrix.sdk.utils.CommonUtils;

public class InAppMessageManager {
    private SoftReference<IInAppMessageViewFactory> displayedInAppMessageViewFactory;
    private SoftReference<DfnInAppMessage> carriedInAppMessage;
    private WeakReference<Activity> currentActivity;
    private SoftReference<Activity> displayedActivity;
    private final AtomicBoolean isPresented = new AtomicBoolean(false);
    private final AtomicBoolean isInProgress = new AtomicBoolean(false);
    private final AtomicBoolean isPausedByUser = new AtomicBoolean(false);
    private final AtomicBoolean wasUsingKeypad = new AtomicBoolean(false);
    private IABXComponentsFactory factory;
    private ExecutorService executorService;

    public static final String FULL_SCREEN = "full_screen";
    public static final String MODAL = "modal";
    public static final String STICKY_BANNER = "sticky_banner";
    public void showByEventName(String eventName){
        if(!canShowInAppMessage()){
            return;
        }
        DfnInAppMessage inAppMessage = getInAppMessageByEventName(eventName);
        show(inAppMessage, null);
    }
    public void showByEventNameWithParam(String eventName, JSONObject eventParam){
        if(!canShowInAppMessage()){
            return;
        }
        DfnInAppMessage inAppMessage;
        if(CommonUtils.isNullOrEmpty(eventParam)){
            inAppMessage = getInAppMessageByEventName(eventName);
        } else{
            inAppMessage = getInAppMessageByEventNameWithParam(eventName, eventParam);
        }
        show(inAppMessage, null);
    }
    public void showByEventNameWithParamList(String eventName, List<JSONObject> eventParamList){
        if(!canShowInAppMessage()){
            return;
        }
        DfnInAppMessage inAppMessage;
        if(CommonUtils.isNullOrEmpty(eventParamList)){
            inAppMessage = getInAppMessageByEventName(eventName);
        } else{
            inAppMessage = getInAppMessageByEventNameWithParamList(eventName, eventParamList);
        }
        show(inAppMessage, null);
    }
    public void showByCampaignId(String campaignId, Completion<Result<Empty>> completion){
        if(!canShowInAppMessage()){
            return;
        }
        DfnInAppMessage inAppMessage = getInAppMessageByCampaignId(campaignId);
        show(inAppMessage, completion);
    }

    public void show(DfnInAppMessage inAppMessage, Completion<Result<Empty>> completion){
        show(inAppMessage, completion, false);
    }

    public void show(DfnInAppMessage inAppMessage, Completion<Result<Empty>> completion, boolean isCarried){
        ExecutorService executorService = getExecutorService();
        if (executorService == null){
            AbxLog.w("show() inAppMessageExecutorService is null", true);
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("inAppMessageExecutorService is null"));
            }
            return;
        }
        if(executorService.isShutdown()){
            AbxLog.w("show() inAppMessageExecutorService is shutdown", true);
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("inAppMessageExecutorService is shutdown"));
            }
            return;
        }
        if(inAppMessage == null){
            AbxLog.w("show() No Valid InAppMessage", true);
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("No Valid InAppMessage"));
            }
            return;
        }
        if(CommonUtils.isNull(factory)){
            AbxLog.w("show() factory is null", true);
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("factory is null"));
            }
            return;
        }
        if (factory.createOrGetDataRegistry().safeGetLong(DataRegistryKey.LONG_S3_CONFIG_IN_APP_MESSAGE_ACTIVE, -1) != 1) {
            AbxLog.w("show() InAppMessage is not active", true);
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("InAppMessage is not active"));
            }
            return;
        }
        if(isShowing()){
            AbxLog.w("show() InAppMessage is already shown", true);
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("InAppMessage is already shown"));
            }
            return;
        }
        if(CommonUtils.notNull(displayedInAppMessageViewFactory)){
            if(CommonUtils.notNull(displayedInAppMessageViewFactory.get())){
                AbxLog.w("show() inAppMessageViewFactory is not null", true);
                if(CommonUtils.notNull(completion)){
                    completion.handle(Error.of("inAppMessageViewFactory is not null"));
                }
                return;
            }
        }
        if(isInProgress()){
            AbxLog.w("show() InAppMessage is in progress", true);
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("InAppMessage is in progress"));
            }
            return;
        }
        if(isPausedByUser.get()){
            AbxLog.w("InAppMessage is paused. If you want to use, then you should call AdbrixRM.resumeInAppMessage().", true);
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("InAppMessage is paused. If you want to use, then you should call AdbrixRM.resumeInAppMessage()."));
            }
            return;
        }
        Activity activity = getCurrentActivity();
        if(CommonUtils.isNull(activity)){
            AbxLog.w("show() activity is null", true);
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("activity is null"));
            }
            cacheInAppMessage(inAppMessage);
            return;
        }
        setIsInProgress(true);
        executorService.submit(new Runnable() {
            @Override
            public void run() {
                final DfnInAppMessage finalSetInAppMessage;
                final boolean finalSetIsCarried;
                if(!isCarried){
                    DfnInAppMessage carriedInAppMessage = getCarriedInAppMessage();
                    if(CommonUtils.notNull(carriedInAppMessage)){
                        finalSetInAppMessage = carriedInAppMessage;
                        finalSetIsCarried = true;
                        flushCarriedInAppMessage();
                    } else{
                        finalSetIsCarried = isCarried;
                        finalSetInAppMessage = inAppMessage;
                    }
                }else{
                    finalSetIsCarried = isCarried;
                    finalSetInAppMessage = inAppMessage;
                }
                Activity activity = getCurrentActivity();
                if(CommonUtils.isNull(activity)){
                    AbxLog.w("show() activity is null", true);
                    if(CommonUtils.notNull(completion)){
                        completion.handle(Error.of("activity is null"));
                    }
                    cacheInAppMessage(inAppMessage);
                    return;
                }
                AbxLog.d("InAppMessageManager show(). "+System.currentTimeMillis(), true);
                InAppMessageDAO inAppMessageDAO = getInAppMessageDao(activity);
                inAppMessageDAO.increaseFrequencyByCampaignId(finalSetInAppMessage.getCampaignId());
                InAppMessageFactoryContainer inAppMessageFactoryContainer = new InAppMessageFactoryContainer(inAppMessageDAO,factory.getAbxContextController());
                IInAppMessageViewFactory inAppMessageViewFactory = inAppMessageFactoryContainer.getInAppMessageViewFactory(finalSetInAppMessage);
                displayedInAppMessageViewFactory = new SoftReference<>(inAppMessageViewFactory);
                setDisplayedActivity(activity);
                displayedInAppMessageViewFactory.get().createInAppMessage(activity, finalSetInAppMessage, finalSetIsCarried);
                if(CommonUtils.notNull(completion)){
                    completion.handle(Success.empty());
                }
            }
        });
    }

    public void close(IAMEnums.CloseType closeType){
        AbxLog.i("onClose() "+closeType, true);
        if(CommonUtils.isNull(displayedActivity)){
            AbxLog.w("onClose() activity Reference is null", true);
            flushCarriedInAppMessageViewFactory();
            setIsInProgress(false);
            return;
        }
        Activity activity = displayedActivity.get();
        if(CommonUtils.isNull(activity)){
            AbxLog.w("onClose() activity is null", true);
            flushCarriedInAppMessageViewFactory();
            setIsInProgress(false);
            return;
        }
        if(!isShowing()){
            AbxLog.w("onClose() inAppMessage is not shown", true);
        }
        if(CommonUtils.isNull(displayedInAppMessageViewFactory)){
            AbxLog.w("onClose() inAppMessageViewFactory Reference is null", true);
            return;
        }
        if(CommonUtils.isNull(displayedInAppMessageViewFactory.get())){
            AbxLog.w("onClose() inAppMessageViewFactory is null", true);
            return;
        }
        setIsInProgress(true);
        cacheInAppMessage(closeType);
        displayedInAppMessageViewFactory.get().dismissInAppMessage(activity, closeType);
    }

    public void cacheInAppMessage(IAMEnums.CloseType closeType){
        if(CommonUtils.isNull(displayedInAppMessageViewFactory)){
            return;
        }
        if(CommonUtils.isNull(displayedInAppMessageViewFactory.get())){
            return;
        }
        if(CommonUtils.isNull(displayedInAppMessageViewFactory.get().getInAppMessage())){
            return;
        }
        switch (closeType){
            case ON_PAUSE:
            case OPEN_DEEPLINK:{
                this.carriedInAppMessage = new SoftReference<>(displayedInAppMessageViewFactory.get().getInAppMessage());
                break;
            }
            default:{
                flushCarriedInAppMessage();
            }
        }
    }
    private void cacheInAppMessage(DfnInAppMessage inAppMessage){
        this.carriedInAppMessage = new SoftReference<>(inAppMessage);
    }

    public boolean isShowing(){
        return isPresented.get();
    }
    public void setShowing(boolean isShown){
        isPresented.set(isShown);
    }

    public void setIsInProgress(boolean isInProgress) {
        this.isInProgress.set(isInProgress);
    }

    public boolean isInProgress() {
        return isInProgress.get();
    }

    public Activity getCurrentActivity() {
        if(currentActivity == null){
            return null;
        }
        return currentActivity.get();
    }
    public Activity getDisplayedActivity() {
        if(displayedActivity == null){
            return null;
        }
        return displayedActivity.get();
    }

    public void setCurrentActivity(Activity currentActivity) {
        if(currentActivity == null){
            if(CommonUtils.notNull(this.currentActivity)){
                this.currentActivity.clear();
            }
            return;
        }
        if(CommonUtils.notNull(this.currentActivity)){
            this.currentActivity.clear();
        }
        this.currentActivity = new WeakReference<>(currentActivity);
    }
    public void setDisplayedActivity(Activity activity){
        this.displayedActivity = new SoftReference<>(activity);
    }

    public void init(IABXComponentsFactory factory) {
        this.factory = factory;
    }

    private ExecutorService getExecutorService(){
        if (executorService == null || executorService.isShutdown()){
            executorService = Executors.newSingleThreadExecutor();
            return executorService;
        }
        return executorService;
    }

    public DfnInAppMessage getCarriedInAppMessage() {
        if(CommonUtils.isNull(carriedInAppMessage)){
            return null;
        }
        return carriedInAppMessage.get();
    }

    public void flushCarriedInAppMessageViewFactory(){
        if(CommonUtils.notNull(displayedInAppMessageViewFactory)){
            displayedInAppMessageViewFactory.clear();
            displayedInAppMessageViewFactory = null;
        }
    }
    public void flushCarriedInAppMessage(){
        if(CommonUtils.notNull(carriedInAppMessage)){
            carriedInAppMessage.clear();
            carriedInAppMessage = null;
        }
    }
    public void flushDisplayedActivity(){
        if(CommonUtils.notNull(displayedActivity)){
            displayedActivity.clear();
            displayedActivity = null;
        }
    }

    private DfnInAppMessage getInAppMessageByCampaignId(String campaignId){
        if(!isInAppMessageActive()){
            return null;
        }
        Activity activity = getCurrentActivity();
        if(CommonUtils.isNull(activity)){
            AbxLog.d("getInAppMessageByCampaignId activity is null", true);
            return null;
        }
        if(CommonUtils.isNull(factory)){
            AbxLog.d("factory is null", true);
            return null;
        }

        InAppMessageDAO inAppMessageDAO = getInAppMessageDao(activity);
        DfnInAppMessage inAppMessage = null;
        try {
            inAppMessage = inAppMessageDAO.getInAppMessageByCampaignId(campaignId);
        }catch (Exception e){
            AbxLog.w(e, true);
        }
        return inAppMessage;
    }

    public DfnInAppMessage getInAppMessageByEventName(String eventName){
        if(!isInAppMessageActive()){
            return null;
        }
        Activity activity = getCurrentActivity();
        if(CommonUtils.isNull(activity)){
            AbxLog.d("getInAppMessageByEventName activity is null", true);
            return null;
        }
        if(CommonUtils.isNull(factory)){
            AbxLog.d("factory is null", true);
            return null;
        }
        InAppMessageDAO inAppMessageDAO = getInAppMessageDao(activity);
        DfnInAppMessage inAppMessage = null;
        try {
            inAppMessage = inAppMessageDAO.getInAppMessageByEventName(eventName);
        }catch (Exception e){
            AbxLog.w(e, true);
        }
        return inAppMessage;
    }

    public DfnInAppMessage getInAppMessageByEventNameWithParam(String eventName, JSONObject eventParam){
        if(!isInAppMessageActive()){
            return null;
        }
        Activity activity = getCurrentActivity();
        if(CommonUtils.isNull(activity)){
            AbxLog.d("getInAppMessageByEventNameWithParam activity is null", true);
            return null;
        }
        if(CommonUtils.isNull(factory)){
            AbxLog.d("factory is null", true);
            return null;
        }

        InAppMessageDAO inAppMessageDAO = getInAppMessageDao(activity);
        DfnInAppMessage inAppMessage = null;
        try {
            inAppMessage = inAppMessageDAO.getInAppMessageByEventNameWithParam(eventName, eventParam);
        }catch (Exception e){
            AbxLog.w(e, true);
        }
        return inAppMessage;
    }

    public DfnInAppMessage getInAppMessageByEventNameWithParamList(String eventName, List<JSONObject> eventParamList){
        if(!isInAppMessageActive()){
            return null;
        }
        Activity activity = getCurrentActivity();
        if(CommonUtils.isNull(activity)){
            AbxLog.d("getInAppMessageByEventNameWithParamList activity is null", true);
            return null;
        }
        if(CommonUtils.isNull(factory)){
            AbxLog.d("factory is null", true);
            return null;
        }

        InAppMessageDAO inAppMessageDAO = getInAppMessageDao(activity);
        DfnInAppMessage inAppMessage = null;
        for(JSONObject eventParam : eventParamList){
            try {
                inAppMessage = inAppMessageDAO.getInAppMessageByEventNameWithParam(eventName, eventParam);
            }catch (Exception e){
                AbxLog.w(e, true);
            }
            if(CommonUtils.notNull(inAppMessage)){
                break;
            }
        }
        return inAppMessage;
    }

    public void setPause(){
        isPausedByUser.set(true);
    }
    public void setResume(){
        isPausedByUser.set(false);
    }
    private boolean isInAppMessageActive(){
        boolean result = false;
        if(CommonUtils.isNull(factory)){
            return result;
        }
        if(CommonUtils.isNull(factory.createOrGetDataRegistry())){
            return result;
        }
        if (factory.createOrGetDataRegistry().safeGetLong(DataRegistryKey.LONG_S3_CONFIG_IN_APP_MESSAGE_ACTIVE, -1) != 1) {
            return result;
        }
        result = true;
        return result;
    }

    private InAppMessageDAO getInAppMessageDao(Activity activity){
        InAppMessageDAO inAppMessageDAO = new InAppMessageDAO(activity.getApplicationContext(), factory.createOrGetDataRegistry());
        return inAppMessageDAO;
    }

    private boolean isPortrait(Activity activity){
        boolean result = true;
        try {
            result = activity.getResources().getConfiguration().orientation == Configuration.ORIENTATION_PORTRAIT;
        }catch (Exception e){
            AbxLog.w(e, true);
        }
        return result;
    }

    public static InAppMessageManager getInstance() {
        return InAppMessageManager.Singleton.INSTANCE;
    }
    private static class Singleton {
        private static final InAppMessageManager INSTANCE = new InAppMessageManager();
    }

    private boolean canShowInAppMessage(){
        boolean result = false;
        ExecutorService executorService = getExecutorService();
        if (executorService == null){
            AbxLog.w("canShowInAppMessage() inAppMessageExecutorService is null", true);
            return result;
        }
        if(executorService.isShutdown()){
            AbxLog.w("canShowInAppMessage() inAppMessageExecutorService is shutdown", true);
            return result;
        }
        if(CommonUtils.isNull(factory)){
            AbxLog.w("canShowInAppMessage() factory is null", true);
            return result;
        }
        if (factory.createOrGetDataRegistry().safeGetLong(DataRegistryKey.LONG_S3_CONFIG_IN_APP_MESSAGE_ACTIVE, -1) != 1) {
            AbxLog.w("canShowInAppMessage() InAppMessage is not active", true);
            return result;
        }
        if(isShowing()){
            AbxLog.w("shcanShowInAppMessageow() InAppMessage is already shown", true);
            return result;
        }
        if(CommonUtils.notNull(displayedInAppMessageViewFactory)){
            if(CommonUtils.notNull(displayedInAppMessageViewFactory.get())){
                AbxLog.w("canShowInAppMessage() inAppMessageViewFactory is not null", true);
                return result;
            }
        }
        if(isInProgress()){
            AbxLog.w("canShowInAppMessage() InAppMessage is in progress", true);
            return result;
        }
        if(isPausedByUser.get()){
            AbxLog.w("InAppMessage is paused. If you want to use, then you should call AdbrixRM.resumeInAppMessage().", true);
            return result;
        }
        result = true;
        return result;
    }
    public void setUsingKeypad(boolean isUsingKeypad){
        wasUsingKeypad.set(isUsingKeypad);
    }
    public boolean wasUsingKeypad(){
        return wasUsingKeypad.get();
    }
}
