package io.adbrix.sdk.data.net;

import android.os.Build;

import java.io.IOException;
import java.net.InetAddress;
import java.net.Socket;
import java.net.UnknownHostException;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;

import io.adbrix.sdk.component.AbxLog;

public class TlsSocketFactory extends SSLSocketFactory {
    private SSLSocketFactory internalSSLSocketFactory;

    public TlsSocketFactory(TrustManager[] trustManager, SecureRandom secureRandom) throws KeyManagementException, NoSuchAlgorithmException {
        SSLContext context = SSLContext.getInstance("TLS");
        context.init(null, trustManager, secureRandom);
        internalSSLSocketFactory = context.getSocketFactory();
    }

    @Override
    public String[] getDefaultCipherSuites() {
        return internalSSLSocketFactory.getDefaultCipherSuites();
    }

    @Override
    public String[] getSupportedCipherSuites() {
        return internalSSLSocketFactory.getSupportedCipherSuites();
    }

    @Override
    public Socket createSocket() throws IOException {
        return enableTLSOnSocket(internalSSLSocketFactory.createSocket());
    }

    @Override
    public Socket createSocket(Socket s, String host, int port, boolean autoClose) throws IOException {
        return enableTLSOnSocket(internalSSLSocketFactory.createSocket(s, host, port, autoClose));
    }

    @Override
    public Socket createSocket(String host, int port) throws IOException, UnknownHostException {
        return enableTLSOnSocket(internalSSLSocketFactory.createSocket(host, port));
    }

    @Override
    public Socket createSocket(String host, int port, InetAddress localHost, int localPort) throws IOException, UnknownHostException {
        return enableTLSOnSocket(internalSSLSocketFactory.createSocket(host, port, localHost, localPort));
    }

    @Override
    public Socket createSocket(InetAddress host, int port) throws IOException {
        return enableTLSOnSocket(internalSSLSocketFactory.createSocket(host, port));
    }

    @Override
    public Socket createSocket(InetAddress address, int port, InetAddress localAddress, int localPort) throws IOException {
        return enableTLSOnSocket(internalSSLSocketFactory.createSocket(address, port, localAddress, localPort));
    }

    /**
     * 2022.07.11 bobos
     * Android 4.1의 TLS 지원을 위한 코드
     * API LEVEL 16~20은 강제로 enable 해야 TLS를 지원함
     * @see "https://developer.android.com/reference/javax/net/ssl/SSLSocket#default-configuration-for-different-android-versions"
     * @param socket
     * @return
     */
    private Socket enableTLSOnSocket(Socket socket) {
        try {
            if(Build.VERSION.SDK_INT >= 16) {
                if(Build.VERSION.SDK_INT <= 20){
                    if(socket != null && (socket instanceof SSLSocket)) {
                        ((SSLSocket)socket).setEnabledProtocols(new String[] {"TLSv1","TLSv1.1", "TLSv1.2"});
                    }
                }
            }
        }catch (Exception e){
            AbxLog.e(e, true);
        }
        return socket;
    }
}
