package io.adbrix.sdk.utils;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;

import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509TrustManager;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.net.ApiConnection;
import io.adbrix.sdk.data.net.TlsSocketFactory;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;

public class HttpConnectionUtils {
    public static HttpURLConnection createConnection(String urlString, ApiConnection.RequestMethod requestMethod, DataRegistry dataRegistry) throws IOException{
        URL url = new URL(urlString);
        HttpURLConnection connection = (HttpURLConnection) url.openConnection();
        /**
         * Read Timeout 발생시 SDK는 실패로 간주하나 HttpUrlConnection에 자체 재시도 로직이 있어 서버에는 데이터가 적재됨
         * 이로 인해 중복된 logId를 가진 이벤트가 적재되는 현상이 발생
         * 대응 코드
         * - setChunkedStreamingMode(0)
         * - setReadTimeout(0)
         * @see "https://stackoverflow.com/questions/27094544/android-java-httpurlconnection-silent-retry-on-read-timeout"
         */
        int readTimeout = 0;
        int connectionTimeout = CoreConstants.NETWORK_TIMEOUT;
        connection.setChunkedStreamingMode(0);
        connection.setReadTimeout(readTimeout);
        connection.setConnectTimeout(connectionTimeout);
        if(isDeferredDeeplinkRequest(urlString, dataRegistry)){
            connection.setConnectTimeout(CoreConstants.NETWORK_TIMEOUT_FOR_DEFERRED_DEEPLINK);
        }

        if (urlString.startsWith("https")) {
            configureHttpsConnection((HttpsURLConnection) connection);
        }
        connection.setRequestMethod(requestMethod.getMethodString());
        connection.setInstanceFollowRedirects(true);

        return connection;
    }

    private static boolean isDeferredDeeplinkRequest(String urlString, DataRegistry dataRegistry){
        boolean result = false;
        if(dataRegistry == null){
            return result;
        }
        String appKey = dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY, null);
        String deferredDeeplinkUrlString = String.format(CoreConstants.deferredDeeplinkUrlReq, appKey);
        if(urlString.equals(deferredDeeplinkUrlString)){
            result = true;
            return result;
        }
        return result;
    }

    private static void configureHttpsConnection(HttpsURLConnection connection) {
        /**
         * 2022.07.11 bobos
         * 앱에서 setDefaultSSLSocketFactory를 호출했을 경우 통신에 문제가 생기는 것을 방지하기 위함
         */
        SSLSocketFactory sslSocketFactory = getSSLSocketFactory();
        if(sslSocketFactory == null){
            return;
        }
        connection.setSSLSocketFactory(sslSocketFactory);
    }

    public static SSLSocketFactory getSSLSocketFactory() {
        SSLSocketFactory sslSocketFactory = null;
        TrustManager[] trustManagers = getPlatformTrustManager();
        if(trustManagers == null){
            return null;
        }
        try {
            sslSocketFactory = new TlsSocketFactory(trustManagers, new java.security.SecureRandom());
        } catch (KeyManagementException e) {
            AbxLog.e(e, true);
        } catch (NoSuchAlgorithmException e) {
            AbxLog.e(e, true);
        }
        return sslSocketFactory;
    }

    /**
     * 2022.07.11 bobos
     * OS 기본 값을 가져와서 Google Play 안전하지 않은 TrustManager 경고 대응
     * @see "https://github.com/square/okhttp/blob/3ad1912f783e108b3d0ad2c4a5b1b89b827e4db9/okhttp/src/jvmMain/kotlin/okhttp3/internal/platform/Platform.kt"
     * @return
     */
    public static TrustManager[] getPlatformTrustManager() {
        TrustManagerFactory trustManagerFactory = null;
        try {
            trustManagerFactory = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
            trustManagerFactory.init((KeyStore) null);
        } catch (NoSuchAlgorithmException e) {
            AbxLog.e(e, true);
        } catch (KeyStoreException e) {
            AbxLog.e(e, true);
        }
        if(trustManagerFactory == null){
            AbxLog.e("trustManagerFactory is null", true);
            return null;
        }
        TrustManager[] trustManagers = trustManagerFactory.getTrustManagers();
        if (trustManagers.length != 1 || !(trustManagers[0] instanceof X509TrustManager)) {
            AbxLog.e("Unexpected default trust managers:" + Arrays.toString(trustManagers), true);
            return null;
        }
        return trustManagers;
    }
}
