package io.adbrix.sdk.component;

import com.igaworks.v2.core.AdBrixRm;

import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;

import io.adbrix.sdk.configuration.AbxFacade;
import io.adbrix.sdk.data.dataprovider.INoWaitEventNameProvider;
import io.adbrix.sdk.data.modelprovider.CommonModelProvider;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.function.Completion;
import io.adbrix.sdk.domain.model.CommonModel;
import io.adbrix.sdk.domain.model.Empty;
import io.adbrix.sdk.domain.model.EventData;
import io.adbrix.sdk.domain.model.EventModel;
import io.adbrix.sdk.domain.model.EventPackage;
import io.adbrix.sdk.domain.model.Response;
import io.adbrix.sdk.domain.model.Result;
import io.adbrix.sdk.domain.model.Error;
import io.adbrix.sdk.domain.model.Success;
import io.adbrix.sdk.utils.CommonUtils;


public class EventBuffer implements IObserver<Void> {

    private final EventUploadIntervalManager eventUploadIntervalManager;
    private final EventPackageContainer eventPackageContainer;
    private final INoWaitEventNameProvider noWaitEventNameProvider;

    public EventBuffer(
            EventUploadIntervalManager eventUploadIntervalManager,
            EventPackageContainer eventPackageContainer,
            INoWaitEventNameProvider noWaitEventNameProvider
    ) {
        this.eventPackageContainer = eventPackageContainer;
        this.noWaitEventNameProvider = noWaitEventNameProvider;
        this.eventUploadIntervalManager = eventUploadIntervalManager;
        this.eventUploadIntervalManager.add(this);
    }

    public void addEventModel(EventModel model) {
        addEventModel(model, null);
    }

    public void addEventModel(EventModel model, Completion<Result<Response>> completion) {
        Queue<EventModel> buffer = new LinkedList<>();
        buffer.add(model);
        sendEventToEventListener(model);
        sendEventBuffer(buffer, completion);
    }

    public void addEventModels(List<EventModel> eventModelList){
        Queue<EventModel> buffer = new LinkedList<>();
        int countInterval = eventUploadIntervalManager.getCountInterval();
        int loopCount = eventModelList.size()/countInterval+1;
        int lastLoopEventSize = eventModelList.size()%countInterval;
        //item이 많아서 나뉘어진 이벤트의 경우
        for(int i=0; i<loopCount; i++){
            if(i == loopCount-1){
                for(int j=0; j<lastLoopEventSize; j++){
                    int index = (i*countInterval)+j;
                    EventModel model = eventModelList.get(index);
                    buffer.add(model);
                    sendEventToEventListener(model);
                }
                sendEventBuffer(buffer, null);
            } else{
                for(int j=0; j<countInterval; j++){
                    int index = (i*countInterval)+j;
                    EventModel model = eventModelList.get(index);
                    buffer.add(model);
                    sendEventToEventListener(model);
                }
                sendEventBuffer(buffer, null);
            }
        }
    }
    public void sendEventBuffer(Queue<EventModel> buffer, Completion<Result<Response>> completion){
        if(CommonUtils.isNullOrEmpty(buffer)){
            AbxLog.d("buffer is null", true);
            return;
        }
        Queue<EventModel> mergedEventModelsWithUnsentEvents = getMergedEventsWithUnsentEvents(buffer);
        EventModel firstEventModel = buffer.peek();
        if(CommonUtils.isNull(firstEventModel)){
            AbxLog.d("firstEventModel is null", true);
            return;
        }
        if (noWaitEventNameProvider.isNowaitEvent(firstEventModel)){
            //바로 보내야하는 이벤트
            if(CommonUtils.isNull(completion)){
                clearBufferAndFlushContainer(mergedEventModelsWithUnsentEvents, emptyResult -> {});
                return;
            }
            clearBufferAndFlushContainer(mergedEventModelsWithUnsentEvents, completion);
            return;
        }
        //보내지 않고 저장함
        eventPackageContainer.updateUnsentEventPackagesInDataRegistry(new EventPackage(mergedEventModelsWithUnsentEvents));
    }
    private Queue<EventModel> getMergedEventsWithUnsentEvents(Queue<EventModel> buffer){
        Queue<EventModel> mergedEventModels = new LinkedList<>();
        mergedEventModels.addAll(getUnsentEvents());
        mergedEventModels.addAll(buffer);
        return mergedEventModels;
    }

    private Queue<EventModel> getUnsentEvents(){
        Queue<EventPackage> unsentEventPackage = eventPackageContainer.getPreviousUnformattedPackageQueue();
        Queue<EventModel> unsentEventModels = eventPackageContainer.getEventModelsFromEventPackage(unsentEventPackage);
        return unsentEventModels;
    }

    public void flushAllNow(Completion<Result<Empty>> completion) {
        clearBufferAndFlushContainer(new Completion<Result<Response>>() {
            @Override
            public void handle(Result<Response> responseResult) {
                completion.handle(Success.empty());
            }
        });
    }

    public void flushContainerAtIntervals() {
        this.eventPackageContainer.flushAtIntervals();
    }

    @Override
    public void update(Void v) {
        flushAllNow(emptyResult -> {});
    }

    public void saveUnsentEvents() {
        // 이벤트 호출 후 그때마다 저장
    }

    private void clearBufferAndFlushContainer(Completion<Result<Response>> completion) {
        Queue<EventModel> buffer = getUnsentEvents();
        if(CommonUtils.isNullOrEmpty(buffer)){
            completion.handle(Error.of("unsentEvents is null or empty") );
            return;
        }
        eventPackageContainer.flushEventsNotSent(new EventPackage(buffer), completion);
    }

    private void clearBufferAndFlushContainer(Queue<EventModel> buffer, Completion<Result<Response>> completion) {
        eventPackageContainer.flushEventsNotSent(new EventPackage(buffer), completion);
    }

    private void sendEventToEventListener(EventModel eventModel){
        AdBrixRm.EventListener eventListener = AdBrixRm.getEventListener();
        if(CommonUtils.isNull(eventListener)){
            return;
        }
        if(isBlockedEventForEventListener(eventModel)){
            return;
        }
        CommonModel commonModel = eventPackageContainer.getEventSender().getCommonModel();
        EventData eventData = new EventData(eventModel, commonModel);
        eventListener.onEvent(eventData);
    }
    private boolean isBlockedEventForEventListener(EventModel eventModel){
        boolean result = false;
        if(CommonUtils.isNull(eventModel)){
            return result;
        }
        String eventName = eventModel.eventName;
        if(CommonUtils.isNullOrEmpty(eventName)){
            return result;
        }
        eventName = eventName.replace("abx:","");
        if(CoreConstants.EVENT_SET_PUSH.equals(eventName)){
            result = true;
            return result;
        }
        if(CoreConstants.EVENT_ADID_CHANGED.equals(eventName)){
            result = true;
            return result;
        }
        return result;
    }

    public void deactivate(){

    }
}
