package io.adbrix.sdk.utils.image.download;

import android.content.Context;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.data.net.ApiConnection;
import io.adbrix.sdk.utils.HttpConnectionUtils;
import io.adbrix.sdk.utils.IoUtils;

public class DefaultImageDownloader implements ImageDownloader{
    protected static final int BUFFER_SIZE = 32 * 1024; // 32 Kb
    protected static final int MAX_REDIRECT_COUNT = 5;
    private final Context context;
    private HttpURLConnection conn;

    public DefaultImageDownloader(Context context) {
        this.context = context;
    }

    @Override
    public InputStream getStream(String imageUri) throws IOException {
        switch (Scheme.ofUri(imageUri)) {
            case HTTP:
            case HTTPS:
                return getStreamFromNetwork(imageUri);
            case DRAWABLE:
                return getStreamFromDrawable(imageUri);
            case UNKNOWN:
            default:
                return getStreamFromOtherSource(imageUri);
        }
    }

    @Override
    public void close() {
        if(conn!=null){
            conn.disconnect();
        }
    }

    private InputStream getStreamFromNetwork(String imageUri) throws IOException {
        conn = HttpConnectionUtils.createConnection(imageUri, ApiConnection.RequestMethod.GET, null);

        int redirectCount = 0;
        while (conn.getResponseCode() / 100 == 3 && redirectCount < MAX_REDIRECT_COUNT) {
            conn = HttpConnectionUtils.createConnection(conn.getHeaderField("Location"), ApiConnection.RequestMethod.GET, null);
            redirectCount++;
        }

        InputStream imageStream;
        try {
            imageStream = conn.getInputStream();
        } catch (IOException e) {
            // Read all data to allow reuse connection (http://bit.ly/1ad35PY)
            IoUtils.readAndCloseStream(conn.getErrorStream());
            throw e;
        }
        if (!shouldBeProcessed(conn)) {
            IoUtils.closeSilently(imageStream);
            throw new IOException("Image request failed with response code " + conn.getResponseCode());
        }
        final int contentLength = conn.getContentLength();
        return new ContentLengthInputStream(new BufferedInputStream(imageStream, BUFFER_SIZE), contentLength);
    }

    private InputStream getStreamFromDrawable(String imageUri) {
        String drawableIdString = Scheme.DRAWABLE.crop(imageUri);
        int drawableId = Integer.parseInt(drawableIdString);
        InputStream inputStream = null;
        try {
            inputStream = context.getResources().openRawResource(drawableId);
        } catch (Exception e){
            AbxLog.w(e, true);
        }
        return inputStream;
    }

    private InputStream getStreamFromOtherSource(String imageUri) throws IOException {
        throw new UnsupportedOperationException("doesn't supported uri. "+imageUri);
    }

    private boolean shouldBeProcessed(HttpURLConnection conn) throws IOException {
        return conn.getResponseCode() == 200;
    }
}