package io.adbrix.sdk.utils.image.decode;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;

import java.io.IOException;
import java.io.InputStream;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.utils.IoUtils;
import io.adbrix.sdk.utils.image.download.DefaultImageDownloader;

public class DefaultImageDecoder implements ImageDecoder{
    private Context context;

    public DefaultImageDecoder(Context context) {
        this.context = context;
    }

    @Override
    public Bitmap decode(String uri, int reqWidth, int reqHeight) throws IOException {
        DefaultImageDownloader imageDownloader = new DefaultImageDownloader(this.context);
        InputStream inputStream = getImageStream(uri, imageDownloader);
        if (inputStream == null) {
            AbxLog.d( "imageStream is null", true);
            return null;
        }
        Bitmap decodedBitmap = null;
        try {
            // First decode with inJustDecodeBounds=true to check dimensions
            final BitmapFactory.Options options = new BitmapFactory.Options();
            options.inJustDecodeBounds = true;
            BitmapFactory.decodeStream(inputStream, null, options);

            // Calculate inSampleSize
            options.inSampleSize = calculateInSampleSize(options, reqWidth, reqHeight);
            inputStream = resetStream(inputStream, uri, imageDownloader);
            // Decode bitmap with inSampleSize set
            options.inJustDecodeBounds = false;
            decodedBitmap = BitmapFactory.decodeStream(inputStream, null, options);
            if(decodedBitmap == null){
                AbxLog.e("can't decode image", true);
            }else{
                AbxLog.d("bitmap size - width: " +decodedBitmap.getWidth() + ", height: " + decodedBitmap.getHeight(), true);
            }
        }finally {
            IoUtils.closeSilently(inputStream);
            imageDownloader.close();
        }
        return decodedBitmap;
    }

    private int calculateInSampleSize(BitmapFactory.Options options, int reqWidth, int reqHeight) {
        // Raw height and width of image
        final int height = options.outHeight;
        final int width = options.outWidth;
        int inSampleSize = 1;

        if (height > reqHeight || width > reqWidth) {

            final int halfHeight = height / 2;
            final int halfWidth = width / 2;

            // Calculate the largest inSampleSize value that is a power of 2 and keeps both
            // height and width larger than the requested height and width.
            while ((halfHeight / inSampleSize) >= reqHeight
                    && (halfWidth / inSampleSize) >= reqWidth) {
                inSampleSize *= 2;
            }
        }
        AbxLog.d( "scale = " + inSampleSize + ", orig-width: " + options.outWidth + ", orig-height: " + options.outHeight, true);
        return inSampleSize;
    }

    private InputStream getImageStream(String uri, DefaultImageDownloader downloader) throws IOException {
        return downloader.getStream(uri);
    }

    private InputStream resetStream(InputStream imageStream, String uri, DefaultImageDownloader downloader) throws IOException {
        if (imageStream.markSupported()) {
            try {
                imageStream.reset();
                return imageStream;
            } catch (IOException ignored) {
            }
        }
        IoUtils.closeSilently(imageStream);
        return getImageStream(uri, downloader);
    }
}
