package io.adbrix.sdk.utils;

import java.util.Arrays;

public class Base64 {

    private Base64() {
    }
    //Mapping table from 6-bit nibbles to Base64 characters.
    private static char[] map1;
    //Mapping table from Base64 characters to 6-bit nibbles.
    private static byte[] map2;

    static {
        map1 = new char[64];

        int i = 0;

        for (char c = 'A'; c <= 'Z'; c++) {
            map1[i++] = c;
        }

        for (char c = 'a'; c <= 'z'; c++) {
            map1[i++] = c;
        }

        for (char c = '0'; c <= '9'; c++) {
            map1[i++] = c;
        }

        map1[i++] = '+';
        map1[i++] = '/';
    }

    static {
        map2 = new byte[128];
        Arrays.fill(map2, (byte) -1);

        for (int i = 0; i < 64; i++) {
            map2[map1[i]] = (byte) i;
        }
    }

    public static String encode(String string) {
        return CommonUtils.isNullOrEmpty(string) ? "" : new String(encode(string.getBytes()));
    }

    private static char[] encode(byte[] input) {
        int inputLength = input.length;
        int outputLengthWithoutPadding = (inputLength * 4 + 2) / 3;
        int outputLengthIncludingPadding = ((inputLength + 2) / 3) * 4;
        char[] output = new char[outputLengthIncludingPadding];

        int inputPointer = 0;
        int outputPointer = 0;
        while (inputPointer < inputLength) {
            int i0 = input[inputPointer++] & 0xff;
            int i1 = inputPointer < inputLength ? input[inputPointer++] & 0xff : 0;
            int i2 = inputPointer < inputLength ? input[inputPointer++] & 0xff : 0;

            int o0 = i0 >>> 2;
            int o1 = ((i0 & 0x3) << 4) | (i1 >>> 4);
            int o2 = ((i1 & 0xf) << 2) | (i2 >>> 6);
            int o3 = i2 & 0x3F;

            output[outputPointer++] = map1[o0];
            output[outputPointer++] = map1[o1];
            output[outputPointer] = outputPointer < outputLengthWithoutPadding ? map1[o2] : '=';
            outputPointer++;
            output[outputPointer] = outputPointer < outputLengthWithoutPadding ? map1[o3] : '=';
            outputPointer++;
        }

        return output;
    }

    /**
     * Decodes a string from Base64 format.
     *
     * @param string a Base64 String to be decoded.
     * @return A String containing the decoded data.
     * @throws IllegalArgumentException if the input is not valid Base64 encoded data.
     */
    public static String decode(String string) {
        return CommonUtils.isNullOrEmpty(string) ? "" : new String(decode(string.toCharArray()));
    }

    /**
     * Decodes a byte array from Base64 format.
     * No blanks or line breaks are allowed within the Base64 encoded data.
     *
     * @param input a character array containing the Base64 encoded data.
     * @return An array containing the decoded data bytes.
     * @throws IllegalArgumentException if the input is not valid Base64 encoded data.
     */
    private static byte[] decode(char[] input) {
        int inputLength = input.length;
        if (inputLength % 4 != 0)
            throw new IllegalArgumentException("Length of Base64 encoded input string is not a multiple of 4.");

        while (inputLength > 0 && input[inputLength - 1] == '=') {
            inputLength--;
        }

        int outputLength = (inputLength * 3) / 4;
        byte[] output = new byte[outputLength];

        int inputPointer = 0;
        int outputPointer = 0;
        while (inputPointer < inputLength) {
            int i0 = input[inputPointer++];
            int i1 = input[inputPointer++];
            int i2 = inputPointer < inputLength ? input[inputPointer++] : 'A';
            int i3 = inputPointer < inputLength ? input[inputPointer++] : 'A';

            if (i0 > 127 || i1 > 127 || i2 > 127 || i3 > 127)
                throw new IllegalArgumentException("Illegal character in Base64 encoded data.");

            int b0 = map2[i0];
            int b1 = map2[i1];
            int b2 = map2[i2];
            int b3 = map2[i3];

            if (b0 < 0 || b1 < 0 || b2 < 0 || b3 < 0)
                throw new IllegalArgumentException("Illegal character in Base64 encoded data.");

            int o0 = (b0 << 2) | (b1 >>> 4);
            int o1 = ((b1 & 0xf) << 4) | (b2 >>> 2);
            int o2 = ((b2 & 3) << 6) | b3;

            output[outputPointer++] = (byte) o0;
            if (outputPointer < outputLength) {
                output[outputPointer++] = (byte) o1;
            }

            if (outputPointer < outputLength) {
                output[outputPointer++] = (byte) o2;
            }
        }

        return output;
    }
}
