package io.adbrix.sdk.ui.push.models;

import android.content.Context;
import android.os.Bundle;

import com.google.firebase.messaging.RemoteMessage;

import org.json.JSONException;
import org.json.JSONObject;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.domain.ABXConstants;
import io.adbrix.sdk.domain.CompatConstants;
import io.adbrix.sdk.ui.push.utils.PushUtils;
import io.adbrix.sdk.utils.CommonUtils;

public class AbxRemotePushModel {
    public String title = "";
    public String body = "";
    public String bigTextTitle = "";
    public String bigTextBody = "";
    public String summaryText = "";
    public String imageUrl = "";
    public String largeIconUrl = "";
    public boolean genWhileRun = false;
    public boolean genVibe = false;
    public boolean genSound = false;
    public String deeplinkUrl = "";
    public JSONObject deeplinkJson = new JSONObject();

    public String campaignId = "";
    public int campaignRevisionNo = 0;
    public String stepId = "";
    public String cycleTime = "";
    public int notificationId = 0;

    public AbxRemotePushModel(){}

    public AbxRemotePushModel(Context context, RemoteMessage remoteMessage) {
        if(remoteMessage == null){
            AbxLog.e("remoteMessage is null", true);
            return;
        }
        Bundle bundle = CommonUtils.convertToBundle(remoteMessage.getData());
        String properties = bundle.getString(ABXConstants.PUSH_REMOTE_FCM_KEY);
        if(CommonUtils.isNullOrEmpty(properties)){
            AbxLog.w("ERROR :: notification properties is empty", false);
            return;
        }
        JSONObject objFromServer;
        try {
            objFromServer = new JSONObject(properties);
        } catch (JSONException e) {
            AbxLog.e("Adbrix push data don't exist!",e, true);
            return;
        }
        JSONObject jsonObject = new JSONObject();
        try {
            jsonObject.put(ABXConstants.PUSH_REMOTE_FCM_KEY, objFromServer);
            AbxLog.i("remotePushPayload: "+jsonObject.toString(), true);
        } catch (JSONException e) {
            AbxLog.e(e, true);
        }
        initWithJsonObject(context, jsonObject);
    }

    public AbxRemotePushModel(Context context, Bundle bundle){
        if(CommonUtils.isNullOrEmpty(bundle)){
            AbxLog.e("bundle is empty", false);
            return;
        }
        String properties = bundle.getString(ABXConstants.PUSH_REMOTE_FCM_KEY);
        if(CommonUtils.isNullOrEmpty(properties)){
            AbxLog.w("ERROR :: notification properties is empty", false);
            return;
        }

        JSONObject jsonObject = new JSONObject();
        try {
            JSONObject propertiesJsonObject = new JSONObject(properties);
            jsonObject.put(ABXConstants.PUSH_REMOTE_FCM_KEY, propertiesJsonObject);
            AbxLog.i("remotePushPayload: "+jsonObject.toString(), true);
        } catch (JSONException e){
            AbxLog.e(e, true);
        }
        initWithJsonObject(context, jsonObject);
    }

    public AbxRemotePushModel(Context context, JSONObject pushJson) {
        initWithJsonObject(context, pushJson);
    }

    public AbxRemotePushModel(String campaignId, int campaignRevisionNo, String stepId, String cycleTime){
        this.campaignId = campaignId;
        this.campaignRevisionNo = campaignRevisionNo;
        this.stepId = stepId;
        this.cycleTime = cycleTime;
    }

    private void initWithJsonObject(Context context, JSONObject jsonObject){
        JSONObject objFromServer = new JSONObject();

        try {
            objFromServer = jsonObject.getJSONObject(ABXConstants.PUSH_REMOTE_FCM_KEY);
        } catch (JSONException e) {
            AbxLog.e("Adbrix push data don't exist!", e, true);
            return;
        }

        try {
            this.campaignId = objFromServer.getString(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_ID);
            this.campaignRevisionNo = objFromServer.getInt(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_REVISION_NO);
            this.stepId = objFromServer.getString(ABXConstants.GROWTH_EVENT_KEY_STEP_ID);
            this.cycleTime = objFromServer.getString(ABXConstants.GROWTH_EVENT_KEY_CYCLE_TIME);
        } catch (JSONException e) {
            AbxLog.e("Adbrix push tracking parameters don't exist!",e, true);
            return;
        }

        try {
            if (PushUtils.checkHasVaildKey(objFromServer, ABXConstants.PUSH_REMOTE_KEY_DEEPLINKURL)) {
                this.deeplinkUrl = objFromServer.getString(ABXConstants.PUSH_REMOTE_KEY_DEEPLINKURL);
            }
            if (PushUtils.checkHasVaildKey(objFromServer, ABXConstants.PUSH_REMOTE_KEY_DEEPLINKJSON)) {
                try {
                    deeplinkJson = objFromServer.getJSONObject(ABXConstants.PUSH_REMOTE_KEY_DEEPLINKJSON);
                } catch (JSONException e){
                    AbxLog.w("deepLinkJson parsing error", false);
                }
            }
            if(PushUtils.checkHasVaildKey(objFromServer, ABXConstants.PUSH_REMOTE_KEY_NOTIFICATION_ID)) {
                notificationId = objFromServer.getInt(ABXConstants.PUSH_REMOTE_KEY_NOTIFICATION_ID);
            }
            if (PushUtils.checkHasVaildKey(objFromServer, ABXConstants.PUSH_REMOTE_KEY_PUSH_ALWAYS_SHOWN)) {
                genWhileRun = objFromServer.getBoolean(ABXConstants.PUSH_REMOTE_KEY_PUSH_ALWAYS_SHOWN);
            }
            if(PushUtils.checkHasVaildKey(objFromServer, ABXConstants.PUSH_REMOTE_KEY_PUSH_GEN_VIBE)) {
                genVibe = objFromServer.getBoolean(ABXConstants.PUSH_REMOTE_KEY_PUSH_GEN_VIBE);
            }
            if(PushUtils.checkHasVaildKey(objFromServer, ABXConstants.PUSH_REMOTE_KEY_PUSH_GEN_SOUND)) {
                genSound = objFromServer.getBoolean(ABXConstants.PUSH_REMOTE_KEY_PUSH_GEN_SOUND);
            }
            if(PushUtils.checkHasVaildKey(objFromServer, ABXConstants.PUSH_REMOTE_KEY_ALERT)){
                JSONObject alertJSON = objFromServer.optJSONObject(ABXConstants.PUSH_REMOTE_KEY_ALERT);
                initPushData(alertJSON);
            }
            if(PushUtils.checkHasVaildKey(objFromServer, ABXConstants.PUSH_REMOTE_KEY_CUSTOM_ALERT)){
                JSONObject customAlertJSON = objFromServer.optJSONObject(ABXConstants.PUSH_REMOTE_KEY_CUSTOM_ALERT);
                String language = CommonUtils.getLanguage(context);
                AbxLog.d("Notification setting default language : " + language, false);
                JSONObject languageJson = PushUtils.getLanguageJson(language, customAlertJSON, objFromServer);
                initPushData(languageJson);
            }
            if(CommonUtils.isNullOrEmpty(title)){
                title = PushUtils.getApplicationName(context);
            }
        } catch (JSONException e) {
            AbxLog.e("Adbrix push data parsing error",e, true);
            return;
        }
    }

    private void initPushData(JSONObject value){
        if(value.length() == 0){
            AbxLog.d("JSONObject is empty", true);
            return;
        }
        try {
            if (PushUtils.checkHasVaildKey(value, ABXConstants.PUSH_REMOTE_KEY_TITLE)) {
                title = value.getString(ABXConstants.PUSH_REMOTE_KEY_TITLE);
            }
            if (PushUtils.checkHasVaildKey(value, ABXConstants.PUSH_REMOTE_KEY_BODY)) {
                body = value.getString(ABXConstants.PUSH_REMOTE_KEY_BODY);
            }
            if (PushUtils.checkHasVaildKey(value, ABXConstants.PUSH_REMOTE_KEY_BIG_TITLE)){
                bigTextTitle = value.getString(ABXConstants.PUSH_REMOTE_KEY_BIG_TITLE);
            }
            if (PushUtils.checkHasVaildKey(value, ABXConstants.PUSH_REMOTE_KEY_BIG_BODY)){
                bigTextBody = value.getString(ABXConstants.PUSH_REMOTE_KEY_BIG_BODY);
            }
            if (PushUtils.checkHasVaildKey(value, ABXConstants.PUSH_REMOTE_KEY_PUSH_IMAGE)){
                imageUrl = value.getString(ABXConstants.PUSH_REMOTE_KEY_PUSH_IMAGE);
            }
            if (PushUtils.checkHasVaildKey(value, ABXConstants.PUSH_REMOTE_KEY_SUMMARY_TEXT)){
                summaryText = value.getString(ABXConstants.PUSH_REMOTE_KEY_SUMMARY_TEXT);
            }
            if (PushUtils.checkHasVaildKey(value, ABXConstants.PUSH_REMOTE_KEY_PUSH_LARGE_ICON)) {
                largeIconUrl = value.getString(ABXConstants.PUSH_REMOTE_KEY_PUSH_LARGE_ICON);
            }
        } catch (Exception e){
            AbxLog.e(e, true);
        }
    }

    public JSONObject toOpenPushEventParamJson() {
        JSONObject pushParam = new JSONObject();

        try {
            pushParam
                .put(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_ID, this.campaignId)
                .put(ABXConstants.GROWTH_EVENT_KEY_STEP_ID, this.stepId)
                .put(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_REVISION_NO, this.campaignRevisionNo)
                .put(ABXConstants.GROWTH_EVENT_KEY_CYCLE_TIME, this.cycleTime);

        } catch (JSONException e) {
            AbxLog.e(e, true);
        }

        return pushParam;
    }

    public boolean isAvailable(){
        boolean result = false;
        if(!isOpenPushEventParamAvailable()){
            return result;
        }
        result = true;
        return result;
    }

    public boolean isOpenPushEventParamAvailable(){
        boolean result = false;
        if(CommonUtils.isNullOrEmpty(this.campaignId)){
            return result;
        }
        if(CommonUtils.isNullOrEmpty(this.stepId)){
            return result;
        }
        if(CommonUtils.isNullOrEmpty(this.cycleTime)){
            return result;
        }
        result = true;
        return result;
    }

    @Override
    public String toString() {
        return "AbxRemotePushModel{\n" +
                " title='" + title + '\'' +
                ",\n body='" + body + '\'' +
                ",\n bigTextTitle='" + bigTextTitle + '\'' +
                ",\n bigTextBody='" + bigTextBody + '\'' +
                ",\n summaryText='" + summaryText + '\'' +
                ",\n imageUrl='" + imageUrl + '\'' +
                ",\n largeIconUrl='" + largeIconUrl + '\'' +
                ",\n genWhileRun=" + genWhileRun +
                ",\n genVibe=" + genVibe +
                ",\n genSound=" + genSound +
                ",\n deeplinkUrl='" + deeplinkUrl + '\'' +
                ",\n deeplinkJson=" + deeplinkJson +
                ",\n campaignId='" + campaignId + '\'' +
                ",\n campaignRevisionNo=" + campaignRevisionNo +
                ",\n stepId='" + stepId + '\'' +
                ",\n cycleTime='" + cycleTime + '\'' +
                ",\n notificationId=" + notificationId +
                "\n}";
    }
}
