package io.adbrix.sdk.domain.model;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.domain.CompatConstants;
import io.adbrix.sdk.domain.IAMConstants;
import io.adbrix.sdk.utils.CommonUtils;

public class Triggers {
    public static final String PROPERTY_ABX_ITEM_PREFIX = "abx:item_";

    String CampaignId;
    String Type;
    String EventName;
    int Priority;
    JSONArray evt_properties;
    Long LastUpdatedTime;
    public Triggers(String campaignId, String type, String eventName, int priority, String evt_properties, Long LastUpdatedTime) {
        this.CampaignId = campaignId;
        this.Type = type;
        this.EventName = eventName;
        this.Priority = priority;
        try {
            this.evt_properties = new JSONArray(evt_properties);
        } catch (Exception e) {
            AbxLog.w("evt_properties is null", true);
            this.evt_properties = null;
        }
        this.LastUpdatedTime = LastUpdatedTime;
    }

    public String getCampaignId() {
        return CampaignId;
    }

    public String getType() {
        return Type;
    }

    public String getEventName() {
        return EventName;
    }

    public int getPriority() {
        return Priority;
    }

    public JSONArray getEvt_properties() {
        return evt_properties;
    }

    public boolean isValidByEventProperties(JSONObject eventParam){
        boolean result = false;
        if(CommonUtils.isNullOrEmpty(this.evt_properties)){
            //eventProperty를 적용하지 않는 트리거의 경우 이미 eventName이 트리거에 해당하기 때문에 true.
            result = true;
            return result;
        }
        if(CommonUtils.isNullOrEmpty(eventParam)){
            return result;
        }
        for(int i=0; i<this.evt_properties.length(); i++){
            JSONObject jsonObject = this.evt_properties.optJSONObject(i);
            if(CommonUtils.isNullOrEmpty(jsonObject)){
                continue;
            }
            String conditionProperty = getProperty(jsonObject);
            if(!hasConditionPropertyInEventParam(eventParam, conditionProperty)){
                continue;
            }
            Condition condition = getCondition(jsonObject);
            if(CommonUtils.isNull(condition)){
                continue;
            }
            ValueType valueType = getValueType(jsonObject);
            if(CommonUtils.isNull(valueType)){
                continue;
            }
            List<String> values = getValues(jsonObject);
            if(CommonUtils.isNullOrEmpty(values)){
                continue;
            }
            List<String> eventParamProperties = getEventParamValue(conditionProperty, eventParam);
            if(CommonUtils.isNullOrEmpty(eventParamProperties)){
                continue;
            }
            boolean itemResult = false;
            for(int j=0; j<eventParamProperties.size(); j++){
                String property = eventParamProperties.get(j);
                if(isTextCondition(condition)){
                    if(isAndTextCondition(condition)){
                        //Properties 모두가 조건에 부합하면 true
                        if(j==0){
                            itemResult = isValidTextCondition(condition, values, property);
                            continue;
                        }
                        itemResult &= isValidTextCondition(condition, values, property);;
                    } else{
                        //Properties 중 하나라도 조건에 부합하면 true
                        itemResult = isValidTextCondition(condition, values, property);
                        if(itemResult){
                            break;
                        }
                    }
                } else{
                    if(isNeedTwoPropertyCondition(condition)){
                        if(!hasTwoProperties(values)){
                            continue;
                        }
                    }
                    if(isAndNumericCondition(condition)){
                        //Properties 모두가 조건에 부합하면 true
                        if(j==0){
                            itemResult = isValidNumericCondition(condition, values, property, valueType);
                            continue;
                        }
                        itemResult &= isValidNumericCondition(condition, values, property, valueType);
                    } else{
                        //Properties 중 하나라도 조건에 부합하면 true
                        itemResult = isValidNumericCondition(condition, values, property, valueType);
                        if(itemResult){
                            break;
                        }
                    }
                }
            }
            //다중 evt_properties 조건 AND 처리
            if(i == 0){
                result = itemResult;
                continue;
            }
            result &= itemResult;
        }
        return result;
    }

    private String getPrefixRemovedItemsProperty(String property){
        if(CommonUtils.isNullOrEmpty(property)){
            return property;
        }
        if(property.length() < 9){
            return property;
        }
        property = property.substring(9, property.length());
        return property;
    }

    private boolean isAndTextCondition(Condition condition){
        boolean result = false;
        switch (condition){
            case notContains:
            case notEqual:{
                result = true;
                return result;
            }
            default:{
                return result;
            }
        }
    }
    private boolean isAndNumericCondition(Condition condition){
        boolean result = false;
        switch (condition){
            case notBetween:
                result = true;
                return result;
        }
        return result;
    }
    private boolean hasTwoProperties(List<String> values){
        if(values.size() == 2){
            return true;
        } else{
            return false;
        }
    }

    private boolean isNeedTwoPropertyCondition(Condition condition){
        boolean result = false;
        switch (condition){
            case between:
            case notBetween:
                result = true;
                return result;
        }
        return result;
    }

    private boolean hasConditionPropertyInEventParam(JSONObject eventParam, String property){
        boolean result = false;
        if(isItemsProperty(property)){
            if(!eventParam.has(CompatConstants.ABX_ITEMS)){
                return result;
            }
            property = getPrefixRemovedItemsProperty(property);
            JSONArray items = new JSONArray();
            try {
                items = eventParam.optJSONArray(CompatConstants.ABX_ITEMS);
            }catch (Exception e) {
                AbxLog.w(e, true);
            }
            if(CommonUtils.isNullOrEmpty(items)){
                return result;
            }
            for(int i=0; i<items.length(); i++){
                JSONObject item = items.optJSONObject(i);
                if(CommonUtils.isNullOrEmpty(item)){
                    continue;
                }
                if(item.has(property)){
                    result = true;
                    return result;
                }
            }
            return result;
        }
        if(eventParam.has(property)){
            result = true;
            return result;
        }
        return result;
    }

    private boolean isTextCondition(Condition condition){
        boolean result = false;
        switch (condition){
            case contains:
            case notContains:
            case notEqual:
            case equal:
            case startWith:
                result = true;
        }
        return result;
    }

    private boolean isValidTextCondition(Condition condition, List<String> values, String eventParamProperty){
        boolean result = false;
        switch (condition){
            case contains:{
                for(String value : values){
                    if(eventParamProperty.contains(value)){
                        result = true;
                        break;
                    }
                }
                break;
            }
            case notContains:{
                for(String value : values){
                    if(eventParamProperty.contains(value)){
                        result = true;
                    }
                }
                result = !result;
                break;
            }
            case equal:{
                for(String value : values){
                    if(eventParamProperty.equals(value)){
                        result = true;
                        break;
                    }
                }
                break;
            }
            case notEqual:{
                for(String value : values){
                    if(eventParamProperty.equals(value)){
                        result = true;
                    }
                }
                result = !result;
                break;
            }
            case startWith:{
                for(String value : values){
                    if(eventParamProperty.startsWith(value)){
                        result = true;
                        break;
                    }
                }
                break;
            }
        }
        return result;
    }

    private boolean isValidNumericCondition(Condition condition, List<String> values, String eventParamProperty, ValueType valueType){
        boolean result = false;
        try {
            switch (valueType){
                case NUMBER:{
                    Integer originalValue = Integer.parseInt(eventParamProperty);
                    Integer comparisonValue1 = Integer.parseInt(values.get(0));
                    switch (condition){
                        case greater:{
                            if(originalValue > comparisonValue1){
                                result = true;
                            }
                            break;
                        }
                        case greaterThanEqual:{
                            if(originalValue >= comparisonValue1){
                                result = true;
                            }
                            break;
                        }
                        case less:{
                            if(originalValue < comparisonValue1){
                                result = true;
                            }
                            break;
                        }
                        case lessThanEqual:{
                            if(originalValue <= comparisonValue1){
                                result = true;
                            }
                            break;
                        }
                        case between:{
                            Integer comparisonValue2 = Integer.parseInt(values.get(1));
                            if(originalValue > comparisonValue1 && originalValue < comparisonValue2){
                                result = true;
                            }
                            break;
                        }
                        case notBetween:{
                            Integer comparisonValue2 = Integer.parseInt(values.get(1));
                            if(originalValue <= comparisonValue1 || originalValue >= comparisonValue2){
                                result = true;
                            }
                            break;
                        }
                    }
                    break;
                }
                case LONG:{
                    Long originalValue = Long.parseLong(eventParamProperty);
                    Long comparisonValue1 = Long.parseLong(values.get(0));
                    switch (condition){
                        case greater:{
                            if(originalValue > comparisonValue1){
                                result = true;
                            }
                            break;
                        }
                        case greaterThanEqual:{
                            if(originalValue >= comparisonValue1){
                                result = true;
                            }
                            break;
                        }
                        case less:{
                            if(originalValue < comparisonValue1){
                                result = true;
                            }
                            break;
                        }
                        case lessThanEqual:{
                            if(originalValue <= comparisonValue1){
                                result = true;
                            }
                            break;
                        }
                        case between:{
                            Long comparisonValue2 = Long.parseLong(values.get(1));
                            if(originalValue > comparisonValue1 && originalValue < comparisonValue2){
                                result = true;
                            }
                            break;
                        }
                        case notBetween:{
                            Long comparisonValue2 = Long.parseLong(values.get(1));
                            if(originalValue <= comparisonValue1 || originalValue >= comparisonValue2){
                                result = true;
                            }
                            break;
                        }
                    }
                    break;
                }
                case DOUBLE:{
                    Double originalValue = Double.parseDouble(eventParamProperty);
                    Double comparisonValue1 = Double.parseDouble(values.get(0));
                    switch (condition){
                        case greater:{
                            if(originalValue > comparisonValue1){
                                result = true;
                            }
                            break;
                        }
                        case greaterThanEqual:{
                            if(originalValue >= comparisonValue1){
                                result = true;
                            }
                            break;
                        }
                        case less:{
                            if(originalValue < comparisonValue1){
                                result = true;
                            }
                            break;
                        }
                        case lessThanEqual:{
                            if(originalValue <= comparisonValue1){
                                result = true;
                            }
                            break;
                        }
                        case between:{
                            Double comparisonValue2 = Double.parseDouble(values.get(1));
                            if(originalValue > comparisonValue1 && originalValue < comparisonValue2){
                                result = true;
                            }
                            break;
                        }
                        case notBetween:{
                            Double comparisonValue2 = Double.parseDouble(values.get(1));
                            if(originalValue <= comparisonValue1 || originalValue >= comparisonValue2){
                                result = true;
                            }
                            break;
                        }
                    }
                    break;
                }
            }
        }catch (NumberFormatException e){
            AbxLog.w(e, true);
            return result;
        }
        return result;
    }

    private boolean isItemsProperty(String property){
        boolean result = false;
        if(property.startsWith(PROPERTY_ABX_ITEM_PREFIX)){
            result = true;
            return result;
        }
        return result;
    }

    private Condition getCondition(JSONObject jsonObject){
        Condition value = null;
        String conditionString = jsonObject.optString(IAMConstants.RESPONSE_IAM_TRIGGERS_EVT_PROPERTIES_CONDITION);
        if(CommonUtils.isNullOrEmpty(conditionString)){
            return value;
        }
        try {
            value = Condition.valueOf(conditionString);
        }catch (Exception e){
            return value;
        }
        return value;
    }

    private ValueType getValueType(JSONObject jsonObject){
        ValueType value = null;
        String valueTypeString = jsonObject.optString(IAMConstants.RESPONSE_IAM_TRIGGERS_EVT_PROPERTIES_VALUE_TYPE);
        if(CommonUtils.isNullOrEmpty(valueTypeString)){
            return value;
        }
        valueTypeString = valueTypeString.toUpperCase();
        try {
            value = ValueType.valueOf(valueTypeString);
        }catch (Exception e){
            return value;
        }
        return value;
    }

    private String getProperty(JSONObject jsonObject){
        String value = jsonObject.optString(IAMConstants.RESPONSE_IAM_TRIGGERS_EVT_PROPERTIES_PROPERTY);
        return value;
    }

    private List<String> getValues(JSONObject jsonObject){
        List<String> list = new ArrayList<>();
        JSONArray valuesString = new JSONArray();
        try {
            valuesString = jsonObject.optJSONArray(IAMConstants.RESPONSE_IAM_TRIGGERS_EVT_PROPERTIES_VALUES);
        }catch (Exception e){
            return list;
        }
        if(CommonUtils.isNullOrEmpty(valuesString)){
            return list;
        }
        for(int i=0; i<valuesString.length(); i++){
            String value = valuesString.optString(i);
            if(CommonUtils.isNullOrEmpty(value)){
                continue;
            }
            list.add(valuesString.optString(i));
        }
        return list;
    }

    private List<String> getEventParamValue(String property, JSONObject eventParam){
        List<String> values = new ArrayList<>();
        if(isItemsProperty(property)){
            if(!eventParam.has(CompatConstants.ABX_ITEMS)){
                return values;
            }
            JSONArray items = new JSONArray();
            try {
                items = eventParam.optJSONArray(CompatConstants.ABX_ITEMS);
                if(CommonUtils.isNullOrEmpty(items)){
                    return values;
                }
            }catch (Exception e){
                AbxLog.w(e, true);
            }
            property = getPrefixRemovedItemsProperty(property);
            for(int i=0; i<items.length(); i++){
                JSONObject item = items.optJSONObject(i);
                if(CommonUtils.isNullOrEmpty(item)){
                    continue;
                }
                if(item.has(property)){
                    String value = item.optString(property);
                    if(CommonUtils.isNullOrEmpty(value)){
                        continue;
                    }
                    values.add(getPrefixRemovedValue(value));
                }
            }
            return values;
        }
        if(eventParam.has(property)){
            String value = eventParam.optString(property);
            if(!CommonUtils.isNullOrEmpty(value)){
                values.add(getPrefixRemovedValue(value));
                return values;
            }
        }
        return values;
    }

    private String getPrefixRemovedValue(String value){
        if(!value.contains(":")){
            return value;
        }
        String[] splitString = value.split(":");
        return splitString[1];
    }

    public enum Condition {
        greater,
        greaterThanEqual,
        less,
        lessThanEqual,
        between,
        contains,
        notContains,
        notEqual,
        equal,
        startWith,
        notBetween,
    }

    public enum ValueType {
        TEXT,
        NUMBER,
        BOOLEAN,
        DATETIME,
        ENUM,
        LONG,
        DOUBLE
    }

    @Override
    public String toString() {
        return "Triggers{" +
                "CampaignId='" + CampaignId + '\'' +
                ", Type='" + Type + '\'' +
                ", EventName='" + EventName + '\'' +
                ", Priority=" + Priority +
                ", evt_properties=" + evt_properties +
                ", LastUpdatedTime=" + LastUpdatedTime +
                '}';
    }
}
