package io.adbrix.sdk.data.net;

import com.igaworks.v2.core.AdBrixRm;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.Timer;
import java.util.TimerTask;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.Error;
import io.adbrix.sdk.domain.model.Response;
import io.adbrix.sdk.domain.model.Success;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.HttpConnectionUtils;

public class ApiConnectionManager<Param> {
    private static final int MAX_RETRY = 3;
    private final Result<Param> result;
    private Param param;
    private int attempt = 0;
    private Timer retryTimer;
    private TimerTask retryTimerTask;

    public ApiConnectionManager(Result<Param> result, Param param) {
        this.param = param;
        if (result == null)
            this.result = new NullResult<>();
        else this.result = result;
    }

    public ApiConnectionManager() {
        this.result = new NullResult<>();
    }

    public void executeWithRetryByResultCode(IApiConnection apiConnection){
        executeWithRetry(apiConnection, true);
    }
    public void executeWithRetry(IApiConnection apiConnection){
        executeWithRetry(apiConnection, false);
    }

    public void executeWithRetry(IApiConnection apiConnection, boolean retryByResultCode) {
        logRequest(apiConnection);
        String response = null;
        try {
            response = apiConnection.request();
        } catch (Exception e) {
            AbxLog.e(e, true);
            return;
        }
        logResponse(apiConnection, response);

        if (apiConnection.isHttpOK()) {
            if(retryByResultCode){
                if(shouldRetry(apiConnection, response)){
                    retry(apiConnection, true);
                } else{
                    result.connectSuccess(response, apiConnection.getResponseCode(), param);
                }
                return;
            }
            result.connectSuccess(response, apiConnection.getResponseCode(), param);
        } else if (apiConnection.isWrongAppkey()) {
            // HTTP 500: wrong appKey
            //DFNSessionState.getInstance().adbrixError.getAndSet(true);
            AbxLog.w("Warning!! :: wrong appkey : " + response, true);

            retry(apiConnection, retryByResultCode);
        } else if (apiConnection.isInvalidAppkey()) {
            //AppKey ERR
            //DFNSessionState.getInstance().adbrixError.getAndSet(true);
            AbxLog.w("Warning!! :: Appkey is not valid. " + response, true);
            // not retry
            AbxLog.d("Broken connection to AdBrixRm. Skip retry uploading events. " + response, true);

            result.connectFail(apiConnection.getResponseCode(), param);
        } else {
            // not retry
            AbxLog.d("Broken connection to AdBrixRm. Skip retry uploading events. " + response, true);
            result.connectFail(apiConnection.getResponseCode(), param);
        }
    }

    public void execute(IApiConnection apiConnection) {
        logRequest(apiConnection);
        String response = null;
        try {
            response = apiConnection.request();
        } catch (Exception e) {
            AbxLog.e(e, true);
            return;
        }
        logResponse(apiConnection, response);

        if (apiConnection.isHttpOK()) {
            result.connectSuccess(response, apiConnection.getResponseCode(), param);
        } else if (apiConnection.isWrongAppkey()) {
            // HTTP 500: wrong appKey
            //DFNSessionState.getInstance().adbrixError.getAndSet(true);
            AbxLog.w("Warning!! :: wrong appkey : " + response, true);
            result.connectFail(apiConnection.getResponseCode(), param);
        } else if (apiConnection.isInvalidAppkey()) {
            //AppKey ERR
            //DFNSessionState.getInstance().adbrixError.getAndSet(true);
            AbxLog.w("Warning!! :: Appkey is not valid. " + response, true);
            // not retry
            AbxLog.w("Broken connection to AdBrixRm. Skip retry uploading events. " + response, true);
            result.connectFail(apiConnection.getResponseCode(), param);
        } else {
            // not retry
            AbxLog.d("Broken connection to AdBrixRm. Skip retry uploading events. " + response, true);
            result.connectFail(apiConnection.getResponseCode(), param);
        }
    }

    private void retry(IApiConnection apiConnection, boolean retryByResultCode) {
        if (attempt >= MAX_RETRY) {
            AbxLog.d("[RETRY] failed. Skip retry uploading events. "+apiConnection.getUrl(), true);
            result.connectFail(apiConnection.getResponseCode(), param);
            attempt = 0;
            return;
        }
        retryTimerTask = new TimerTask() {
            @Override
            public void run() {
                AbxLog.d("[RETRY] : (" + (attempt + 1) + "/3) "+apiConnection.getUrl(), true);
                attempt++;
                if(retryByResultCode){
                    executeWithRetryByResultCode(apiConnection);
                }else{
                    executeWithRetry(apiConnection);
                }
            }
        };
        retryTimer = new Timer();
        retryTimer.schedule(retryTimerTask, CoreConstants.NETWORK_DELAY_FOR_RETRY);
    }

    /**
     * 2022.11.21 bobos
     * deferredDeeplink, GetAttributionData Response
     * resultCode < 0 : 재처리해야함
     * @see "https://www.notion.so/dfinery/API-4540aceef04049d092eb53cea923c147"
     * @see "https://www.notion.so/dfinery/Attribution-API-Feat-Adobe-Analysis-2e91dbe2ff2e40f3bc3898c92c5cc344"
     * @param apiConnection
     * @param response
     * @return
     */
    private boolean shouldRetry(IApiConnection apiConnection, String response){
        boolean result = false;
        if(CommonUtils.isNullOrEmpty(response)){
            result = true;
            return result;
        }

        JSONObject jsonObject;
        try {
            jsonObject = new JSONObject(response);
            if(!jsonObject.has("result")){
                return result;
            }
            int apiResult = jsonObject.getInt("result");
            if(apiResult >= 0){
                return result;
            }
            result = true;
            return result;
        } catch (JSONException e) {
            AbxLog.w("parsing failed. response: "+response, e, true);
        }

        return result;
    }

    public interface Result<Param> {
        void connectSuccess(String responseString, int responseCode, Param param);

        void connectFail(int responseCode, Param param);
    }

    private static class NullResult<Param> implements Result<Param> {
        @Override
        public void connectSuccess(String responseString, int responseCode, Param param) {
            //do nothing
        }

        @Override
        public void connectFail(int responseCode, Param param) {
            //do nothing
        }
    }
    public void setTimerToNull() {
        CommonUtils.cancelTimerTask(retryTimerTask);
        retryTimerTask = null;
        CommonUtils.cancelTimer(retryTimer);
        retryTimer = null;
    }

    private void logRequest(IApiConnection apiConnection){
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append("[");
        stringBuilder.append(apiConnection.getRequestMethod());
        stringBuilder.append("->] ");
        stringBuilder.append(apiConnection.getUrl());
        JSONObject body = apiConnection.getBody();
        if(body!=null){
            stringBuilder.append("\n");
            try {
                stringBuilder.append(apiConnection.getBody().toString(1));
            } catch (JSONException e) {
                AbxLog.w(e, true);
            }
        }
        AbxLog.i(stringBuilder.toString(), true);
    }
    private void logResponse(IApiConnection apiConnection, String response){
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append("[");
        stringBuilder.append(apiConnection.getRequestMethod());
        stringBuilder.append("<-");
        stringBuilder.append(apiConnection.getResponseCode());
        stringBuilder.append("] ");
        stringBuilder.append(apiConnection.getUrl());
        stringBuilder.append("\n");
        JSONObject jsonObject = null;
        String jsonString = null;
        try {
            jsonObject = new JSONObject(response);
            jsonString = jsonObject.toString(1);
        } catch (Exception e) {
            jsonObject = null;
        }
        if(jsonString == null){
            stringBuilder.append(response);
        } else{
            stringBuilder.append(jsonString);
        }
        AbxLog.i(stringBuilder.toString(), true);
    }
}
