package io.adbrix.sdk.data.modelprovider;

import android.content.Context;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.Arrays;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.IABXComponentsFactory;
import io.adbrix.sdk.component.IUserPropertyManager;
import io.adbrix.sdk.data.dataprovider.DeviceRealtimeDataProvider;
import io.adbrix.sdk.data.dataprovider.DeviceStaticDataProvider;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.CommonModel;
import io.adbrix.sdk.domain.model.DRModel;
import io.adbrix.sdk.domain.model.EventModel;
import io.adbrix.sdk.domain.model.IdentityModel;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.utils.CommonUtils;

public class DRModelProvider implements IApiModelProvider<DRModel> {
    private Context context;
    private DeviceRealtimeDataProvider deviceRealtimeDataProvider;
    private DeviceStaticDataProvider deviceStaticDataProvider;
    private DataRegistry dataRegistry;
    private String userId;
    private DRModel.OperationType operationType;

    public DRModelProvider(
            IABXComponentsFactory factory,
            String userId,
            DRModel.OperationType operationType
    ) {
        try {
            this.context = factory.getAndroidContext();
            this.deviceRealtimeDataProvider = factory.createOrGetDeviceRealtimeDataProvider();
            this.deviceStaticDataProvider = factory.createOrGetDeviceStaticDataProvider();
            this.dataRegistry = factory.createOrGetDataRegistry();
        } catch (Exception e) {
            AbxLog.e("DeleteRestartModelProvider:: ComponentsCanNotCreateException!", true);
        }

        this.userId = userId;
        this.operationType = operationType;
    }

    @Override
    public DRModel provide() {
        deviceStaticDataProvider.provideDefaultValues();
        deviceRealtimeDataProvider.refreshData();

        IdentityModel identityModel = new IdentityModel(
                this.dataRegistry.safeGetString(DataRegistryKey.STRING_ADID, null),
                this.dataRegistry.safeGetString(DataRegistryKey.STRING_GAID, null),
                this.dataRegistry.safeGetString(DataRegistryKey.STRING_UUID, null),
                this.dataRegistry.safeGetString(DataRegistryKey.STRING_IDFA, null),
                this.dataRegistry.safeGetString(DataRegistryKey.STRING_IDFV, null),
                this.dataRegistry.safeGetString(DataRegistryKey.STRING_IGAW_ID, null),
                this.dataRegistry.safeGetString(DataRegistryKey.STRING_DFN_ID, null),
                this.dataRegistry.safeGetBoolean(DataRegistryKey.BOOLEAN_AD_ID_OPT_OUT, false),
                this.dataRegistry.safeGetString(DataRegistryKey.STRING_DEVICE_ID, null),
                this.dataRegistry.safeGetString(DataRegistryKey.STRING_REGISTRATION_ID, null),
                this.dataRegistry.safeGetBoolean(DataRegistryKey.BOOLEAN_IS_PUSH_ENABLE, false),
                false
        );

        String appKey = dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY, null);
        JSONObject req = new JSONObject();

        if (operationType == DRModel.OperationType.INITIALIZE) {
            try {
                long currentTimeMillis = System.currentTimeMillis();
                String eventDatetime = CommonUtils.getCurrentUTCInDBFormat(currentTimeMillis);
                String lastFirstOpenId = CommonUtils.randomUUIDWithCurrentTime(currentTimeMillis);

                //DataRegistry에 Init Restart Event Datetime 저장. Init Restart Not Synced 상황에서 사용 예정.
                saveEventDateTime(eventDatetime);

                //DataRegistry에 last_first_open_event_id 설정
                saveLastOpenId(lastFirstOpenId);

                LogEventParameter logEventParameter = new LogEventParameter(
                        lastFirstOpenId,
                        null,
                        CoreConstants.GROUP_ABX,
                        "",
                        null,
                        0,
                        0,
                        1L,
                        1L,
                        eventDatetime
                );

                CommonModelProvider commonModelProvider = new CommonModelProvider(dataRegistry, context, deviceRealtimeDataProvider);
                CommonModel commonModel = commonModelProvider.provide();

                EventModelProvider eventModelProvider = new EventModelProvider(new IUserPropertyManager.NullUserPropertyManager(), dataRegistry);

                eventModelProvider.setLogEventParameter(logEventParameter);
                EventModel eventModel = eventModelProvider.provide();

                req.put("common", commonModel.getJson());
                req.put("evt", eventModel.getJson());

                dataRegistry.putDataRegistry(
                        new DataUnit(DataRegistryKey.STRING_PREV_ID,
                                lastFirstOpenId,
                                5,
                                this.getClass().getName(),
                                true));

                return new DRModel(identityModel, userId, operationType, eventDatetime, req, appKey);
            } catch (JSONException e) {
                AbxLog.e(e, true);
            }
        }

        return new DRModel(identityModel, userId, operationType, req, appKey);
    }

    public DRModel getNotSyncedInitRestartModel() {
        deviceStaticDataProvider.provideDefaultValues();
        deviceRealtimeDataProvider.refreshData();

        IdentityModel identityModel = new IdentityModel(
                this.dataRegistry.safeGetString(DataRegistryKey.STRING_ADID, null),
                this.dataRegistry.safeGetString(DataRegistryKey.STRING_GAID, null),
                this.dataRegistry.safeGetString(DataRegistryKey.STRING_UUID, null),
                this.dataRegistry.safeGetString(DataRegistryKey.STRING_IDFA, null),
                this.dataRegistry.safeGetString(DataRegistryKey.STRING_IDFV, null),
                this.dataRegistry.safeGetString(DataRegistryKey.STRING_IGAW_ID, null),
                this.dataRegistry.safeGetString(DataRegistryKey.STRING_DFN_ID, null),
                this.dataRegistry.safeGetBoolean(DataRegistryKey.BOOLEAN_AD_ID_OPT_OUT, false),
                this.dataRegistry.safeGetString(DataRegistryKey.STRING_DEVICE_ID, null),
                this.dataRegistry.safeGetString(DataRegistryKey.STRING_REGISTRATION_ID, null),
                this.dataRegistry.safeGetBoolean(DataRegistryKey.BOOLEAN_IS_PUSH_ENABLE, false),
                false
        );

        long currentTimeMillis = System.currentTimeMillis();
        String eventDatetime = getNotSyncedInitRestartDatetime();
        if(CommonUtils.isNullOrEmpty(eventDatetime)){
            AbxLog.w("getNotSyncedInitRestartDatetime is null", true);
            eventDatetime = CommonUtils.getCurrentUTCInDBFormat(currentTimeMillis);
            saveEventDateTime(eventDatetime);
        }
        String lastFirstOpenId = getNotSyncedInitRestartLastFirstOpenId();
        if(CommonUtils.isNullOrEmpty(lastFirstOpenId)){
            AbxLog.w("getNotSyncedInitRestartLastFirstOpenId is null", true);
            lastFirstOpenId = CommonUtils.randomUUIDWithCurrentTime(currentTimeMillis);
            saveLastOpenId(lastFirstOpenId);
        }
        String appKey = dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY, null);

        //reset prevId. If not reset prev id, last first open id equals prev id
        dataRegistry.putDataRegistry(
                new DataUnit(DataRegistryKey.STRING_PREV_ID,
                        null,
                        5,
                        this.getClass().getName(),
                        true));

        JSONObject req = new JSONObject();

        try {
            LogEventParameter logEventParameter = new LogEventParameter(
                    lastFirstOpenId,
                    null,
                    CoreConstants.GROUP_ABX,
                    "",
                    null,
                    0,
                    0,
                    eventDatetime
            );

            CommonModelProvider commonModelProvider = new CommonModelProvider(dataRegistry, context, deviceRealtimeDataProvider);
            CommonModel commonModel = commonModelProvider.provide();

            EventModelProvider eventModelProvider = new EventModelProvider(new IUserPropertyManager.NullUserPropertyManager(), dataRegistry);
            eventModelProvider.setLogEventParameter(logEventParameter);
            EventModel eventModel = eventModelProvider.provide();

            req.put("common", commonModel.getJson());
            req.put("evt", eventModel.getJson());

            dataRegistry.putDataRegistry(
                    new DataUnit(DataRegistryKey.STRING_PREV_ID,
                            lastFirstOpenId,
                            5,
                            this.getClass().getName(),
                            true));

        } catch (JSONException e) {
            AbxLog.e(e, true);
        }
        return new DRModel(identityModel, userId, operationType, eventDatetime, req, appKey);
    }

    private String getNotSyncedInitRestartDatetime() {
        return dataRegistry.safeGetString(DataRegistryKey.STRING_INIT_RESTART_EVENT_DATETIME, null);
    }

    private String getNotSyncedInitRestartLastFirstOpenId() {
        return dataRegistry.safeGetString(DataRegistryKey.STRING_LAST_OPEN_ID, null);
    }

    private void saveEventDateTime(String eventDatetime){
        //DataRegistry에 Init Restart Event Datetime 저장. Init Restart Not Synced 상황에서 사용 예정.
        dataRegistry.putDataRegistry(
                new DataUnit(
                        DataRegistryKey.STRING_INIT_RESTART_EVENT_DATETIME,
                        eventDatetime,
                        5,
                        this.getClass().getName(),
                        true
                )
        );
    }

    private void saveLastOpenId(String lastFirstOpenId){
        //DataRegistry에 last_first_open_event_id 설정
        dataRegistry.putDataRegistry(
                new DataUnit(
                        DataRegistryKey.STRING_LAST_FIRSTOPEN_ID,
                        lastFirstOpenId,
                        5,
                        this.getClass().getName(),
                        true
                ));

        //DataRegistry에 last_open_id 설정
        dataRegistry.putDataRegistry(
                new DataUnit(
                        DataRegistryKey.STRING_LAST_OPEN_ID,
                        lastFirstOpenId,
                        5,
                        getClass().getName(),
                        true
                ));
    }
}
