package io.adbrix.sdk.configuration;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;

import com.igaworks.v2.core.AdBrixRm;
import com.igaworks.v2.core.result.GetAttributionDataResult;
import com.igaworks.v2.core.result.GetSubscriptionStatusResult;
import com.igaworks.v2.core.result.SetCiProfileResult;
import com.igaworks.v2.core.result.SetSubscriptionStatusResult;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import javax.annotation.Nullable;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.GooglePlayReferrerProperties;
import io.adbrix.sdk.component.IABXComponentsFactory;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.net.IApiConnection;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.model.DfnInAppMessage;
import io.adbrix.sdk.domain.Repository;
import io.adbrix.sdk.domain.interactor.DeleteUserDataUseCase;
import io.adbrix.sdk.domain.interactor.RestartSDKUseCase;
import io.adbrix.sdk.domain.model.ActionHistory;
import io.adbrix.sdk.domain.model.ActionHistoryIdType;
import io.adbrix.sdk.domain.function.Completion;
import io.adbrix.sdk.domain.model.Empty;
import io.adbrix.sdk.domain.model.Error;
import io.adbrix.sdk.domain.model.SelfServeInAppMessage;
import io.adbrix.sdk.domain.model.Response;
import io.adbrix.sdk.domain.model.Result;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.domain.model.ResultCallback;
import io.adbrix.sdk.domain.model.SubscriptionStatus;
import io.adbrix.sdk.domain.model.UserPropertyCommand;
import io.adbrix.sdk.utils.CommonUtils;

public class DisabledABXContext implements IABXContext {

    private static final String WARNING_FOR_DISABLED_CONTEXT = "Disabled된 ABX SDK의 Method가 호출되었습니다.";
    private final Repository repository;
    private final IABXComponentsFactory componentsFactory;

    public DisabledABXContext(String disabledReason, IABXComponentsFactory componentsFactory, Repository repository) {
        AbxLog.d("ABX SDK가 Disabled 되었습니다. 이유는 = " + disabledReason, true);
        this.componentsFactory = componentsFactory;
        this.repository = repository;
    }

    @Override
    public void initialize(Context context, String appkey, String secretkey, ResultCallback<String> callback) {
        AbxLog.w("initialize-"+WARNING_FOR_DISABLED_CONTEXT, true);
        callback.callback(ResultCallback.Status.FAILURE, WARNING_FOR_DISABLED_CONTEXT);
    }

    @Override
    public void getInstallReferrer() {
        GooglePlayReferrerProperties googlePlayReferrerProperties = new GooglePlayReferrerProperties(componentsFactory.getAndroidContext(), componentsFactory);
        if(googlePlayReferrerProperties.needRefresh()){
            googlePlayReferrerProperties.saveInstallReferrerCompleted(false);
            googlePlayReferrerProperties.startConnection(new GooglePlayReferrerProperties.Callback() {
                @Override
                public void getReferrerMap(HashMap<String, Object> referrerMap, GooglePlayReferrerProperties googlePlayReferrerProperties) {
                    AbxLog.d("getInstallReferrer success: "+referrerMap.toString(), true);
                    googlePlayReferrerProperties.endConnection();
                    componentsFactory.getAbxContextController().flushQueuedEvent();
                }
            });
        }
    }

    @Override
    public void saveUserProperty(UserPropertyCommand userPropertyCommand) {
        AbxLog.w("saveUserProperty-"+WARNING_FOR_DISABLED_CONTEXT, true);
    }

    @Override
    public Boolean saveUserPropertyWithoutEvent(UserPropertyCommand userPropertyCommand) {
        AbxLog.w("saveUserPropertyWithoutEvent-"+WARNING_FOR_DISABLED_CONTEXT, true);
        return null;
    }

    @Override
    public void clearUserProperty() {
        AbxLog.w("clearUserProperty-"+WARNING_FOR_DISABLED_CONTEXT, true);
    }

    @Override
    public void getUserId(Completion<Result<String>> completion) {
        if(CommonUtils.notNull(completion)){
            completion.handle(Error.of(WARNING_FOR_DISABLED_CONTEXT));
        }
        AbxLog.w("getUserId-"+WARNING_FOR_DISABLED_CONTEXT, true);
    }

    @Override
    public void login(String userId, Completion<Result<Response>> completion) {
        AbxLog.w("login-"+WARNING_FOR_DISABLED_CONTEXT, true);
    }

    @Override
    public void logout(Completion<Result<Response>> completion) {
        AbxLog.w("logout-"+WARNING_FOR_DISABLED_CONTEXT, true);
    }

    @Override
    public void logEvent(LogEventParameter logEventParameter) {
        AbxLog.w("logEvent-"+WARNING_FOR_DISABLED_CONTEXT, true);
    }

    @Override
    public void logSameEventWithPaging(String eventName, String group, List<JSONObject> eventParamList) {
        AbxLog.w("logSameEventWithPaging-"+WARNING_FOR_DISABLED_CONTEXT, true);
    }

    @Override
    public void flushAllEvents(Completion<Result<Empty>> completion) {
        if(CommonUtils.notNull(completion)){
            completion.handle(Error.of(WARNING_FOR_DISABLED_CONTEXT));
        }
        AbxLog.w("flushAllEvents-"+WARNING_FOR_DISABLED_CONTEXT, true);
    }

    @Override
    public void runOnIdleTime() {
        AbxLog.w("runOnIdleTime-"+WARNING_FOR_DISABLED_CONTEXT, false);
    }

    @Override
    public void onResume(Activity activity) {
        AbxLog.w("onResume-"+WARNING_FOR_DISABLED_CONTEXT, false);
    }

    @Override
    public void onPause() {
        AbxLog.w("onPause-"+WARNING_FOR_DISABLED_CONTEXT, false);
    }

    @Override
    public void deeplink(Activity deeplinkActivity) {
        AbxLog.w("deeplink-"+WARNING_FOR_DISABLED_CONTEXT, false);
    }

    @Override
    public void deeplinkWithIntent(Intent deeplinkIntent) {
        AbxLog.w("deeplinkWithIntent-"+WARNING_FOR_DISABLED_CONTEXT, false);
    }

    @Override
    public void putDataRegistry(DataUnit dataUnit) {
        AbxLog.w("putDataRegistry-"+WARNING_FOR_DISABLED_CONTEXT, false);
    }

    @Override
    public void deleteUserData(String userId, ResultCallback<String> callback) {
        AbxLog.d("Delete user data api called!", true);
        new DeleteUserDataUseCase(repository, userId, callback).execute();
    }

    @Override
    public void restartSDK(String userId, ResultCallback<String> callback) {
        AbxLog.d("Restart SDK api called!", true);
        new RestartSDKUseCase(repository, userId, callback).execute();
    }

    @Override
    public void fetchActionHistoryFromServer(@Nullable String token, ActionHistoryIdType idType, List<String> actionType, Completion<Result<List<ActionHistory>>> completion) {
        if(CommonUtils.notNull(completion)){
            completion.handle(Error.of(WARNING_FOR_DISABLED_CONTEXT));
        }
        AbxLog.w("fetchActionHistoryFromServer-"+WARNING_FOR_DISABLED_CONTEXT, false);
    }

    @Override
    public void insertPushData(String pushDataString) {
        AbxLog.w("insertPushData-"+WARNING_FOR_DISABLED_CONTEXT, false);
    }

    @Override
    public void getActionHistory(int skip, int limit, List<String> actionType, Completion<Result<List<ActionHistory>>> completion) {
        if(CommonUtils.notNull(completion)){
            completion.handle(Error.of(WARNING_FOR_DISABLED_CONTEXT));
        }
        AbxLog.w("getActionHistory-"+WARNING_FOR_DISABLED_CONTEXT, false);
    }

    @Override
    public void getAllActionHistory(List<String> actionType, Completion<Result<List<ActionHistory>>> completion) {
        if(CommonUtils.notNull(completion)){
            completion.handle(Error.of(WARNING_FOR_DISABLED_CONTEXT));
        }
        AbxLog.w("getAllActionHistory-"+WARNING_FOR_DISABLED_CONTEXT, false);
    }

    @Override
    public void deleteActionHistory(@Nullable String token, String historyId, long timestamp, Completion<Result<Empty>> completion) {
        if(CommonUtils.notNull(completion)){
            completion.handle(Error.of(WARNING_FOR_DISABLED_CONTEXT));
        }
        AbxLog.w("deleteActionHistory-"+WARNING_FOR_DISABLED_CONTEXT, false);
    }

    @Override
    public void deleteAllActionHistory(@Nullable String token, @Nullable ActionHistoryIdType idType, Completion<Result<Empty>> completion) {
        if(CommonUtils.notNull(completion)){
            completion.handle(Error.of(WARNING_FOR_DISABLED_CONTEXT));
        }
        AbxLog.w("deleteAllActionHistory-"+WARNING_FOR_DISABLED_CONTEXT, false);
    }

    @Override
    public void clearSyncedActionHistoryInLocalDB(Completion<Result<Empty>> completion) {
        if(CommonUtils.notNull(completion)){
            completion.handle(Error.of(WARNING_FOR_DISABLED_CONTEXT));
        }
        AbxLog.w("clearSyncedActionHistoryInLocalDB-"+WARNING_FOR_DISABLED_CONTEXT, false);
    }

    @Override
    public void clearAllActionHistoryInLocalDB() {
        AbxLog.w("clearAllActionHistoryInLocalDB-"+WARNING_FOR_DISABLED_CONTEXT, false);
    }

    @Override
    public void fetchInAppMessage(boolean isDirectCall, Completion<Result<Empty>> completion) {
        if(CommonUtils.notNull(completion)){
            completion.handle(Error.of(WARNING_FOR_DISABLED_CONTEXT));
        }
        AbxLog.w("fetchInAppMessage-"+WARNING_FOR_DISABLED_CONTEXT, false);
    }

    @Override
    public void getAllInAppMessage(Completion<Result<List<DfnInAppMessage>>> completion) {
        if(CommonUtils.notNull(completion)){
            completion.handle(Error.of(WARNING_FOR_DISABLED_CONTEXT));
        }
        AbxLog.w("getAllInAppMessage-"+WARNING_FOR_DISABLED_CONTEXT, false);
    }

    @Override
    public void getSelfServeInAppMessages(AdBrixRm.GetSelfServeInAppMessagesCallback callback){
        if(CommonUtils.notNull(callback)){
            List<SelfServeInAppMessage> list = new ArrayList<>();
            callback.onCallback(list);
        }
        AbxLog.w("getAllSelfServeInAppMessages"+WARNING_FOR_DISABLED_CONTEXT, false);
    }

    @Override
    public void openInAppMessage(String campaignId, Completion<Result<Empty>> completion) {
        if(CommonUtils.notNull(completion)){
            completion.handle(Error.of(WARNING_FOR_DISABLED_CONTEXT));
        }
        AbxLog.w("openInAppMessage-"+WARNING_FOR_DISABLED_CONTEXT, false);
    }

    @Override
    public void deleteAllInAppMessageDBContents() {
        AbxLog.w("deleteAllInAppMessageDBContents-"+WARNING_FOR_DISABLED_CONTEXT, false);
    }

    @Override
    public void requestGetAttributionData(String logId, AdBrixRm.GetAttributionDataCallback callback, DataRegistry dataRegistry) {
        if(CommonUtils.notNull(callback)){
            callback.onCallback(new GetAttributionDataResult(IApiConnection.RESPONSE_CODE_ERROR, GetAttributionDataResult.Result.UNKNOWN_ERROR.getValue()));
        }
        AbxLog.w("requestGetAttributionData-"+WARNING_FOR_DISABLED_CONTEXT, false);
    }

    @Override
    public void getSubscriptionStatus(AdBrixRm.GetSubscriptionStatusCallback callback, DataRegistry dataRegistry) {
        if(CommonUtils.notNull(callback)){
            callback.onCallback(new GetSubscriptionStatusResult(IApiConnection.RESPONSE_CODE_ERROR, GetSubscriptionStatusResult.RESULT_CODE_UNKNOWN_ERROR));
        }
        AbxLog.w("getSubscriptionStatus-"+WARNING_FOR_DISABLED_CONTEXT, false);
    }

    @Override
    public void setSubscriptionStatus(SubscriptionStatus subscriptionStatus, AdBrixRm.SetSubscriptionStatusCallback callback, DataRegistry dataRegistry) {
        if(CommonUtils.notNull(callback)){
            callback.onCallback(new SetSubscriptionStatusResult(IApiConnection.RESPONSE_CODE_ERROR, SetSubscriptionStatusResult.RESULT_CODE_UNKNOWN_ERROR));
        }
        AbxLog.w("setSubscriptionStatus-"+WARNING_FOR_DISABLED_CONTEXT, false);
    }

    @Override
    public void setCiProperty(String key, String value, AdBrixRm.SetCiProfileCallback callback, DataRegistry dataRegistry) {
        if(CommonUtils.notNull(callback)){
            callback.onCallback(new SetCiProfileResult(IApiConnection.RESPONSE_CODE_ERROR, SetCiProfileResult.RESULT_CODE_UNKNOWN_ERROR));
        }
        AbxLog.w("setCiProperty-"+WARNING_FOR_DISABLED_CONTEXT, false);
    }
}
