package io.adbrix.sdk.component;

import java.util.NoSuchElementException;
import java.util.Objects;

import io.adbrix.sdk.domain.function.Completion;
import io.adbrix.sdk.domain.function.Function;
import io.adbrix.sdk.domain.function.Supplier;
import io.adbrix.sdk.domain.function.ThrowingFunction;
import io.adbrix.sdk.domain.model.Result;

public class Optional<T> {

    private static final Optional<?> EMPTY = new Optional();
    private final T value;

    private Optional() {
        this.value = null;
    }

    protected Optional(T value) {
        this.value = value;
    }

    public static <T> Optional<T> of(T value) {
        return new Optional<T>(value);
    }

    @SuppressWarnings("unchecked")
    public static <T> Optional<T> empty() {
        return (Optional<T>) EMPTY;
    }

    public static <T> Optional<T> ofNullable(T value) {
        if (value == null) {
            return empty();
        } else {
            return of(value);
        }
    }

    public boolean isPresent() {
        return value != null;
    }

    public T get() {
        if (isPresent()) {
            return value;
        } else {
            throw new NoSuchElementException();
        }
    }

    public T orElse(T other) {
        if (isPresent()) {
            return value;
        } else {
            return other;
        }
    }

    public T orElseGet(Supplier<? extends T> other) {
        if (isPresent()) {
            return value;
        } else {
            return other.get();
        }
    }

    public <Ex extends Throwable> T orElseThrow(Supplier<? extends Ex> exceptionSupplier) throws Ex {
        if (isPresent()) {
            return value;
        } else {
            throw exceptionSupplier.get();
        }
    }

    public Optional<T> ifPresent(IObserver<? super T> observer) {
        if (value != null)
            observer.update(value);

        return this;
    }

    public Optional<T> ifNotPresent(Runnable runnable) {
        if(value == null)
            runnable.run();

        return this;
    }

    public <U> Optional<U> map(Function<? super T, ? extends U> mapper) {
        Objects.requireNonNull(mapper);
        if (!isPresent())
            return empty();
        else {
            return Optional.ofNullable(mapper.apply(value));
        }
    }

    public <U, E extends Exception> Optional<U> mapCatching(ThrowingFunction<? super T, ? extends U, E> mapper) {
        Objects.requireNonNull(mapper);
        if (!isPresent())
            return empty();

        try {
            return Optional.ofNullable(mapper.apply(value));
        } catch (Exception e) {
            return empty();
        }
    }
}
