package io.adbrix.sdk.component;

import org.json.JSONArray;
import org.json.JSONException;

import java.util.Arrays;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;

import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.net.IApiConnection;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.function.Completion;
import io.adbrix.sdk.domain.model.DRState;
import io.adbrix.sdk.domain.model.Empty;
import io.adbrix.sdk.domain.model.Error;
import io.adbrix.sdk.domain.model.EventModel;
import io.adbrix.sdk.domain.model.EventPackage;
import io.adbrix.sdk.domain.model.Response;
import io.adbrix.sdk.domain.model.Result;
import io.adbrix.sdk.domain.model.Success;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.CoreUtils;

public class EventPackageContainer {
    private final EventUploadIntervalManager eventUploadIntervalManager;
    private final IEventSender eventSender;
    private final DataRegistry dataRegistry;

    public EventPackageContainer(EventUploadIntervalManager eventUploadIntervalManager, IEventSender eventSender, DataRegistry dataRegistry) {
        this.eventUploadIntervalManager = eventUploadIntervalManager;
        this.dataRegistry = dataRegistry;
        this.eventSender = eventSender;
    }

    public void mergeV1EventsAndV2Events(EventPackage unsentV1EventPackage) {
        try {
            if (unsentV1EventPackage == null)
                return;

            Queue<EventPackage> mergedQueue = new LinkedList<>();
            EventPackage packageQueue = getPreviousPackageQueue();
            mergedQueue.offer(unsentV1EventPackage);
            mergedQueue.offer(packageQueue);

            EventPackage formattedEventPackage = formatEventPackageQueue(mergedQueue);
            updateUnsentEventPackagesInDataRegistry(formattedEventPackage);
            AbxLog.d("Merging v1 events and v2 events is finished. EventQueue Size : " + formattedEventPackage.toString(), true);
        } catch (Exception e) {
            AbxLog.e(e, true);
        }
    }

    public void updateUnsentEventPackagesInDataRegistry(EventPackage packageQueue) {
        JSONArray jsonArray = new JSONArray();
        try {
            jsonArray.put(packageQueue.toJson());
        } catch (JSONException e) {
            AbxLog.e(e, true);
        }

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_UNSENT_EVENTPACKAGES,
                jsonArray.toString(),
                5,
                this.getClass().getName(),
                true
        ));
    }

    private EventPackage getPreviousPackageQueue() {
        Queue<EventPackage> tempQueue = getPreviousUnformattedPackageQueue();
        return formatEventPackageQueue(tempQueue);
    }

    protected Queue<EventPackage> getPreviousUnformattedPackageQueue(){
        String unsentEventPackagesString = dataRegistry.safeGetString(DataRegistryKey.STRING_UNSENT_EVENTPACKAGES, null);
        if (CommonUtils.isNullOrEmpty(unsentEventPackagesString)) return new LinkedList<>();
        JSONArray jsonArray = null;
        try {
            jsonArray = new JSONArray(unsentEventPackagesString);
        } catch (JSONException e) {
            AbxLog.w(e, true);
        }

        if (jsonArray == null) return new LinkedList<>();

        Queue<EventPackage> tempQueue = new LinkedList<>();
        try {
            for (int i = 0; i < jsonArray.length(); i++) {
                EventPackage eventPackage = EventPackage.fromJson(jsonArray.getJSONObject(i));
                if (eventPackage == null) {
                    continue;
                }
                tempQueue.offer(eventPackage);
            }
        } catch (Exception e) {
            AbxLog.w(e, true);
        }
        return tempQueue;
    }

    protected Queue<EventModel> getEventModelsFromEventPackage(Queue<EventPackage> eventPackageQueue){
        Queue<EventModel> flattedEventModelQueue = new LinkedList<>();

        for (EventPackage eventPackage : eventPackageQueue) {
            flattedEventModelQueue.addAll(eventPackage.eventModels);
        }
        while (flattedEventModelQueue.size() > CoreConstants.MAX_EVENT_COUNT)
            flattedEventModelQueue.poll();
        return flattedEventModelQueue;
    }

    private EventPackage formatEventPackageQueue(Queue<EventPackage> unformattedQueue) {
        Queue<EventModel> flattedEventModelQueue = getEventModelsFromEventPackage(unformattedQueue);
        EventPackage formattedEventPackageQueue = partition(flattedEventModelQueue);
        return formattedEventPackageQueue;
    }

    private EventPackage partition(Queue<EventModel> eventModels) {
        //10개씩 나누지 않고 한 패키지안에 다 넣어버림
//        LinkedList<EventPackage> eventPackages = new LinkedList<>();
//        LinkedList<EventModel> chunked = new LinkedList<>();

//        for (EventModel eventModel : eventModels) {
//            chunked.add(eventModel);
//            if (chunked.size() == CoreConstants.EVENT_BUFFER_COUNT) {
//                eventPackages.add(new EventPackage(chunked));
//                chunked = new LinkedList<>();
//            }
//        }
//        if (!chunked.isEmpty()) {
//            eventPackages.add(new EventPackage(chunked));
//        }
//        eventPackages.add(new EventPackage(eventModels));
        return new EventPackage(eventModels);
    }

    protected void clearUnsentEventsInDataRegistry(){
        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_UNSENT_EVENTPACKAGES,
                "",
                5,
                this.getClass().getName(),
                true
        ));
    }

    public void flushAtIntervals() {

        Runnable eventSendingTask = new Runnable() {
            @Override
            public void run() {
                flushEventsNotSent(new Completion<Result<Response>>() {
                    @Override
                    public void handle(Result<Response> responseResult) {
                        AbxLog.d("flushAtIntervals: "+responseResult.toString(), true);
                    }
                });
            }
        };

        eventUploadIntervalManager.manageEventUploadInterval(eventSendingTask, getPreviousPackageQueue());
    }

    public void flushSingleEventPackageImmediately(EventPackage eventPackage, Completion<Result<Response>> completion) {
        eventSender.sendEvent(eventPackage, new IEventSender.IEventSendActionListener() {
            @Override
            public void onSuccess(Response response, EventPackage successfulEventPackage) {
                if(completion==null){
                    AbxLog.e("completion is null", true);
                    return;
                }
                completion.handle(Success.of(response));
            }

            @Override
            public void onFail(Response response, EventPackage failedEventPackage) {
                if(failedEventPackage == null){
                    completion.handle(Error.of("onFail() failedEventPackage is null. "+response.getCode()));
                    return;
                }
                if(response.getCode() != IApiConnection.RESPONSE_CODE_REQUEST_ENTITY_TOO_LARGE){
                    updateUnsentEventPackagesInDataRegistry(failedEventPackage);
                }
                if(completion==null){
                    AbxLog.e("completion is null", true);
                    return;
                }
                completion.handle(Error.of(response.toString()));
            }
        });
    }

    public void flushEventsNotSent(Completion<Result<Response>> completion){
        Queue<EventPackage> unsentEventPackage = getPreviousUnformattedPackageQueue();
        Queue<EventModel> unsentEventModels = getEventModelsFromEventPackage(unsentEventPackage);
        flushEventsNotSent(new EventPackage(unsentEventModels), completion);
    }

    public void flushEventsNotSent(EventPackage eventPackage, Completion<Result<Response>> completion) {
        if (CoreUtils.getDRState(dataRegistry) == DRState.INIT_RESTART_NOT_SYNCED) {
            String message = "Initializing restart not synced! Cannot send events.";
            AbxLog.d(message, true);
            completion.handle(Error.of(message));
            return;
        }
        if(CommonUtils.notNull(eventPackage)){
            if(CommonUtils.isNullOrEmpty(eventPackage.eventModels)){
                completion.handle(Success.of(new Response(200, "eventModels is empty")));
                return;
            }
        }
        clearUnsentEventsInDataRegistry();
        AbxLog.d("flushEventsNotSent() eventModelSize: "+eventPackage.eventModels.size(), true);
        flushSingleEventPackageImmediately(eventPackage, completion);
    }

    public IEventSender getEventSender() {
        return eventSender;
    }
    public void deactivate(){

    }
}
