package io.adbrix.sdk.component;

import android.content.Context;

import org.json.JSONException;
import org.json.JSONObject;

import io.adbrix.sdk.data.dataprovider.DeviceRealtimeDataProvider;
import io.adbrix.sdk.data.modelprovider.CommonModelProvider;
import io.adbrix.sdk.data.modelprovider.EventBundleModelProvider;
import io.adbrix.sdk.data.net.ApiConnectionManager;
import io.adbrix.sdk.data.net.IApiConnection;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.model.CommonModel;
import io.adbrix.sdk.domain.model.EventBundleModel;
import io.adbrix.sdk.domain.model.EventPackage;
import io.adbrix.sdk.domain.model.Response;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.CoreUtils;

public class DefaultEventSender implements IEventSender {
    private final DataRegistry dataRegistry;
    private final Context androidContext;
    private final DeviceRealtimeDataProvider deviceRealtimeDataProvider;
    private final IApiConnection apiConnection;

    public DefaultEventSender(DataRegistry dataRegistry, Context androidContext, DeviceRealtimeDataProvider deviceRealtimeDataProvider, IApiConnection apiConnection) {
        this.dataRegistry = dataRegistry;
        this.androidContext = androidContext;
        this.deviceRealtimeDataProvider = deviceRealtimeDataProvider;
        this.apiConnection = apiConnection;
    }

    @Override
    public void sendEvent(final EventPackage eventPackage, IEventSendActionListener sendActionListener) {
        if (eventPackage == null || eventPackage.eventModels == null) {
            AbxLog.d("Invalid eventPackage!", true);
            return;
        }

        if (!CoreUtils.isOnline(androidContext)) {
            AbxLog.e("Fail to sending event. You are not connected to the network.", true);
            sendActionListener.onFail(new Response(IEventSender.RESPONSE_CODE_DISCONNECTED, ""), eventPackage);
            return;
        }

        if (CoreUtils.isGdprForgetMe(dataRegistry, () -> { })) {
            AbxLog.d("GdprForgetMe is true", true);
            sendActionListener.onFail(new Response(IEventSender.RESPONSE_CODE_CONNECTION_ABORTED, ""), eventPackage);
            return;
        }

        CommonModel commonModel = getCommonModel();
        EventBundleModel eventBundleModel = new EventBundleModelProvider(commonModel, eventPackage).provide();

        IApiConnection apiConnection;
        ApiConnectionManager<EventPackage> manager = new ApiConnectionManager<>(new ApiConnectionManager.Result<EventPackage>() {
            @Override
            public void connectSuccess(String responseString, int responseCode, EventPackage eventPackage) {
                if(sendActionListener == null){
                    AbxLog.d("sendActionListener is null", true);
                    return;
                }
                sendActionListener.onSuccess(new Response(responseCode, responseString), eventPackage);
//                AbxLog.d("Sending event is complete. responseCode : " + responseCode + ", responseString : " + responseString, true);
            }

            @Override
            public void connectFail(int responseCode, EventPackage eventPackage) {
                if(sendActionListener == null){
                    AbxLog.d("sendActionListener is null", true);
                    return;
                }
                if(eventPackage == null){
                    AbxLog.d("eventPackage is null", true);
                    sendActionListener.onFail(new Response(responseCode, ""), null);
                    return;
                }
                sendActionListener.onFail(new Response(responseCode, ""), eventPackage);
                AbxLog.d("Sending event is failed. responseCode : " + responseCode, true);
            }
        }, eventPackage);
        apiConnection = this.apiConnection.post(eventBundleModel);
        manager.execute(apiConnection);
    }

    @Override
    public CommonModel getCommonModel() {
        CommonModel commonModel = new CommonModelProvider(dataRegistry, androidContext, deviceRealtimeDataProvider).provide();
        return commonModel;
    }
}
