package io.adbrix.sdk.data.dataprovider;

import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.res.Configuration;
import android.graphics.Point;
import android.location.Geocoder;
import android.net.ConnectivityManager;
import android.net.Network;
import android.net.NetworkCapabilities;
import android.net.NetworkInfo;
import android.os.BatteryManager;
import android.os.Build;
import android.provider.Settings;
import android.telephony.TelephonyManager;
import android.text.TextUtils;
import android.view.Display;
import android.view.WindowManager;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;
import java.util.UUID;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.IObservable;
import io.adbrix.sdk.component.IObserver;
import io.adbrix.sdk.configuration.IABXContextController;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.CoreUtils;

public class DeviceRealtimeDataProvider extends AbstractDataProvider implements IObservable<LogEventParameter> {
    private Context context;
    protected IABXContextController defaultABXContextController;
    private List<IObserver<LogEventParameter>> observers = new ArrayList<>();
    private IGoogleAdvertisingIdProvider googleAdvertisingIdProvider;
    public static final String GAID_ESCAPE_STRING = "00000000-0000-0000-0000-000000000000";

    public DeviceRealtimeDataProvider(DataRegistry dataRegistry, Context context, IABXContextController defaultABXContextController, IGoogleAdvertisingIdProvider googleAdvertisingIdProvider) {
        super(dataRegistry);
        this.context = context;
        this.defaultABXContextController = defaultABXContextController;
        this.googleAdvertisingIdProvider = googleAdvertisingIdProvider;
        //TODO InAppMessage -> dev 머지 후 아래 주석 해제
        //add(DefaultABXContextController.getInstance());
    }

    public void refreshAdInfo(IDataProviderResultListener listener){
        googleAdvertisingIdProvider.requestAdInfo(listener, new GoogleAdvertisingIdProvider.Callback() {
            @Override
            public void getAdInfo(IGoogleAdvertisingIdProvider.AdInfo adInfo) {
                putADID(adInfo);
                putGAID(adInfo.advertisingId);
                putBoolean(DataRegistryKey.BOOLEAN_AD_ID_OPT_OUT, adInfo.limitAdTracking, 5, true);
                putString(DataRegistryKey.STRING_IGAW_ID, getIgawId(adInfo), 5, true);
                listener.onSuccess();
            }
        });
    }

    public void refreshData() {
        putString(DataRegistryKey.STRING_RESOLUTION, getResolution(), 5, false);
        putBoolean(DataRegistryKey.BOOLEAN_IS_PORTRAIT, isPortrait(), 5, false);
        putString(DataRegistryKey.STRING_NETWORK, getNetwork(), 5, false);
        putInt(DataRegistryKey.INT_BATT_L, getBatteryLevel(), 5, false);
        putBoolean(DataRegistryKey.BOOLEAN_BATT_C, isBatteryCharging(), 5, false);
        putString(DataRegistryKey.STRING_LANGUAGE, getLanguage(), 5, false);
        putString(DataRegistryKey.STRING_COUNTRY, getCountry(), 5, true);
        putInt(DataRegistryKey.INT_TIME_ZONE_OFFSET, getTimeZoneOffset(), 5, true);
        putInt(DataRegistryKey.INT_DEVICE_TIME_TYPE, getDeviceTimeType(), 5, true);
    }

    private void putADID(IGoogleAdvertisingIdProvider.AdInfo adInfo) {
        String storedAdid = safeGetString(DataRegistryKey.STRING_ADID, null);
        if (TextUtils.isEmpty(storedAdid)) {
            String adid;
            if (isAvailableAdid(adInfo))
                //1. 가능한 경우 광고식별자 사용
                adid = adInfo.advertisingId;
            else {
                //2. 사용자 광고 추적 제한으로 인해 사용 못할 경우 Preference에 저장된 UUID 사용
                String uuid = safeGetString(DataRegistryKey.STRING_UUID, null);
                adid = uuid != null ? "uuid:" + uuid : null;
            }
            putString(DataRegistryKey.STRING_ADID, adid, 5, true);
        }
    }

    private void putGAID(String gaid) {
        if (gaid == null || gaid.equals(GAID_ESCAPE_STRING)) {
            gaid = "";
        }

        putString(DataRegistryKey.STRING_GAID, gaid, 5, true);
    }

    @Override
    public void provideDefaultValues() {
        putString(DataRegistryKey.STRING_RESOLUTION, getResolution(), 5, false);
        putBoolean(DataRegistryKey.BOOLEAN_IS_PORTRAIT, isPortrait(), 5, false);
        putString(DataRegistryKey.STRING_NETWORK, getNetwork(), 5, false);
        putInt(DataRegistryKey.INT_BATT_L, getBatteryLevel(), 5, false);
        putBoolean(DataRegistryKey.BOOLEAN_BATT_C, isBatteryCharging(), 5, false);
        putString(DataRegistryKey.STRING_LANGUAGE, getLanguage(), 5, false);
        putString(DataRegistryKey.STRING_COUNTRY, getCountry(), 5, true);
        putInt(DataRegistryKey.INT_TIME_ZONE_OFFSET, getTimeZoneOffset(), 5, true);
        putInt(DataRegistryKey.INT_DEVICE_TIME_TYPE, getDeviceTimeType(), 5, true);
    }

    @Override
    public void asyncProvide(IDataProviderResultListener dataProviderResultListener) {
        this.refreshData();
        this.refreshAdInfo(dataProviderResultListener);
    }

    private String getRequestDateTime() {
        return CommonUtils.getCurrentUTCInDBFormat();
    }

    private String getResolution() {
        WindowManager windowManager = ((WindowManager) context.getSystemService(Context.WINDOW_SERVICE));
        if (windowManager == null) {
            // Remaster에서는 windowManager가 null인 경우를 catch하지 않았다.
            return null;
        }
        Display defaultDisplay = windowManager.getDefaultDisplay();
        Point point = new Point();
        defaultDisplay.getSize(point);

        if (isPortrait()) {
            return point.x + "x" + point.y;
        } else {
            return point.y + "x" + point.x;
        }
    }

    private boolean isPortrait() {
        return context.getResources().getConfiguration().orientation == Configuration.ORIENTATION_PORTRAIT;
    }

    private String getNetwork() {
        ConnectivityManager connectivityManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        if (connectivityManager == null)
            return "unknown";

        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.M) {
            Network network = connectivityManager.getActiveNetwork();
            if (network != null) {
                NetworkCapabilities networkCapabilities = connectivityManager.getNetworkCapabilities(network);
                if (networkCapabilities != null) {
                    if (networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI)) {
                        return "wifi";
                    } else if (networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR)) {
                        return "mobile";
                    }
                }
            }
        } else {
            try {
                NetworkInfo mobile = connectivityManager.getNetworkInfo(0);
                NetworkInfo wifi = connectivityManager.getNetworkInfo(1);

                if (mobile != null &&
                        (mobile.getState() == NetworkInfo.State.CONNECTED || mobile.getState() == NetworkInfo.State.CONNECTING)) {
                    return "mobile";
                } else if (wifi != null &&
                        (wifi.getState() == NetworkInfo.State.CONNECTED || wifi.getState() == NetworkInfo.State.CONNECTING)) {
                    return "wifi";
                }
            } catch (Exception e) {
                AbxLog.w(Arrays.toString(e.getStackTrace()), true);
            }
        }

        return "unknown";
    }

    private String getLanguage() {
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.N) {
            try {
                if (context != null &&
                        context.getResources() != null &&
                        context.getResources().getConfiguration() != null &&
                        context.getResources().getConfiguration().getLocales().size() > 0) {
                    String language = context.getResources().getConfiguration().getLocales().get(0).getLanguage();
                    if (!CommonUtils.isNullOrEmpty(language)) {
                        return language.toUpperCase(Locale.US);
                    }
                }
            } catch (Exception e) {
                // 예외 처리
            }
        }

        String defaultLanguage = Locale.getDefault().getLanguage();
        if (CommonUtils.isNullOrEmpty(defaultLanguage)) {
            return "unknown";
        } else {
            return defaultLanguage.toUpperCase(Locale.US);
        }
    }

    public int getBatteryLevel() {
        Intent intent = context.registerReceiver(null, new IntentFilter(Intent.ACTION_BATTERY_CHANGED));

        int level = -1;
        int scale = -1;
        if (intent != null) {
            level = intent.getIntExtra(BatteryManager.EXTRA_LEVEL, -1);
            scale = intent.getIntExtra(BatteryManager.EXTRA_SCALE, -1);
        }

        return (int) ((level / (float) scale) * 100);
    }

    private boolean isBatteryCharging() {
        Intent intent = context.registerReceiver(null, new IntentFilter(Intent.ACTION_BATTERY_CHANGED));

        int state = -1;
        if (intent != null) {
            state = intent.getIntExtra(BatteryManager.EXTRA_PLUGGED, -1);
        }

        return state == BatteryManager.BATTERY_PLUGGED_AC ||
                state == BatteryManager.BATTERY_PLUGGED_USB ||
                (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1 && state == BatteryManager.BATTERY_PLUGGED_WIRELESS);
    }

    /**
     * 2021.02.18 - walter
     * 개인 정보 이슈로 위치정보 수집 기능을 제거함.
     * 따라서 위치 기반 Country Code 수집도 제외함.
     *
     * @return
     */
    private String getCountry() {
        // This should not be called on the main thread.

        // Prioritize reverse geocode, but until we have a result from that,
        // we try to grab the country from the network, and finally the locale

        try {
//            String country = getCountryFromLocation();
//            if (!CommonUtils.isNullOrEmpty(country))
//            {
//                return country;
//            }

            TelephonyManager telephonyManager = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
            if (telephonyManager.getSimState() == TelephonyManager.SIM_STATE_READY &&
                    telephonyManager.getPhoneType() != TelephonyManager.PHONE_TYPE_CDMA) {
                String countryCode = telephonyManager.getNetworkCountryIso();
                if (!CommonUtils.isNullOrEmpty(countryCode)) {
                    return countryCode.toUpperCase(Locale.US);
                }
            }
        } catch (Exception e) {
        }

        return Locale.getDefault().getCountry();
    }

    // @VisibleForTesting
    protected Geocoder getGeocoder() {
        return new Geocoder(context, Locale.ENGLISH);
    }

    // TODO Realtime인지, Static인지
    private int getTimeZoneOffset() {
        try {
            TimeZone defaultTimeZone = TimeZone.getDefault();
            String gmt = defaultTimeZone.getDisplayName(
                    defaultTimeZone.inDaylightTime(new Date()),
                    TimeZone.SHORT
            );

            if (!CommonUtils.isNullOrEmpty(gmt) && gmt.length() == 9) {
                String[] timeArr = gmt.substring(4).split(":");
                int absOffset = Integer.valueOf(timeArr[0]) * 60 + Integer.valueOf(timeArr[1]);
                return gmt.charAt(3) == '+' ? absOffset : absOffset * -1;
            }
        } catch (AssertionError | Exception e) {
            AbxLog.w(Arrays.toString(e.getStackTrace()), true);
        }

        return 0;
    }

    private int getDeviceTimeType() {
        int useAutoTime;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            useAutoTime = Settings.Global.getInt(
                    context.getContentResolver(),
                    Settings.Global.AUTO_TIME,
                    -1);
        } else {
            useAutoTime = android.provider.Settings.System.getInt(
                    context.getContentResolver(),
                    Settings.System.AUTO_TIME,
                    -1);
        }

        if (useAutoTime == 1) {
            return 1;
        } else if (useAutoTime == 0) {
            return 2;
        } else {
            return 0;
        }
    }



    private String getIgawId(IGoogleAdvertisingIdProvider.AdInfo adInfo) {
        if (isAvailableAdid(adInfo)) {
            //1. 가능한 경우 광고식별자 사용
            return "iga:" + CoreUtils.md5(adInfo.advertisingId + "!gaWrk");
        }
        String dfnId = safeGetString(DataRegistryKey.STRING_DFN_ID, null);
        if(!TextUtils.isEmpty(dfnId)){
            //2. 광고 식별자 사용 못할 경우 dfnId 사용
            return "iga:" + dfnId;
        }
        //3. dfnId 없을 경우 저장되어 있던 uuid 사용
        String uuid = safeGetString(DataRegistryKey.STRING_UUID, null);
        return "iga:" + CoreUtils.md5(uuid + "!gaWrk");
    }

    @Override
    public void add(IObserver<LogEventParameter> observer) {
        observers.add(observer);
    }

    @Override
    public void delete(IObserver<LogEventParameter> observer) {
        observers.remove(observer);
    }

    @Override
    public void notifyObserver(LogEventParameter logEventParameter) {
        for (IObserver<LogEventParameter> observer : observers) {
            observer.update(logEventParameter);
        }
    }

    @Override
    public void clear() {
        observers.clear();
    }

    private boolean isAvailableAdid(IGoogleAdvertisingIdProvider.AdInfo adInfo){
        boolean result = false;
        if(adInfo == null){
            return result;
        }
        if(TextUtils.isEmpty(adInfo.advertisingId)){
            return result;
        }
        if("00000000-0000-0000-0000-000000000000".equals(adInfo.advertisingId)){
            return result;
        }
        result = true;
        return result;
    }
}
