package io.adbrix.sdk.configuration;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;

import com.igaworks.v2.core.AdBrixRm;

import org.json.JSONObject;

import java.util.HashMap;
import java.util.List;

import javax.annotation.Nullable;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.GooglePlayReferrerProperties;
import io.adbrix.sdk.component.IABXComponentsFactory;
import io.adbrix.sdk.component.IObserver;
import io.adbrix.sdk.component.autoEvent.DailyFirstOpenAutoEventGenerator;
import io.adbrix.sdk.component.autoEvent.DeeplinkAutoEventGenerator;
import io.adbrix.sdk.component.autoEvent.SessionAutoEventGenerator;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.domain.interactor.ClearUserPropertyUseCase;
import io.adbrix.sdk.domain.interactor.DeleteAllInAppMessageDBContentsUseCase;
import io.adbrix.sdk.domain.interactor.GetAllSelfServeInAppMessagesUseCase;
import io.adbrix.sdk.domain.interactor.GetSubscriptionStatusUseCase;
import io.adbrix.sdk.domain.interactor.GetUserIdUseCase;
import io.adbrix.sdk.domain.interactor.LoginUseCase;
import io.adbrix.sdk.domain.interactor.LogoutUseCase;
import io.adbrix.sdk.domain.interactor.RequestGetAttributionDataUseCase;
import io.adbrix.sdk.domain.interactor.SetCiPropertyUseCase;
import io.adbrix.sdk.domain.interactor.SetSubscriptionStatusUseCase;
import io.adbrix.sdk.domain.model.DfnInAppMessage;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.LogMessageFormat;
import io.adbrix.sdk.domain.Repository;
import io.adbrix.sdk.domain.function.Completion;
import io.adbrix.sdk.domain.interactor.ClearAllActionHistoryInLocalDBUseCase;
import io.adbrix.sdk.domain.interactor.ClearSyncedActionHistoryInLocalDBUseCase;
import io.adbrix.sdk.domain.interactor.DeleteActionHistoryUseCase;
import io.adbrix.sdk.domain.interactor.DeleteAllActionHistoryUseCase;
import io.adbrix.sdk.domain.interactor.DeleteUserDataUseCase;
import io.adbrix.sdk.domain.interactor.FetchActionHistoryFromServerUseCase;
import io.adbrix.sdk.domain.interactor.FetchInAppMessageUseCase;
import io.adbrix.sdk.domain.interactor.FlushAllNowUseCase;
import io.adbrix.sdk.domain.interactor.FlushAtIntervalsUseCase;
import io.adbrix.sdk.domain.interactor.GetActionHistoryUseCase;
import io.adbrix.sdk.domain.interactor.GetAllActionHistoryUseCase;
import io.adbrix.sdk.domain.interactor.GetAllInAppMessageUseCase;
import io.adbrix.sdk.domain.interactor.InsertPushDataUseCase;
import io.adbrix.sdk.domain.interactor.LogEventUseCase;
import io.adbrix.sdk.domain.interactor.LogSameEventWithPagingUseCase;
import io.adbrix.sdk.domain.interactor.OpenInAppMessageUseCase;
import io.adbrix.sdk.domain.interactor.RestartSDKUseCase;
import io.adbrix.sdk.domain.interactor.SaveUnsentEventsUseCase;
import io.adbrix.sdk.domain.interactor.SaveUserPropertyUseCaseWithoutEvent;
import io.adbrix.sdk.domain.model.ActionHistory;
import io.adbrix.sdk.domain.model.ActionHistoryIdType;
import io.adbrix.sdk.domain.model.Empty;
import io.adbrix.sdk.domain.model.SelfServeInAppMessage;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.domain.model.Response;
import io.adbrix.sdk.domain.model.Result;
import io.adbrix.sdk.domain.model.ResultCallback;
import io.adbrix.sdk.domain.model.SubscriptionStatus;
import io.adbrix.sdk.domain.model.UserPropertyCommand;
import io.adbrix.sdk.utils.CoreUtils;

public class ActiveABXContext implements IABXContext, IObserver<LogEventParameter> {
    private Repository repository;
    private IABXComponentsFactory componentsFactory;
    private SessionAutoEventGenerator sessionAutoEventGenerator;
    private DeeplinkAutoEventGenerator deeplinkAutoEventGenerator;
    private DailyFirstOpenAutoEventGenerator dailyFirstOpenAutoEventGenerator;
    private Context context;

    public ActiveABXContext(
            IABXComponentsFactory componentsFactory,
            Repository repository
    ) {
        this.componentsFactory = componentsFactory;
        this.repository = repository;
        this.sessionAutoEventGenerator = componentsFactory.createOrGetSessionAutoEventGenerator();
        this.sessionAutoEventGenerator.add(this);
        this.deeplinkAutoEventGenerator = componentsFactory.createOrGetDeeplinkAutoEventGenerator();
        this.deeplinkAutoEventGenerator.add(this);
        this.dailyFirstOpenAutoEventGenerator = componentsFactory.createOrGetDailyFirstOpenAutoEventGenerator();
        this.dailyFirstOpenAutoEventGenerator.add(this);
        this.context = componentsFactory.getAndroidContext();
        if (CoreUtils.isOnline(context)) {
            new FlushAllNowUseCase(repository).execute();
        }
    }

    @Override
    public void update(LogEventParameter logEventParameter) {
        this.logEvent(logEventParameter);
    }

    @Override
    public void initialize(Context context, String appkey, String secretkey, ResultCallback<String> callback) {
        AbxLog.w(String.format(LogMessageFormat.INVALID_CONTEXT, "Active", "initialize"), true);
        callback.callback(ResultCallback.Status.SUCCESS, String.format(LogMessageFormat.INVALID_CONTEXT, "Active", "initialize"));
    }

    @Override
    public void getInstallReferrer() {
        GooglePlayReferrerProperties googlePlayReferrerProperties = new GooglePlayReferrerProperties(componentsFactory.getAndroidContext(), componentsFactory);
        if(googlePlayReferrerProperties.needRefresh()){
            googlePlayReferrerProperties.saveInstallReferrerCompleted(false);
            googlePlayReferrerProperties.startConnection(new GooglePlayReferrerProperties.Callback() {
                @Override
                public void getReferrerMap(HashMap<String, Object> referrerMap, GooglePlayReferrerProperties googlePlayReferrerProperties) {
                    AbxLog.d("getInstallReferrer success: "+referrerMap.toString(), true);
                    googlePlayReferrerProperties.endConnection();
                    componentsFactory.getAbxContextController().flushQueuedEvent();
                }
            });
        }
    }

    @Override
    public void saveUserProperty(UserPropertyCommand userPropertyCommand) {

        if (saveUserPropertyWithoutEvent(userPropertyCommand)) {
            LogEventParameter eventParameter = new LogEventParameter(
                    CoreConstants.GROUP_ABX,
                    CoreConstants.EVENT_USER_PROPERTY_CHANGED,
                    null,
                    0,
                    0
            );

            logEvent(eventParameter);
        }
    }

    @Override
    public Boolean saveUserPropertyWithoutEvent(UserPropertyCommand userPropertyCommand) {
        return new SaveUserPropertyUseCaseWithoutEvent(repository).execute(userPropertyCommand);
    }

    @Override
    public void clearUserProperty() {
        new ClearUserPropertyUseCase(repository).execute();
    }

    @Override
    public void getUserId(Completion<Result<String>> completion) {
        new GetUserIdUseCase(repository, completion).execute();
    }

    @Override
    public void login(String userId, Completion<Result<Response>> completion) {
        new LoginUseCase(repository, userId, completion).execute();
    }

    @Override
    public void logout(Completion<Result<Response>> completion) {
        new LogoutUseCase(repository, completion).execute();
    }

    @Override
    public void logEvent(LogEventParameter logEventParameter) {
        new LogEventUseCase(repository).execute(logEventParameter);
    }

    @Override
    public void logSameEventWithPaging(String eventName, String group, List<JSONObject> eventParamList) {
        new LogSameEventWithPagingUseCase(repository).execute(eventName, group, eventParamList);
    }

    @Override
    public void flushAllEvents(Completion<Result<Empty>> completion) {
        new FlushAllNowUseCase(repository, completion).execute();
    }

    @Override
    public void runOnIdleTime() {
        new FlushAtIntervalsUseCase(repository).execute();
    }

    @Override
    public void onResume(Activity activity) {
        this.sessionAutoEventGenerator.onResume(activity);
        this.dailyFirstOpenAutoEventGenerator.process();
    }

    @Override
    public void onPause() {
        this.sessionAutoEventGenerator.onPause();
        if (CoreUtils.isOnline(this.context)) {
            new FlushAllNowUseCase(repository).execute();
        }

        new SaveUnsentEventsUseCase(repository).execute();
    }

    @Override
    public void deeplink(Activity deeplinkActivity) {
        try {
            if (this.sessionAutoEventGenerator.isStartSessionByDeeplink(deeplinkActivity)) {
                return;
            }
            this.deeplinkAutoEventGenerator.process(deeplinkActivity);

        } catch (Exception e) {
            AbxLog.e(e, true);
        }
    }

    @Override
    public void deeplinkWithIntent(Intent deeplinkIntent) {
        try {
            if (this.sessionAutoEventGenerator.isStartSessionByDeeplinkIntent(deeplinkIntent)) {
                return;
            }
            this.deeplinkAutoEventGenerator.process(deeplinkIntent);

        } catch (Exception e) {
            AbxLog.e(e, true);
        }
    }

    @Override
    public void putDataRegistry(DataUnit dataUnit) {
        DataRegistry dataRegistry = componentsFactory.createOrGetDataRegistry();
        dataRegistry.putDataRegistry(dataUnit);
    }

    @Override
    public void deleteUserData(String userId, ResultCallback<String> callback) {
        AbxLog.d("Delete user data api called!", true);
        new DeleteUserDataUseCase(repository, userId, callback).execute();
    }

    @Override
    public void restartSDK(String userId, ResultCallback<String> callback) {
        AbxLog.d("Restart SDK api called!", true);
        new RestartSDKUseCase(repository, userId, callback).execute();
    }

    @Override
    public void fetchActionHistoryFromServer(@Nullable String token, ActionHistoryIdType idType, List<String> actionType, Completion<Result<List<ActionHistory>>> completion) {
        new FetchActionHistoryFromServerUseCase(repository, token, idType, actionType, completion).execute();
    }

    @Override
    public void insertPushData(String pushDataString) {
        new InsertPushDataUseCase(repository, pushDataString).execute();
    }

    @Override
    public void getActionHistory(int skip, int limit, List<String> actionType, Completion<Result<List<ActionHistory>>> completion) {
        new GetActionHistoryUseCase(repository, skip, limit, actionType, completion).execute();
    }

    @Override
    public void getAllActionHistory(List<String> actionType, Completion<Result<List<ActionHistory>>> completion) {
        new GetAllActionHistoryUseCase(repository, actionType, completion).execute();
    }

    @Override
    public void deleteActionHistory(@Nullable String token, String historyId, long timestamp, Completion<Result<Empty>> completion) {
        new DeleteActionHistoryUseCase(repository, token, historyId, timestamp, completion).execute();
    }

    @Override
    public void deleteAllActionHistory(@Nullable String token, @Nullable ActionHistoryIdType idType, Completion<Result<Empty>> completion) {
        new DeleteAllActionHistoryUseCase(repository, token, idType, completion).execute();
    }

    @Override
    public void clearSyncedActionHistoryInLocalDB(Completion<Result<Empty>> completion) {
        new ClearSyncedActionHistoryInLocalDBUseCase(repository, completion).execute();
    }

    @Override
    public void clearAllActionHistoryInLocalDB() {
        new ClearAllActionHistoryInLocalDBUseCase(repository).execute();
    }

    @Override
    public void fetchInAppMessage(boolean isDirectCall, Completion<Result<Empty>> completion) {
        new FetchInAppMessageUseCase(isDirectCall, repository, completion).execute();
    }

    @Override
    public void getAllInAppMessage(Completion<Result<List<DfnInAppMessage>>> completion) {
        new GetAllInAppMessageUseCase(repository, completion).execute();
    }

    @Override
    public void getSelfServeInAppMessages(AdBrixRm.GetSelfServeInAppMessagesCallback callback){
        new GetAllSelfServeInAppMessagesUseCase(repository, callback).execute();
    }

    @Override
    public void openInAppMessage(String campaignId, Completion<Result<Empty>> completion) {
        new OpenInAppMessageUseCase(repository, campaignId, completion).execute();
    }

    @Override
    public void deleteAllInAppMessageDBContents() {
        new DeleteAllInAppMessageDBContentsUseCase(repository).execute();
    }

    @Override
    public void requestGetAttributionData(String logId, AdBrixRm.GetAttributionDataCallback callback, DataRegistry dataRegistry) {
        new RequestGetAttributionDataUseCase(logId, callback, dataRegistry, repository).execute();
    }

    @Override
    public void getSubscriptionStatus(AdBrixRm.GetSubscriptionStatusCallback callback, DataRegistry dataRegistry) {
        new GetSubscriptionStatusUseCase(callback, dataRegistry, repository).execute();
    }

    @Override
    public void setSubscriptionStatus(SubscriptionStatus subscriptionStatus, AdBrixRm.SetSubscriptionStatusCallback callback, DataRegistry dataRegistry) {
        new SetSubscriptionStatusUseCase(subscriptionStatus, callback, dataRegistry, repository).execute();
    }

    @Override
    public void setCiProperty(String key, String value, AdBrixRm.SetCiProfileCallback callback, DataRegistry dataRegistry) {
        new SetCiPropertyUseCase(key, value, callback, dataRegistry, repository).execute();
    }
}
