package io.adbrix.sdk.ui.inappmessage;

import android.app.Activity;
import android.content.res.Configuration;
import android.graphics.Bitmap;

import org.json.JSONObject;

import java.lang.ref.SoftReference;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.IABXComponentsFactory;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.domain.IAMConstants;
import io.adbrix.sdk.domain.function.Completion;
import io.adbrix.sdk.domain.model.DfnInAppMessage;
import io.adbrix.sdk.domain.model.Empty;
import io.adbrix.sdk.domain.model.Error;
import io.adbrix.sdk.domain.model.IAMEnums;
import io.adbrix.sdk.domain.model.Result;
import io.adbrix.sdk.domain.model.Success;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.HttpConnectionUtils;

public class InAppMessageManager {
    private SoftReference<IInAppMessageViewFactory> carriedInAppMessageViewFactory;
    private SoftReference<DfnInAppMessage> carriedInAppMessage;
    private SoftReference<Activity> currentActivity;
    private final AtomicBoolean isPresented = new AtomicBoolean(false);
    private final AtomicBoolean isInProgress = new AtomicBoolean(false);
    private final AtomicBoolean isInAnimating = new AtomicBoolean(false);
    private final AtomicBoolean isPaused = new AtomicBoolean(false);
    private final List<SuspendedUpdateImageInfo> suspendedUpdateImageInfoList = new ArrayList<>();
    private IABXComponentsFactory factory;
    private ExecutorService executorService;

    public static final String FULL_SCREEN = "full_screen";
    public static final String MODAL = "modal";
    public static final String STICKY_BANNER = "sticky_banner";
    public void showByEventName(String eventName){
        DfnInAppMessage inAppMessage = getInAppMessageByEventName(eventName);
        show(inAppMessage, null);
    }
    public void showByEventNameWithParam(String eventName, JSONObject eventParam){
        DfnInAppMessage inAppMessage;
        if(CommonUtils.isNullOrEmpty(eventParam)){
            inAppMessage = getInAppMessageByEventName(eventName);
        } else{
            inAppMessage = getInAppMessageByEventNameWithParam(eventName, eventParam);
        }
        show(inAppMessage, null);
    }
    public void showByEventNameWithParamList(String eventName, List<JSONObject> eventParamList){
        DfnInAppMessage inAppMessage;
        if(CommonUtils.isNullOrEmpty(eventParamList)){
            inAppMessage = getInAppMessageByEventName(eventName);
        } else{
            inAppMessage = getInAppMessageByEventNameWithParamList(eventName, eventParamList);
        }
        show(inAppMessage, null);
    }
    public void showByCampaignId(String campaignId, Completion<Result<Empty>> completion){
        DfnInAppMessage inAppMessage = getInAppMessageByCampaignId(campaignId);
        show(inAppMessage, completion);
    }

    public void show(DfnInAppMessage inAppMessage, Completion<Result<Empty>> completion){
        show(inAppMessage, completion, false);
    }

    public void show(DfnInAppMessage inAppMessage, Completion<Result<Empty>> completion, boolean isCarried){
        ExecutorService executorService = getExecutorService();
        if (executorService == null){
            AbxLog.w("show() inAppMessageExecutorService is null", true);
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("inAppMessageExecutorService is null"));
            }
            return;
        }
        if(executorService.isShutdown()){
            AbxLog.w("show() inAppMessageExecutorService is shutdown", true);
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("inAppMessageExecutorService is shutdown"));
            }
            return;
        }
        if(inAppMessage == null){
            AbxLog.w("show() No Valid InAppMessage", true);
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("No Valid InAppMessage"));
            }
            return;
        }
        Activity activity = getCurrentActivity();
        if(CommonUtils.isNull(activity)){
            AbxLog.w("show() activity is null", true);
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("activity is null"));
            }
            return;
        }
        if(CommonUtils.isNull(factory)){
            AbxLog.w("show() factory is null", true);
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("factory is null"));
            }
            return;
        }
        if (factory.createOrGetDataRegistry().safeGetLong(DataRegistryKey.LONG_S3_CONFIG_IN_APP_MESSAGE_ACTIVE, -1) != 1) {
            AbxLog.w("show() InAppMessage is not active", true);
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("InAppMessage is not active"));
            }
            return;
        }
        if(isShowing()){
            AbxLog.w("show() InAppMessage is already shown", true);
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("InAppMessage is already shown"));
            }
            return;
        }
        if(CommonUtils.notNull(carriedInAppMessageViewFactory)){
            if(CommonUtils.notNull(carriedInAppMessageViewFactory.get())){
                AbxLog.w("show() inAppMessageViewFactory is not null", true);
                if(CommonUtils.notNull(completion)){
                    completion.handle(Error.of("inAppMessageViewFactory is not null"));
                }
                return;
            }
        }
        if(isInProgress()){
            AbxLog.w("show() InAppMessage is in progress", true);
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("InAppMessage is in progress"));
            }
            return;
        }
        if(isPaused.get()){
            AbxLog.w("InAppMessage is paused. If you want to use, then you should call AdbrixRM.resumeInAppMessage().", true);
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("InAppMessage is paused. If you want to use, then you should call AdbrixRM.resumeInAppMessage()."));
            }
            return;
        }
        setIsInProgress(true);
        final DfnInAppMessage finalSetInAppMessage;
        final boolean finalSetIsCarried;
        if(!isCarried){
            DfnInAppMessage carriedInAppMessage = getCarriedInAppMessage();
            if(CommonUtils.notNull(carriedInAppMessage)){
                finalSetInAppMessage = carriedInAppMessage;
                finalSetIsCarried = true;
                flushCarriedInAppMessage();
            } else{
                finalSetIsCarried = isCarried;
                finalSetInAppMessage = inAppMessage;
            }
        }else{
            finalSetIsCarried = isCarried;
            finalSetInAppMessage = inAppMessage;
        }
        executorService.submit(new Runnable() {
            @Override
            public void run() {
                InAppMessageDAO inAppMessageDAO = getInAppMessageDao(activity);
                inAppMessageDAO.increaseFrequencyByCampaignId(finalSetInAppMessage.getCampaignId());
                InAppMessageFactoryContainer inAppMessageFactoryContainer = new InAppMessageFactoryContainer(inAppMessageDAO,factory.getAbxContextController());
                IInAppMessageViewFactory inAppMessageViewFactory = inAppMessageFactoryContainer.getInAppMessageViewFactory(finalSetInAppMessage);
                carriedInAppMessageViewFactory = new SoftReference<>(inAppMessageViewFactory);
                carriedInAppMessageViewFactory.get().createInAppMessage(activity, finalSetInAppMessage, finalSetIsCarried);
                if(CommonUtils.notNull(completion)){
                    completion.handle(Success.empty());
                }
            }
        });
    }

    public void close(IAMEnums.CloseType closeType){
        Activity activity = getCurrentActivity();
        if(CommonUtils.isNull(activity)){
            AbxLog.w("onClose() activity is null", true);
            cacheInAppMessage(closeType);
            flushCarriedInAppMessageViewFactory();
            setIsInProgress(false);
            return;
        }
        if(!isShowing()){
            AbxLog.w("onClose() inAppMessage is not shown", true);
            cacheInAppMessage(closeType);
            flushCarriedInAppMessageViewFactory();
            setIsInProgress(false);
            return;
        }
        if(CommonUtils.isNull(carriedInAppMessageViewFactory)){
            AbxLog.w("onClose() inAppMessageViewFactory Reference is null", true);
            return;
        }
        if(CommonUtils.isNull(carriedInAppMessageViewFactory.get())){
            AbxLog.w("onClose() inAppMessageViewFactory is null", true);
            return;
        }
        setIsInProgress(true);
        cacheInAppMessage(closeType);
        carriedInAppMessageViewFactory.get().dismissInAppMessage(activity, closeType);
    }

    private void cacheInAppMessage(IAMEnums.CloseType closeType){
        if(CommonUtils.isNull(carriedInAppMessageViewFactory)){
            return;
        }
        if(CommonUtils.isNull(carriedInAppMessageViewFactory.get())){
            return;
        }
        if(CommonUtils.isNull(carriedInAppMessageViewFactory.get().getInAppMessage())){
            return;
        }
        switch (closeType){
            case ON_PAUSE:
            case OPEN_DEEPLINK:{
                this.carriedInAppMessage = new SoftReference<>(carriedInAppMessageViewFactory.get().getInAppMessage());
                break;
            }
        }
    }

    public boolean isShowing(){
        return isPresented.get();
    }
    public void setShowing(boolean isShown){
        isPresented.set(isShown);
        if(!isShown){
            flushCarriedInAppMessageViewFactory();
        }
    }

    public void setIsInProgress(boolean isInProgress) {
        this.isInProgress.set(isInProgress);
    }

    public boolean isInProgress() {
        return isInProgress.get();
    }

    public void setIsInAnimating(boolean isInAnimating){
        this.isInAnimating.set(isInAnimating);
    }
    public boolean isInAnimating(){
        return isInAnimating.get();
    }

    public Activity getCurrentActivity() {
        if(currentActivity == null){
            return null;
        }
        return currentActivity.get();
    }

    public void setCurrentActivity(Activity currentActivity) {
        if(currentActivity == null){
            if(CommonUtils.notNull(this.currentActivity)){
                this.currentActivity.clear();
            }
            return;
        }
        if(CommonUtils.notNull(this.currentActivity)){
            this.currentActivity.clear();
        }
        this.currentActivity = new SoftReference<>(currentActivity);
    }

    public void init(IABXComponentsFactory factory) {
        this.factory = factory;
    }

    private ExecutorService getExecutorService(){
        if (executorService == null || executorService.isShutdown()){
            executorService = Executors.newSingleThreadExecutor();
            return executorService;
        }
        return executorService;
    }

    public DfnInAppMessage getCarriedInAppMessage() {
        if(CommonUtils.isNull(carriedInAppMessage)){
            return null;
        }
        return carriedInAppMessage.get();
    }

    private void flushCarriedInAppMessageViewFactory(){
        if(CommonUtils.notNull(carriedInAppMessageViewFactory)){
            carriedInAppMessageViewFactory.clear();
            carriedInAppMessageViewFactory = null;
        }
    }
    public void flushCarriedInAppMessage(){
        if(CommonUtils.notNull(carriedInAppMessage)){
            carriedInAppMessage.clear();
            carriedInAppMessage = null;
        }
    }

    private DfnInAppMessage getInAppMessageByCampaignId(String campaignId){
        if(!isInAppMessageActive()){
            return null;
        }
        Activity activity = getCurrentActivity();
        if(CommonUtils.isNull(activity)){
            AbxLog.d("getInAppMessageByCampaignId activity is null", true);
            return null;
        }
        if(CommonUtils.isNull(factory)){
            AbxLog.d("factory is null", true);
            return null;
        }

        InAppMessageDAO inAppMessageDAO = getInAppMessageDao(activity);
        DfnInAppMessage inAppMessage = null;
        try {
            inAppMessage = inAppMessageDAO.getInAppMessageByCampaignId(campaignId);
        }catch (Exception e){
            AbxLog.w(e, true);
        }
        return inAppMessage;
    }

    public DfnInAppMessage getInAppMessageByEventName(String eventName){
        if(!isInAppMessageActive()){
            return null;
        }
        Activity activity = getCurrentActivity();
        if(CommonUtils.isNull(activity)){
            AbxLog.d("getInAppMessageByEventName activity is null", true);
            return null;
        }
        if(CommonUtils.isNull(factory)){
            AbxLog.d("factory is null", true);
            return null;
        }
        boolean isPortrait = isPortrait(activity);
        InAppMessageDAO inAppMessageDAO = getInAppMessageDao(activity);
        DfnInAppMessage inAppMessage = null;
        try {
            inAppMessage = inAppMessageDAO.getInAppMessageByEventName(eventName);
        }catch (Exception e){
            AbxLog.w(e, true);
        }
        return inAppMessage;
    }

    public DfnInAppMessage getInAppMessageByEventNameWithParam(String eventName, JSONObject eventParam){
        if(!isInAppMessageActive()){
            return null;
        }
        Activity activity = getCurrentActivity();
        if(CommonUtils.isNull(activity)){
            AbxLog.d("getInAppMessageByEventNameWithParam activity is null", true);
            return null;
        }
        if(CommonUtils.isNull(factory)){
            AbxLog.d("factory is null", true);
            return null;
        }

        InAppMessageDAO inAppMessageDAO = getInAppMessageDao(activity);
        DfnInAppMessage inAppMessage = null;
        try {
            inAppMessage = inAppMessageDAO.getInAppMessageByEventNameWithParam(eventName, eventParam);
        }catch (Exception e){
            AbxLog.w(e, true);
        }
        return inAppMessage;
    }

    public DfnInAppMessage getInAppMessageByEventNameWithParamList(String eventName, List<JSONObject> eventParamList){
        if(!isInAppMessageActive()){
            return null;
        }
        Activity activity = getCurrentActivity();
        if(CommonUtils.isNull(activity)){
            AbxLog.d("getInAppMessageByEventNameWithParamList activity is null", true);
            return null;
        }
        if(CommonUtils.isNull(factory)){
            AbxLog.d("factory is null", true);
            return null;
        }

        InAppMessageDAO inAppMessageDAO = getInAppMessageDao(activity);
        DfnInAppMessage inAppMessage = null;
        for(JSONObject eventParam : eventParamList){
            try {
                inAppMessage = inAppMessageDAO.getInAppMessageByEventNameWithParam(eventName, eventParam);
            }catch (Exception e){
                AbxLog.w(e, true);
            }
            if(CommonUtils.notNull(inAppMessage)){
                break;
            }
        }
        return inAppMessage;
    }

    public void setPause(){
        isPaused.set(true);
    }
    public void setResume(){
        isPaused.set(false);
    }
    private boolean isInAppMessageActive(){
        boolean result = false;
        if(CommonUtils.isNull(factory)){
            return result;
        }
        if(CommonUtils.isNull(factory.createOrGetDataRegistry())){
            return result;
        }
        if (factory.createOrGetDataRegistry().safeGetLong(DataRegistryKey.LONG_S3_CONFIG_IN_APP_MESSAGE_ACTIVE, -1) != 1) {
            return result;
        }
        result = true;
        return result;
    }

    private InAppMessageDAO getInAppMessageDao(Activity activity){
        InAppMessageDAO inAppMessageDAO = new InAppMessageDAO(activity.getApplicationContext(), factory.createOrGetDataRegistry());
        return inAppMessageDAO;
    }

    private boolean isPortrait(Activity activity){
        boolean result = true;
        try {
            result = activity.getResources().getConfiguration().orientation == Configuration.ORIENTATION_PORTRAIT;
        }catch (Exception e){
            AbxLog.w(e, true);
        }
        return result;
    }

    public static InAppMessageManager getInstance() {
        return InAppMessageManager.Singleton.INSTANCE;
    }
    private static class Singleton {
        private static final InAppMessageManager INSTANCE = new InAppMessageManager();
    }

    public class SuspendedUpdateImageInfo {
        String campaignId;
        String orientation;
        String imageUrl;
        String imagePath;
        String clickAction;

        public SuspendedUpdateImageInfo(String campaignId, String orientation, String imageUrl, String imagePath, String clickAction) {
            this.campaignId = campaignId;
            this.orientation = orientation;
            this.imageUrl = imageUrl;
            this.imagePath = imagePath;
            this.clickAction = clickAction;
        }
    }
}
