package io.adbrix.sdk.domain.model;

import android.graphics.Bitmap;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nullable;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.domain.IAMConstants;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.CoreUtils;

public class DfnInAppMessage {
    @Nullable public final String campaignId;
    @Nullable public final String type;
    @Nullable public final String position;
    @Nullable public final String identifierType;
    @Nullable public final String html;
    @Nullable public final String bgColor;
    @Nullable public final String overlayColor;
    public final String blackColor = "#000000";
    public final String whiteColor = "#FFFFFF";
    public final long startDateTime;
    public final long endDateTime;
    @Nullable public final JSONObject extAttr;
    @Nullable public final DfnGrowthEventParam growthEventParam;
    public final List<DfnIAMTrigger> triggers;

    public DfnInAppMessage(
            @Nullable String campaignId,
            @Nullable String type,
            @Nullable String position,
            String identifierType,
            String html,
            @Nullable String bgColor,
            @Nullable String overlayColor,
            long startDateTime,
            long endDateTime,
            @Nullable JSONObject extAttr,
            @Nullable DfnGrowthEventParam growthEventParam,
            List<DfnIAMTrigger> triggers
    ) {
        String nullString = "null";
        String defaultOverlayColor = "#EEEEEEEE";

        this.campaignId = campaignId;
        this.type = type;
        this.position = position;
        this.identifierType = identifierType;
        this.html = html;
        this.bgColor = bgColor == null || bgColor.equals(nullString) ? whiteColor : bgColor;
        this.overlayColor = overlayColor == null || overlayColor.equals(nullString) ? defaultOverlayColor : overlayColor;
        this.startDateTime = startDateTime;
        this.endDateTime = endDateTime;
        this.extAttr = extAttr;
        this.growthEventParam = growthEventParam;
        this.triggers = triggers;
    }

    public DfnInAppMessage(
            @Nullable String campaignId,
            @Nullable String type,
            @Nullable String position,
            String identifierType,
            String html,
            String bgColor,
            String overlayColor,
            long startDateTime,
            long endDateTime,
            String extAttrString,
            String extConfigString,
            List<DfnIAMTrigger> triggers
    ) {
        String nullString = "null";
        String defaultOverlayColor = "#EEEEEEEE";

        this.campaignId = campaignId;
        this.type = type;
        this.position = position;
        this.identifierType = identifierType;
        this.html = html;
        this.bgColor = nullString.equals(bgColor) ? whiteColor : bgColor;
        this.overlayColor = nullString.equals(overlayColor) ? defaultOverlayColor : overlayColor;

        JSONObject tempExtAttr;
        DfnGrowthEventParam tempDfnGrowthEventParam;

        try {
            if (extAttrString != null) {
                tempExtAttr = new JSONObject(extAttrString);
            } else {
                tempExtAttr = null;
            }

            if (extConfigString != null) {
                tempDfnGrowthEventParam = DfnGrowthEventParam.fromJSON(extConfigString);
            } else {
                tempDfnGrowthEventParam = null;
            }

        } catch (Exception e) {
            tempExtAttr = new JSONObject();
            tempDfnGrowthEventParam = null;
        }
        this.startDateTime = startDateTime;
        this.endDateTime = endDateTime;
        this.extAttr = tempExtAttr;
        this.growthEventParam = tempDfnGrowthEventParam;
        this.triggers = triggers;
    }

    public String getCampaignId() {
        return campaignId;
    }

    public String getType() {
        return type;
    }

    public String getBgColor() {
        return bgColor;
    }

    public String getOverlayColor() {
        return overlayColor;
    }

    @Nullable
    public String getHtml() {
        return html;
    }

    @Nullable
    public String getPosition() {
        return position;
    }

    @Nullable
    public String getIdentifierType() {
        return identifierType;
    }

    public static List<DfnInAppMessage> fromJSON(String responseJsonString) {
        List<DfnInAppMessage> inAppMessageList = new ArrayList<>();

        try {
            JSONObject response = new JSONObject(responseJsonString);
            JSONObject data = response.getJSONObject(IAMConstants.RESPONSE_DATA);
            JSONArray inAppMessages = data.getJSONArray(IAMConstants.RESPONSE_IAM);

            for (int i = 0; i < inAppMessages.length(); i++) {
                JSONObject inAppMessageJson = inAppMessages.getJSONObject(i);
                JSONObject contents = inAppMessageJson.optJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS);
                JSONObject bgStyle = contents == null ? null : contents.optJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_BG_STYLE);
                JSONObject extConfig = inAppMessageJson.optJSONObject(IAMConstants.RESPONSE_IAM_EXT_CONFIG);
                JSONArray triggersArray = inAppMessageJson.optJSONArray(IAMConstants.RESPONSE_IAM_TRIGGERS);
                List<DfnIAMTrigger> triggers = new ArrayList<>();

                for (int j = 0; triggersArray != null && j < triggersArray.length(); j++) {
                    JSONObject trigger = triggersArray.getJSONObject(j);

                    triggers.add(
                            new DfnIAMTrigger(
                                    trigger.optString(IAMConstants.RESPONSE_IAM_TRIGGERS_TYPE),
                                    trigger.optString(IAMConstants.RESPONSE_IAM_TRIGGERS_EVT),
                                    trigger.optInt(IAMConstants.RESPONSE_IAM_TRIGGERS_PRIORITY)
                            )
                    );
                }

                DfnInAppMessage inAppMessage = new DfnInAppMessage(
                        inAppMessageJson.optString(IAMConstants.RESPONSE_IAM_CAMPAIGN_ID),
                        inAppMessageJson.optString(IAMConstants.RESPONSE_IAM_TYPE),
                        inAppMessageJson.optString(IAMConstants.RESPONSE_IAM_POSITION),
                        inAppMessageJson.optString(IAMConstants.RESPONSE_IAM_IDENFITIER_TYPE),
                        inAppMessageJson.optString(IAMConstants.RESPONSE_IAM_HTML),
                        bgStyle == null ? null : bgStyle.optString(IAMConstants.RESPONSE_IAM_CONTENTS_BG_STYLE_BG_COLOR, "#FFFFFF"),
                        bgStyle == null ? null : bgStyle.optString(IAMConstants.RESPONSE_IAM_CONTENTS_BG_STYLE_OVERLAY_COLOR, "#EEEEEEEE"),
                        CoreUtils.getStartDateTimeLong(inAppMessageJson.optString(IAMConstants.RESPONSE_IAM_START_DATETIME)),
                        CoreUtils.getEndDateTimeLong(inAppMessageJson.optString(IAMConstants.RESPONSE_IAM_END_DATETIME)),
                        inAppMessageJson.optJSONObject(IAMConstants.RESPONSE_IAM_EXT_ATTR),
                        extConfig == null ? null : DfnGrowthEventParam.fromJSON(extConfig.toString()),
                        triggers
                );

                inAppMessageList.add(inAppMessage);
            }

            return inAppMessageList;
        } catch (Exception e) {
            AbxLog.e(e, true);
            return inAppMessageList;
        }
    }

    @Override
    public String toString() {
        return "DfnInAppMessage{" +
                "campaignId='" + campaignId + '\'' +
                ", type='" + type + '\'' +
                ", position='" + position + '\'' +
                ", identifierType='" + identifierType + '\'' +
                ", html='" + html + '\'' +
                ", bgColor='" + bgColor + '\'' +
                ", overlayColor='" + overlayColor + '\'' +
                ", blackColor='" + blackColor + '\'' +
                ", whiteColor='" + whiteColor + '\'' +
                ", startDateTime=" + startDateTime +
                ", endDateTime=" + endDateTime +
                ", extAttr=" + extAttr +
                ", growthEventParam=" + growthEventParam +
                ", triggers=" + triggers +
                '}';
    }
}
