package io.adbrix.sdk.data.repository;

import static io.adbrix.sdk.domain.CoreConstants.EVENT_ADID_CHANGED;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.os.Build;
import android.util.Pair;

import com.igaworks.v2.core.AdBrixRm;
import com.igaworks.v2.core.result.GetAttributionDataResult;
import com.igaworks.v2.core.result.GetSubscriptionStatusResult;
import com.igaworks.v2.core.result.SetCiProfileResult;
import com.igaworks.v2.core.result.SetSubscriptionStatusResult;
import com.pci.beacon.PCI;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;

import javax.annotation.Nullable;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.EventBuffer;
import io.adbrix.sdk.component.EventSamplingFilter;
import io.adbrix.sdk.component.IABXComponentsFactory;
import io.adbrix.sdk.component.UserPropertyManager;
import io.adbrix.sdk.data.DFNSessionState;
import io.adbrix.sdk.data.RemoteConfigProvider;
import io.adbrix.sdk.data.actionhistory.ActionHistoryDAO;
import io.adbrix.sdk.data.dataprovider.DeviceRealtimeDataProvider;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.modelprovider.ActionHistoryDeleteModelProvider;
import io.adbrix.sdk.data.modelprovider.ActionHistoryQueryModelProvider;
import io.adbrix.sdk.data.modelprovider.AttributionModelProvider;
import io.adbrix.sdk.data.modelprovider.DFNIDModelProvider;
import io.adbrix.sdk.data.modelprovider.DRModelProvider;
import io.adbrix.sdk.data.modelprovider.EventModelProvider;
import io.adbrix.sdk.data.modelprovider.GDPRModelProvider;
import io.adbrix.sdk.data.modelprovider.GetSubscriptionStatusModelProvider;
import io.adbrix.sdk.data.modelprovider.InAppMessageApiModelProvider;
import io.adbrix.sdk.data.modelprovider.SetCiPropertyModelProvider;
import io.adbrix.sdk.data.modelprovider.SetSubscriptionStatusModelProvider;
import io.adbrix.sdk.data.net.ApiConnectionManager;
import io.adbrix.sdk.data.net.IApiConnection;
import io.adbrix.sdk.domain.CompatConstants;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.Repository;
import io.adbrix.sdk.domain.function.Completion;
import io.adbrix.sdk.domain.model.ActionHistory;
import io.adbrix.sdk.domain.model.ActionHistoryDeleteModel;
import io.adbrix.sdk.domain.model.ActionHistoryError;
import io.adbrix.sdk.domain.model.ActionHistoryIdType;
import io.adbrix.sdk.domain.model.ActionHistoryQueryModel;
import io.adbrix.sdk.domain.model.DFNIDModel;
import io.adbrix.sdk.domain.model.DRModel;
import io.adbrix.sdk.domain.model.DRState;
import io.adbrix.sdk.domain.model.DfnInAppMessage;
import io.adbrix.sdk.domain.model.DfnInAppMessageFetchMode;
import io.adbrix.sdk.domain.model.Empty;
import io.adbrix.sdk.domain.model.Error;
import io.adbrix.sdk.domain.model.EventModel;
import io.adbrix.sdk.domain.model.IAMEnums;
import io.adbrix.sdk.domain.model.IApiModel;
import io.adbrix.sdk.domain.model.InAppMessageApiModel;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.domain.model.Response;
import io.adbrix.sdk.domain.model.Result;
import io.adbrix.sdk.domain.model.ResultCallback;
import io.adbrix.sdk.domain.model.SubscriptionStatus;
import io.adbrix.sdk.domain.model.Success;
import io.adbrix.sdk.domain.model.UserPropertyCommand;
import io.adbrix.sdk.domain.model.UserPropertyModel;
import io.adbrix.sdk.ui.inappmessage.InAppMessageDAO;
import io.adbrix.sdk.ui.inappmessage.InAppMessageManager;
import io.adbrix.sdk.utils.Base64;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.CoreUtils;

public class RepositoryImpl implements Repository {
    private IABXComponentsFactory factory;
    private DataRegistry dataRegistry;
    private Context androidContext;
    private InAppMessageDAO inAppMessageDAO;
    private DeviceRealtimeDataProvider deviceRealtimeDataProvider;
    private ActionHistoryDAO actionHistoryDAO;
    private ExecutorService actionHistoryExecutor;
    private DFNSessionState dfnSessionState;
    private RemoteConfigProvider remoteConfigProvider;
    private UserPropertyManager userPropertyManager;
    private EventSamplingFilter eventSamplingFilter;
    private EventBuffer eventBuffer;

    public RepositoryImpl(IABXComponentsFactory factory) {
        this.factory = factory;

        try {
            this.dataRegistry = factory.createOrGetDataRegistry();
            this.androidContext = factory.getAndroidContext();

            this.inAppMessageDAO = new InAppMessageDAO(androidContext, dataRegistry);
            this.deviceRealtimeDataProvider = factory.createOrGetDeviceRealtimeDataProvider();
            this.actionHistoryDAO = new ActionHistoryDAO(androidContext, dataRegistry);
            this.actionHistoryExecutor = Executors.newSingleThreadExecutor();
            this.dfnSessionState = factory.createOrGetDFNSessionState();
            this.remoteConfigProvider = factory.createOrGetRemoteConfigProvider();
            this.userPropertyManager = factory.createOrGetUserPropertyManager();
            this.eventSamplingFilter = factory.createOrGetEventSamplingFilter();
            this.eventBuffer = factory.createOrGetEventBuffer();
            InAppMessageManager.getInstance().init(factory);
        } catch (Exception e) {
            AbxLog.e(e, true);
        }
    }

    @Override
    public EventModel getEventModel(LogEventParameter logEventParameter) {
        AtomicReference<EventModel> eventModel = new AtomicReference<>();
        EventModelProvider eventModelProvider = new EventModelProvider(
                this.userPropertyManager,
                this.dataRegistry);

        eventModelProvider.setLogEventParameter(logEventParameter);
        eventModel.getAndSet(eventModelProvider.provide());

        return eventModel.get();
    }

    @Override
    public Boolean saveUserPropertyWithoutEvent(UserPropertyCommand userPropertyCommand) {
        AtomicBoolean result = new AtomicBoolean(false);
        result.getAndSet(this.userPropertyManager.merge(userPropertyCommand));
        return result.get();
    }

    @Override
    public void clearUserProperty() {
        this.userPropertyManager.clear();
    }

    @Override
    public void getUserId(Completion<Result<String>> completion) {
        UserPropertyModel userPropertyModel = factory.getAbxContextController().getCurrentUserPropertyModel();
        String userId = "";
        if(CommonUtils.isNullOrEmpty(userPropertyModel.properties)){
            AbxLog.d("userProperty is empty or null", false);
            completion.handle(Success.of(userId));
            return;
        }

        if(!userPropertyModel.properties.containsKey("abx:user_id") && !userPropertyModel.properties.containsKey("user_id")){
            AbxLog.d("userProperty doesn't has user_id", false);
            completion.handle(Success.of(userId));
            return ;
        }
        userId = (String) userPropertyModel.properties.get("abx:user_id");
        if(!CommonUtils.isNullOrEmpty(userId)){
            userId = userId.substring(7);
        }
        else{
            userId = (String) userPropertyModel.properties.get("user_id");
            if(!CommonUtils.isNullOrEmpty(userId)){
                userId = userId.substring(7);
            }
            else {
                AbxLog.d("user_id is null or empty", false);
                completion.handle(Success.of(userId));
                return;
            }
        }
        completion.handle(Success.of(userId));
    }

    @Override
    public void logSameEventWithPaging(String eventName, String group, List<JSONObject> eventParams) {
        if (this.remoteConfigProvider.isBlockedEvent(eventName, group)) {
            return;
        }

        String prefixedEventName = getInAppMessageEventName(group, eventName);
        DfnInAppMessage dfnInAppMessage = InAppMessageManager.getInstance().getInAppMessageByEventNameWithParamList(prefixedEventName, eventParams);
        if(CommonUtils.notNull(dfnInAppMessage)){
            AdBrixRm.openInAppMessage(dfnInAppMessage.getCampaignId(), null);
        }

        boolean isEventBlockedBySamplingFilter = this.eventSamplingFilter.isEventBlockedBySamplingFilter(group+":" + eventName);;
        if (isEventBlockedBySamplingFilter)
            return;

        long currentTimeMillis = System.currentTimeMillis();
        String eventId = CommonUtils.randomUUIDWithCurrentTime(currentTimeMillis);
        String eventDateTime = CommonUtils.getCurrentUTCInDBFormat(currentTimeMillis);

        String prevId;
        long sessionOrderNo = 0L;
        long eventOrderNo = 0L;
        prevId = dataRegistry.safeGetString(DataRegistryKey.STRING_PREV_ID, null);
        sessionOrderNo = dataRegistry.incrementAndGetLong(DataRegistryKey.LONG_SESSION_ORDER_NO);
        eventOrderNo = dataRegistry.incrementAndGetLong(DataRegistryKey.LONG_EVENT_ORDER_NO);

        int count = 1;
        List<EventModel> eventModelList = new ArrayList<>();
        for (JSONObject eventParamJson : eventParams) {
            try {
                JSONObject temp = new JSONObject();
                temp.put(CoreConstants.PAGE_ABX_KEY, count);
                temp = CommonUtils.parseValueWithDataType(temp, CommonUtils.FixType.PREFIX);

                eventParamJson.put(CoreConstants.PAGE_ABX_KEY, temp.getString(CoreConstants.PAGE_ABX_KEY));
            } catch (JSONException e) {
                AbxLog.e(e, true);
            }
            LogEventParameter eventParameter = new LogEventParameter(
                    eventId,
                    prevId,
                    group,
                    eventName,
                    CommonUtils.getMapFromJSONObject(eventParamJson),
                    0,
                    0,
                    sessionOrderNo,
                    eventOrderNo,
                    eventDateTime
            );

            final EventModel eventModel = getEventModel(eventParameter);

            if (eventModel == null) {
                AbxLog.e("repositoryImpl::logEvent error!! eventModel is null", true);
                return;
            }
            eventModelList.add(eventModel);
            count++;
        }
        this.eventBuffer.addEventModels(eventModelList);

        // prev_id를 저장해준다.
        dataRegistry.putDataRegistry(
                new DataUnit(
                        DataRegistryKey.STRING_PREV_ID,
                        eventId,
                        5,
                        this.getClass().getName(),
                        true
                )
        );
    }

    @Override
    public void logEvent(LogEventParameter logEventParameter) {
        if (remoteConfigProvider.isBlockedEvent(logEventParameter.eventName, logEventParameter.group)) {
            return;
        }
        JSONObject eventParam = CommonUtils.getJSONObjectFromMap(logEventParameter.eventParam);
        String prefixedEventName = getInAppMessageEventName(logEventParameter.group, logEventParameter.eventName);
        DfnInAppMessage dfnInAppMessage = InAppMessageManager.getInstance().getInAppMessageByEventNameWithParam(prefixedEventName, eventParam);
        if(CommonUtils.notNull(dfnInAppMessage)){
            AdBrixRm.openInAppMessage(dfnInAppMessage.getCampaignId(), null);
        }

        boolean isEventBlockedBySamplingFilter = this.eventSamplingFilter.isEventBlockedBySamplingFilter(logEventParameter);
        if (isEventBlockedBySamplingFilter)
            return;

        final EventModel eventModel = getEventModel(logEventParameter);

        if (eventModel == null) {
            AbxLog.e("repositoryImpl::logEvent error!! eventModel is null", true);
            return;
        }

        this.eventBuffer.addEventModel(eventModel);
        if(!EVENT_ADID_CHANGED.equals(eventModel.eventName)){
            // prev_id를 저장해준다.
            dataRegistry.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.STRING_PREV_ID,
                            logEventParameter.eventId,
                            5,
                            this.getClass().getName(),
                            true
                    )
            );
        }

        boolean isEndSessionNotSentInBackgroundWhenOffline = !CoreUtils.isOnline(this.androidContext)
                && !this.dfnSessionState.inForeground()
                && logEventParameter.eventName.equals(CoreConstants.EVENT_END_SESSION);

        if (isEndSessionNotSentInBackgroundWhenOffline)
            saveUnsentEvents();
    }

    @Override
    public void login(String userId, Completion<Result<Response>> completion) {
        LogEventParameter logEventParameter = new LogEventParameter(
                CoreConstants.GROUP_ABX,
                CompatConstants.EVENT_LOGIN,
                null,
                0,
                0
        );
        if (remoteConfigProvider.isBlockedEvent(logEventParameter.eventName, logEventParameter.group)) {
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("event is blocked"));
            }
            return;
        }

        Map<String, Object> loginEventParam = new HashMap<>();
        boolean isLoginIdMatched = isLoginIdMatched(userId);
        if(isLoginIdExist()){
            if(!isLoginIdMatched){
                clearUserProperty();
            }
        }
        loginEventParam.put(CompatConstants.IS_USER_ID_CHANGED, CommonUtils.getParseValueWithDataType(!isLoginIdMatched, CommonUtils.FixType.PREFIX));
        logEventParameter.eventParam = loginEventParam;
        AdBrixRm.UserProperties userProperties = new AdBrixRm.UserProperties();
        userProperties.setAttrs("abx:user_id", userId);
        saveUserPropertyWithoutEvent(makeUserPropertyCommand(userProperties.propertiesJson));

        String prefixedEventName = getInAppMessageEventName(CoreConstants.GROUP_ABX, CompatConstants.EVENT_LOGIN);
        DfnInAppMessage dfnInAppMessage = InAppMessageManager.getInstance().getInAppMessageByEventName(prefixedEventName);
        if(!isLoginIdMatched){
            fetchInAppMessage(true, new Completion<Result<Empty>>() {
                @Override
                public void handle(Result<Empty> emptyResult) {
                    AbxLog.i("login->fetchInAppMessage: "+emptyResult.toString(), true);
                    //fetch 후 login event trigger
                    DfnInAppMessage dfnInAppMessage = InAppMessageManager.getInstance().getInAppMessageByEventName(prefixedEventName);
                    if(CommonUtils.notNull(dfnInAppMessage)){
                        AdBrixRm.openInAppMessage(dfnInAppMessage.getCampaignId(), null);
                    }
                }
            });
        } else{
            if(CommonUtils.notNull(dfnInAppMessage)){
                AdBrixRm.openInAppMessage(dfnInAppMessage.getCampaignId(), null);
            }
        }

        boolean isEventBlockedBySamplingFilter = this.eventSamplingFilter.isEventBlockedBySamplingFilter(logEventParameter);
        if (isEventBlockedBySamplingFilter){
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("event is blocked"));
            }
            return;
        }

        final EventModel eventModel = getEventModel(logEventParameter);
        if (eventModel == null) {
            AbxLog.e("repositoryImpl::logEvent error!! eventModel is null", true);
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("repositoryImpl::logEvent error!! eventModel is null"));
            }
            return;
        }
        EventBuffer eventBuffer = factory.createOrGetEventBuffer();
        if(CommonUtils.isNull(eventBuffer)){
            AbxLog.e("eventBuffer is null", true);
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("eventBuffer is null"));
            }
            return;
        }
        eventBuffer.addEventModel(eventModel, completion);
        // prev_id를 저장해준다.
        dataRegistry.putDataRegistry(
                new DataUnit(
                        DataRegistryKey.STRING_PREV_ID,
                        logEventParameter.eventId,
                        5,
                        this.getClass().getName(),
                        true
                )
        );
        Context context = factory.getAndroidContext();
        if(CommonUtils.isNull(context)){
            AbxLog.e("context is null", true);
            return;
        }
        boolean isEndSessionNotSentInBackgroundWhenOffline = !CoreUtils.isOnline(context) && !this.dfnSessionState.inForeground()
                && logEventParameter.eventName.equals(CoreConstants.EVENT_END_SESSION);

        if (isEndSessionNotSentInBackgroundWhenOffline){
            saveUnsentEvents();
        }
        if(!isLoginIdMatched){
            //postUserPropertyChangedEvent
            String prefixedUserPropertyChangedEventName = getInAppMessageEventName(CoreConstants.GROUP_ABX, CompatConstants.EVENT_USER_PROPERTY_CHANGED);
            DfnInAppMessage dfnInAppMessageUserPropertyChanged = InAppMessageManager.getInstance().getInAppMessageByEventName(prefixedUserPropertyChangedEventName);
            if(CommonUtils.notNull(dfnInAppMessageUserPropertyChanged)){
                AdBrixRm.openInAppMessage(dfnInAppMessageUserPropertyChanged.getCampaignId(), null);
            }
            LogEventParameter eventParameter = new LogEventParameter(
                    CoreConstants.GROUP_ABX,
                    CompatConstants.EVENT_USER_PROPERTY_CHANGED,
                    null,
                    0,
                    0
            );
            logEvent(eventParameter);
        }
    }

    @Override
    public void logout(Completion<Result<Response>> completion) {
        LogEventParameter logEventParameter = new LogEventParameter(
                CoreConstants.GROUP_ABX,
                CompatConstants.EVENT_LOGOUT,
                null,
                0,
                0
        );
        if (remoteConfigProvider.isBlockedEvent(logEventParameter.eventName, logEventParameter.group)) {
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("event is blocked"));
            }
            return;
        }

        String loginId = getLoginId();
        boolean isLoginIdExist = true;
        if(CommonUtils.isNullOrEmpty(loginId)){
            isLoginIdExist = false;
        }
        JSONObject param = new JSONObject();
        try {
            param.put(CompatConstants.IS_USER_ID_CHANGED, isLoginIdExist);
            param.put(CompatConstants.PREV_USER_ID, loginId);
        } catch (JSONException e) {
            AbxLog.e("parsing error: ", e, true);
        }
        param = CommonUtils.parseValueWithDataType(param, CommonUtils.FixType.PREFIX);
        logEventParameter.eventParam = CommonUtils.getMapFromJSONObject(param);
        clearUserProperty();
        clearAllActionHistoryInLocalDB();
        deleteAllUserInAppMessageDBcontents();
        String prefixedEventName = getInAppMessageEventName(CoreConstants.GROUP_ABX, CompatConstants.EVENT_LOGOUT);
        DfnInAppMessage dfnInAppMessage = InAppMessageManager.getInstance().getInAppMessageByEventName(prefixedEventName);
        if(CommonUtils.notNull(dfnInAppMessage)){
            AdBrixRm.openInAppMessage(dfnInAppMessage.getCampaignId(), null);
        }
        boolean isEventBlockedBySamplingFilter = this.eventSamplingFilter.isEventBlockedBySamplingFilter(logEventParameter);
        if (isEventBlockedBySamplingFilter){
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("event is blocked"));
            }
            return;
        }
        final EventModel eventModel = getEventModel(logEventParameter);
        if (eventModel == null) {
            AbxLog.e("repositoryImpl::logEvent error!! eventModel is null", true);
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("repositoryImpl::logEvent error!! eventModel is null"));
            }
            return;
        }

        EventBuffer eventBuffer = factory.createOrGetEventBuffer();
        if(CommonUtils.isNull(eventBuffer)){
            AbxLog.e("eventBuffer is null", true);
            if(CommonUtils.notNull(completion)){
                completion.handle(Error.of("eventBuffer is null"));
            }
            return;
        }
        eventBuffer.addEventModel(eventModel, completion);
        // prev_id를 저장해준다.
        dataRegistry.putDataRegistry(
                new DataUnit(
                        DataRegistryKey.STRING_PREV_ID,
                        logEventParameter.eventId,
                        5,
                        this.getClass().getName(),
                        true
                )
        );
        Context context = factory.getAndroidContext();
        if(CommonUtils.isNull(context)){
            AbxLog.e("context is null", true);
            return;
        }
        boolean isEndSessionNotSentInBackgroundWhenOffline = !CoreUtils.isOnline(context) && !this.dfnSessionState.inForeground()
                && logEventParameter.eventName.equals(CoreConstants.EVENT_END_SESSION);

        if (isEndSessionNotSentInBackgroundWhenOffline)
            saveUnsentEvents();
        clearUserProperty();
    }

    @Override
    public void requestGetAttributionData(String logId, AdBrixRm.GetAttributionDataCallback callback, DataRegistry dataRegistry) {
        ApiConnectionManager<Void> apiConnectionManager = new ApiConnectionManager<>(new ApiConnectionManager.Result<Void>() {
            @Override
            public void connectSuccess(String responseString, int responseCode, Void unused) {
                if(CommonUtils.notNull(callback)){
                    callback.onCallback(new GetAttributionDataResult(responseCode, responseString));
                }
            }

            @Override
            public void connectFail(int responseCode, Void unused) {
                if(CommonUtils.notNull(callback)){
                    callback.onCallback(new GetAttributionDataResult(responseCode, GetAttributionDataResult.Result.UNKNOWN_ERROR.getValue()));
                }
            }
        }, null);

        IApiModel apiModel = new AttributionModelProvider(logId, dataRegistry).provide();
        IApiConnection apiConnection = factory.createOrGetAPIConnection();
        apiConnection.post(apiModel);
        apiConnectionManager.execute(apiConnection);
    }

    @Override
    public void getSubscriptionStatus(AdBrixRm.GetSubscriptionStatusCallback callback, DataRegistry dataRegistry) {
        ApiConnectionManager<Void> apiConnectionManager = new ApiConnectionManager<>(new ApiConnectionManager.Result<Void>() {
            @Override
            public void connectSuccess(String responseString, int responseCode, Void unused) {
                if(CommonUtils.notNull(callback)){
                    callback.onCallback(new GetSubscriptionStatusResult(responseCode, responseString));
                }
            }

            @Override
            public void connectFail(int responseCode, Void unused) {
                if(CommonUtils.notNull(callback)){
                    callback.onCallback(new GetSubscriptionStatusResult(responseCode, GetSubscriptionStatusResult.RESULT_CODE_UNKNOWN_ERROR));
                }
            }
        }, null);

        IApiModel apiModel = new GetSubscriptionStatusModelProvider(dataRegistry).provide();
        IApiConnection apiConnection = factory.createOrGetAPIConnection();
        apiConnection.post(apiModel);
        apiConnectionManager.execute(apiConnection);
    }

    @Override
    public void setSubscriptionStatus(SubscriptionStatus subscriptionStatus, AdBrixRm.SetSubscriptionStatusCallback callback, DataRegistry dataRegistry) {
        ApiConnectionManager<Void> apiConnectionManager = new ApiConnectionManager<>(new ApiConnectionManager.Result<Void>() {
            @Override
            public void connectSuccess(String responseString, int responseCode, Void unused) {
                if(CommonUtils.notNull(callback)){
                    callback.onCallback(new SetSubscriptionStatusResult(responseCode, responseString));
                }
            }

            @Override
            public void connectFail(int responseCode, Void unused) {
                if(CommonUtils.notNull(callback)){
                    callback.onCallback(new SetSubscriptionStatusResult(responseCode, SetSubscriptionStatusResult.RESULT_CODE_UNKNOWN_ERROR));
                }
            }
        }, null);

        IApiModel apiModel = new SetSubscriptionStatusModelProvider(subscriptionStatus, dataRegistry).provide();
        IApiConnection apiConnection = factory.createOrGetAPIConnection();
        apiConnection.post(apiModel);
        apiConnectionManager.execute(apiConnection);
    }

    @Override
    public void setCiProperty(String key, String value, AdBrixRm.SetCiProfileCallback callback, DataRegistry dataRegistry) {
        ApiConnectionManager<Void> apiConnectionManager = new ApiConnectionManager<>(new ApiConnectionManager.Result<Void>() {
            @Override
            public void connectSuccess(String responseString, int responseCode, Void unused) {
                if(CommonUtils.notNull(callback)){
                    callback.onCallback(new SetCiProfileResult(responseCode, responseString));
                }
            }

            @Override
            public void connectFail(int responseCode, Void unused) {
                if(CommonUtils.notNull(callback)){
                    callback.onCallback(new SetCiProfileResult(responseCode, SetCiProfileResult.RESULT_CODE_UNKNOWN_ERROR));
                }
            }
        }, null);

        IApiModel apiModel = new SetCiPropertyModelProvider(key, value, dataRegistry).provide();
        IApiConnection apiConnection = factory.createOrGetAPIConnection();
        apiConnection.post(apiModel);
        apiConnectionManager.execute(apiConnection);
    }

    private String getInAppMessageEventName(String eventGroup, String eventName){
        StringBuilder stringBuilder = new StringBuilder(eventGroup.length()+eventName.length()+1);
        stringBuilder.append(eventGroup);
        stringBuilder.append(":");
        stringBuilder.append(eventName);
        return stringBuilder.toString();
    }

    @Override
    public String getLoginId(){
        String result = "";
        UserPropertyModel userPropertyModel = factory.createOrGetUserPropertyManager().getCurrentUserPropertyModel();
        if(userPropertyModel == null){
            return result;
        }
        if(CommonUtils.isNullOrEmpty(userPropertyModel.properties)){
            return result;
        }
        if(!userPropertyModel.properties.containsKey("abx:user_id") && !userPropertyModel.properties.containsKey("user_id")){
            return result;
        }
        String userId = (String) userPropertyModel.properties.get("abx:user_id");
        if(!CommonUtils.isNullOrEmpty(userId)){
            userId = userId.substring(7);
        }
        else{
            userId = (String) userPropertyModel.properties.get("user_id");
            if(!CommonUtils.isNullOrEmpty(userId)){
                userId = userId.substring(7);
            }
        }
        return userId;
    }

    public boolean isLoginIdMatched(String loginId) {
        boolean result = false;
        UserPropertyModel userPropertyModel = factory.getAbxContextController().getCurrentUserPropertyModel();
        if(userPropertyModel == null){
            return result;
        }
        if(CommonUtils.isNullOrEmpty(userPropertyModel.properties)){
            return result;
        }
        if(!userPropertyModel.properties.containsKey("abx:user_id") && !userPropertyModel.properties.containsKey("user_id")){
            return result;
        }
        loginId = "string:" + loginId;
        String userId = (String) userPropertyModel.properties.get("abx:user_id");
        if(loginId.equals(userId)){
            AbxLog.i("userId is matched. input: "+loginId, true);
            return true;
        }
        else{
            userId = (String) userPropertyModel.properties.get("user_id");
            if(loginId.equals(userId)){
                AbxLog.i("userId is matched. input: "+loginId, true);
                return true;
            }
        }
        AbxLog.i("userId is not matched. userId: "+userId, true);
        return result;
    }
    public boolean isLoginIdExist(){
        String loginId = getLoginId();
        if(CommonUtils.isNullOrEmpty(loginId)){
            return false;
        } else{
            return true;
        }
    }

    /*@Override
    public boolean isInAppMessageUserFetchMode() {
        int fetchMode = this.dataRegistry.safeGetInt(DataRegistryKey.INT_IN_APP_MESSAGE_FETCH_MODE_VALUE, -1);
        if(DfnInAppMessageFetchMode.USER_ID.equals(DfnInAppMessageFetchMode.fromInteger(fetchMode))){
            return true;
        } else{
            return false;
        }
    }*/

    private UserPropertyCommand makeUserPropertyCommand(JSONObject userPropertiesJson) {
        UserPropertyCommand command = new UserPropertyCommand();
        UserPropertyModel userPropertyModel = factory.getAbxContextController().getCurrentUserPropertyModel();

        JSONObject truncatedUserPropertiesJson = CommonUtils.truncate(userPropertiesJson, factory.createOrGetRemoteConfigProvider().getConfigPropertyMaxSize());
        JSONObject parsedUserPropertiesJson = CommonUtils.parseValueWithDataType(truncatedUserPropertiesJson, CommonUtils.FixType.PREFIX);

        Iterator<?> keys = parsedUserPropertiesJson.keys();
        int currentSizeOfUserPropertyModel = userPropertyModel.properties.size();

        while (keys.hasNext()) {
            String key = (String) keys.next();

            try {
                if (currentSizeOfUserPropertyModel < this.factory.createOrGetRemoteConfigProvider().getConfigPropertyMaxSize()) {
                    command.set(key, parsedUserPropertiesJson.get(key));
                    currentSizeOfUserPropertyModel++;
                } else {
                    AbxLog.d("UserProperties reaches MAX_PROPERTY_KEYS: "
                            + this.factory.createOrGetRemoteConfigProvider().getConfigPropertyMaxSize(), true);
                    break;
                }
            } catch (JSONException e) {
                AbxLog.e("updateLocalUserProperties Error: ", e, true);
            }
        }

        return command;
    }

    @Override
    public void flushAllNow(Completion<Result<Empty>> completion) {
        this.eventBuffer.flushAllNow(completion);
    }

    @Override
    public void flushAtIntervals() {
        this.eventBuffer.flushContainerAtIntervals();
    }

    @Override
    public void saveUnsentEvents() {
        this.eventBuffer.saveUnsentEvents();
    }

    @Override
    public void gdprForgetMe() {
        try {
            ApiConnectionManager<DataRegistry> apiConnectionManager = new ApiConnectionManager<>(new ApiConnectionManager.Result<DataRegistry>() {
                @Override
                public void connectSuccess(String responseString, int responseCode, DataRegistry dataRegistry) {
                    AbxLog.d("GDPR connect success!!", true);
                    dataRegistry.putDataRegistry(
                            new DataUnit(
                                    DataRegistryKey.LONG_GDPR_FORGETME,
                                    1L,
                                    5,
                                    this.getClass().getName(),
                                    true
                            )
                    );

                    dataRegistry.putDataRegistry(
                            new DataUnit(
                                    DataRegistryKey.LONG_GDPR_FORGETME_SERVER_SYNC,
                                    1L,
                                    5,
                                    this.getClass().getName(),
                                    true
                            )
                    );
                }

                @Override
                public void connectFail(int responseCode, DataRegistry dataRegistry) {
                    AbxLog.d("GDPR connect failed!! responseCode : " + responseCode, true);
                    dataRegistry.putDataRegistry(
                            new DataUnit(
                                    DataRegistryKey.LONG_GDPR_FORGETME,
                                    1L,
                                    5,
                                    this.getClass().getName(),
                                    true
                            )
                    );

                    dataRegistry.putDataRegistry(
                            new DataUnit(
                                    DataRegistryKey.LONG_GDPR_FORGETME_SERVER_SYNC,
                                    0L,
                                    5,
                                    this.getClass().getName(),
                                    true
                            )
                    );
                }
            }, dataRegistry);

            IApiModel gdprModel = new GDPRModelProvider(dataRegistry).provide();

            IApiConnection apiConnection = factory.createOrGetAPIConnection().post(gdprModel);

            apiConnectionManager.executeWithRetry(apiConnection);
        } catch (Exception e) {
            AbxLog.e("GDPR Request Error: ",e, true);
        }
    }

//    @Override
//    public void updateFavoriteApplication() {
//        Timer timer = new Timer();
//
//        timer.schedule(new TimerTask() {
//            @Override
//            public void run() {
//                try {
//                    if (ABXBooleanState.getInstance().isAdbrixAllStop() ||
//                            ABXBooleanState.getInstance().isAdbrixError()){
//                        return;
//                    }
//
//                    AbxLog.i("Application Scanning... per " + ABXConstants.BATCHING_INTERVAL / 1000 + " Seconds", false);
//
//                    AbxAppScanHandler appScanHandler = new AbxAppScanHandler(dataRegistry);
//                    if (!ABXBooleanState.getInstance().isAdbrixPause()) {
//                        if (appScanHandler.getAppScanStatus()) {
//                            if (appScanHandler.checkIsTimeToAppScan())
//                                appScanHandler.abxApplicationScan(androidContext);
//                        } else {
//                            boolean userFlag = dataRegistry.safeGetLong(DataRegistryKey.LONG_APP_SCAN_ON_OFF_USER, -1) == 1;
//                            AbxLog.d("Application Scanning Failed : Server Flag = " + !S3ConfigHandler.config_appScanStop, true);
//                            AbxLog.d("Application Scanning Failed : User flag = " + userFlag, true);
//                        }
//                    }
//                } catch (Exception e){
//                    AbxLog.e(e, true);
//                    Log.e(ABXConstants.LOGTAG, e.getMessage());
//                    timer.cancel();
//                }
//            }
//        }, 60_000, 60_000);
//    }

    @SuppressLint("MissingPermission")
    @Override
    public void resetInAppMessageFrequencySession() {
        if (inAppMessageDAO == null) {
            AbxLog.w("inAppMessageDAO is null", false);
            return;
        }
        try {
            inAppMessageDAO.resetFrequencyCapPerSession();
        }catch (Exception e){
            AbxLog.w(e, true);
        }
    }

    @Override
    public void openInAppMessage(String eventName) {
        try {
            InAppMessageManager.getInstance().showByEventName(eventName);
        } catch (Exception e) {
            AbxLog.e("openInAppMessage: ",e, true);
        }
    }

    @Override
    public void openInAppMessage(String eventName, JSONObject eventParam) {
        try {
            InAppMessageManager.getInstance().showByEventNameWithParam(eventName, eventParam);
        } catch (Exception e) {
            AbxLog.e("openInAppMessage: ",e, true);
        }
    }

    @Override
    public void openInAppMessage(String eventName, List<JSONObject> eventParamList) {
        try {
            InAppMessageManager.getInstance().showByEventNameWithParamList(eventName, eventParamList);
        } catch (Exception e) {
            AbxLog.e("openInAppMessage: ",e, true);
        }
    }

    @Override
    public void openInAppMessage(String campaignId, Completion<Result<Empty>> completion) {
        try {
            InAppMessageManager.getInstance().showByCampaignId(campaignId, completion);
        } catch (Exception e) {
            AbxLog.e("openInAppMessage() ",e, true);
            completion.handle(Error.of(e));
        }
    }

    @Override
    public void dismissInAppMessageDialog(Activity destroyedActivity, IAMEnums.CloseType closeType) {
        InAppMessageManager.getInstance().close(closeType);
    }

    @Override
    public void shutDownInAppMessageExecutor() {
        inAppMessageDAO.closeDataBase();
    }

    @Override
    public void updateCurrentActivity(Activity activity) {

    }

    @Override
    public void deleteCurrentActivity(Activity activity) {

    }

    @Override
    public void fetchInAppMessage(boolean isDirectCall, Completion<Result<Empty>> completion) {
        if (dataRegistry.safeGetLong(DataRegistryKey.LONG_S3_CONFIG_IN_APP_MESSAGE_ACTIVE, -1) != 1) {
            completion.handle(Error.of("InAppMessage is not active!"));
            return;
        }

        if(!isDirectCall){
            int minutesToExpiry = dataRegistry.safeGetInt(DataRegistryKey.INT_IN_APP_MESSAGE_MINUTES_TO_EXPIRY, -1);
            long inAppMessageLastResponseTime = dataRegistry.safeGetLong(DataRegistryKey.LONG_IN_APP_MESSAGE_LAST_RESPONSE_TIME, -1);

            if (inAppMessageLastResponseTime + minutesToExpiry * 60 * 1000 >= System.currentTimeMillis()) {
                AbxLog.d("inAppMessage Data is already cached", true);
                completion.handle(Success.empty());
                return;
            }
        }

        AbxLog.d("inAppMessageApiConnection start ::: ", true);
        try {
            ApiConnectionManager<InAppMessageDAO> apiConnectionManager = new ApiConnectionManager<>(new ApiConnectionManager.Result<InAppMessageDAO>() {
                @Override
                public void connectSuccess(String responseString, int responseCode, InAppMessageDAO inAppMessageDAO) {
                    AbxLog.d("Terminated inAppMessageApiConnection", true);
                    AbxLog.d("inAppMessageApiConnection connectSuccess !! ", true);
                    if(completion == null){
                        AbxLog.d("completion is null", true);
                        return;
                    }

                    try {
                        inAppMessageDAO.insertApiResponse2Table(responseString, getLoginId());
                        completion.handle(Success.empty());
                    } catch (Exception e) {
                        AbxLog.d("inserApiResponse2Table jsonerror : " + e, true);
                        completion.handle(Error.of(e));
                    }
                }

                @Override
                public void connectFail(int responseCode, InAppMessageDAO inAppMessageDAO) {
                    AbxLog.d("Terminated inAppMessageApiConnection", true);
                    AbxLog.d("inAppMessageApiConnection connectFail !! ", true);
                    if(isFetchInAppMessageExpired()){
                        AbxLog.d("fetchInAppMessage is expired. All inAppMessage contents is deleted.", true);
                        inAppMessageDAO.deleteAllInAppMessageDBContents();
                        if(completion == null){
                            AbxLog.d("completion is null", true);
                            return;
                        }
                        completion.handle(Error.of("inAppMessageApiConnection connectFail !! fetchInAppMessage is expired. All inAppMessage contents is deleted."));
                    }
                    if(completion == null){
                        AbxLog.d("completion is null", true);
                        return;
                    }
                    completion.handle(Error.of("inAppMessageApiConnection connectFail !!"));
                }
            }, inAppMessageDAO);

            InAppMessageApiModel inAppMessageApiModel = new InAppMessageApiModelProvider(
                    dataRegistry,
                    androidContext,
                    deviceRealtimeDataProvider
            ).provide();

            String token = dataRegistry.safeGetString(DataRegistryKey.STRING_IN_APP_MESSAGE_TOKEN, null);

            Map<String, String> header = new HashMap<>();
            if (token != null)
                header.put("Authorization", token);

            IApiConnection apiConnection = factory.createOrGetAPIConnection()
                    .header(header)
                    .post(inAppMessageApiModel);

            apiConnectionManager.executeWithRetry(apiConnection);

        } catch (Exception e) {
            AbxLog.e("inAppMessageApiConnection Request Error: ",e, true);
            completion.handle(Error.of(e));
        }
    }

    private boolean isFetchInAppMessageExpired(){
        boolean result = false;
        long inAppMessageLastResponseTime = dataRegistry.safeGetLong(DataRegistryKey.LONG_IN_APP_MESSAGE_LAST_RESPONSE_TIME, -1);
        if (inAppMessageLastResponseTime == -1){
            return result;
        }
        Date requestDate = new Date(inAppMessageLastResponseTime);
        Calendar requestDateAfter10Days = Calendar.getInstance();
        requestDateAfter10Days.setTime(requestDate);
        requestDateAfter10Days.add(Calendar.DATE, 10);
        //요청한지 10일 초과로 경과 되었을 경우 Expire
        Date currentDate = new Date();
        if(currentDate.after(requestDateAfter10Days.getTime())){
            result = true;
        }
        return result;
    }

    @Override
    public void getAllInAppMessage(Completion<Result<List<DfnInAppMessage>>> completion) {
        if (inAppMessageDAO == null) {
            completion.handle(Error.of("InAppMessageDAO is null!"));
            return;
        }

        completion.handle(inAppMessageDAO.getAllInAppMessage());
    }

    @Override
    public void deleteAllInAppMessageDBContents() {
        if (inAppMessageDAO == null) {
            AbxLog.w("inAppMessageDAO is null", false);
            return;
        }

        inAppMessageDAO.deleteAllInAppMessageDBContents();
    }

    @Override
    public void deleteAllUserInAppMessageDBcontents() {
        if (inAppMessageDAO == null) {
            AbxLog.w("inAppMessageDAO is null", false);
            return;
        }

        inAppMessageDAO.deleteAllUserInAppMessageDBContents();
    }

    @Override
    public void deleteUserData(String userId, ResultCallback<String> callback) {
        try {
            DRModel deleteUserDataModel = new DRModelProvider(factory,
                    userId, DRModel.OperationType.DELETE)
                    .provide();

            IApiConnection apiConnection = factory.createOrGetAPIConnection()
                    .post(deleteUserDataModel);
            ApiConnectionManager<Void> apiConnectionManager = new ApiConnectionManager<>(new ApiConnectionManager.Result<Void>() {
                @Override
                public void connectSuccess(String responseString, int responseCode, Void unused) {
                    AbxLog.d("Delete User Data API connect success!!", true);
                    onDeleteComplete(true);
                    callback.callback(ResultCallback.Status.SUCCESS, "Delete API Request Success");
                }

                @Override
                public void connectFail(int responseCode, Void unused) {
                    AbxLog.w("Delete User Data API connect failed!!", true);
                    onDeleteComplete(false);
                    callback.callback(ResultCallback.Status.FAILURE, "Delete API Request Error");
                }
            }, null);
            apiConnectionManager.execute(apiConnection);
        } catch (Exception e) {
            AbxLog.e("Delete API Request Error: ", e, true);
            callback.callback(ResultCallback.Status.FAILURE, e.toString());
        }
    }

    public void onDeleteComplete(boolean isDeleteSynced) {

        if (CoreUtils.getDRState(dataRegistry) == DRState.DELETE_NOT_SYNCED && !isDeleteSynced) {
            AbxLog.e("Delete API failed again!", true);
            return;
        }

        //Restart 시 appKey, secretKey 가 필요하다.
        String appKey = dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY, null);
        String secretKey = dataRegistry.safeGetString(DataRegistryKey.STRING_SECRETKEY, null);

        //in app message 처리
        deleteAllInAppMessageDBContents();

        //in app message 처리
        actionHistoryDAO.deleteDB();
        if (actionHistoryExecutor != null)
            actionHistoryExecutor.shutdownNow();

        //local data 삭제.
        dataRegistry.clearRegistry();

        //다음 SDK 시작시에 stop state를 판단하기 위한 flag 저장.
        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.BOOLEAN_IS_SDK_STOPPED,
                true,
                5,
                this.getClass().getName(),
                true
        ));

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.BOOLEAN_IS_SDK_STOPPED_SERVER_SYNC,
                isDeleteSynced,
                5,
                this.getClass().getName(),
                true
        ));

        //appKey, secretKey 저장
        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_APPKEY,
                appKey,
                5,
                this.getClass().getName(),
                true
        ));

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_SECRETKEY,
                secretKey,
                5,
                this.getClass().getName(),
                true
        ));
    }

    @Override
    public void restartSDK(String userId, ResultCallback<String> callback) {
        try {
            DRModel restartModel = getRestartApiModel(userId);

            IApiConnection apiConnection = factory.createOrGetAPIConnection()
                    .post(restartModel);
            ApiConnectionManager<Void> apiConnectionManager = new ApiConnectionManager<>(new ApiConnectionManager.Result<Void>() {
                @Override
                public void connectSuccess(String responseString, int responseCode, Void unused) {
                    AbxLog.d("Restart SDK API connect success!!", true);
                    onRestartComplete(true);
                    callback.callback(ResultCallback.Status.SUCCESS, "Restart API Request Success");
                }

                @Override
                public void connectFail(int responseCode, Void unused) {
                    AbxLog.w("Restart SDK API connect failed!!", true);
                    onRestartComplete(false);
                    callback.callback(ResultCallback.Status.FAILURE, "Restart API Request Error");
                }
            }, null);
            apiConnectionManager.execute(apiConnection);

        } catch (Exception e) {
            AbxLog.e("Restart API Request Error: ", e, true);
            callback.callback(ResultCallback.Status.FAILURE, e.toString());
        }
    }

    public DRModel getRestartApiModel(String userId) {

        if (CoreUtils.getDRState(dataRegistry) != DRState.NORMAL) {
            DRModelProvider drModelProvider = new DRModelProvider(factory, userId, DRModel.OperationType.INITIALIZE);
            //restart - initialize
            if (CoreUtils.getDRState(dataRegistry) == DRState.INIT_RESTART_NOT_SYNCED) {
                return drModelProvider.getNotSyncedInitRestartModel();
            } else {
                return drModelProvider.provide();
            }
        } else {
            //duplicated restart
            return new DRModelProvider(factory, userId, DRModel.OperationType.RESTART).provide();
        }
    }

    public void onRestartComplete(boolean isRestartSynced) {
        if (CoreUtils.getDRState(dataRegistry) == DRState.INIT_RESTART_NOT_SYNCED && !isRestartSynced) {
            AbxLog.e("Initialize restart failed again!", true);
            return;
        }

        if (isRestartSynced) {
            //Init Restart 성공했으므로 Init Restart Event Datetime 삭제.
            dataRegistry.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.STRING_INIT_RESTART_EVENT_DATETIME,
                            null,
                            5,
                            this.getClass().getName(),
                            true
                    )
            );

            if (CoreUtils.getDRState(dataRegistry) != DRState.NORMAL) {
                actionHistoryDAO.restartDB();
                actionHistoryExecutor = Executors.newSingleThreadExecutor();
            }
        }

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.BOOLEAN_IS_SDK_STOPPED,
                false,
                5,
                this.getClass().getName(),
                true
        ));

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.BOOLEAN_IS_SDK_STOPPED_SERVER_SYNC,
                !isRestartSynced,
                5,
                this.getClass().getName(),
                true
        ));
    }

    @Override
    public void fetchActionHistoryFromServer(@Nullable String token, ActionHistoryIdType idType, List<String> actionType, Completion<Result<List<ActionHistory>>> completion) {
        if (actionHistoryExecutor == null || actionHistoryExecutor.isShutdown()) {
            completion.handle(Error.of("actionHistoryExecutor is not available now!"));
            return;
        }

        actionHistoryExecutor.submit(() -> {
            Result<Pair<String, Integer>> result = null;
            Map<String, String> header = new HashMap<>();

            ActionHistoryQueryModel actionHistoryQueryModel = new ActionHistoryQueryModelProvider(
                    dataRegistry,
                    androidContext,
                    deviceRealtimeDataProvider,
                    factory,
                    idType,
                    actionType,
                    0,
                    0,
                    0
            ).provide();

            if (token != null)
                header.put("Authorization", token);

            IApiConnection apiConnection = factory.createOrGetAPIConnection()
                    .header(header)
                    .post(actionHistoryQueryModel);

            ApiConnectionManager<ActionHistoryDAO> manager = new ApiConnectionManager<>(new ApiConnectionManager.Result<ActionHistoryDAO>() {
                @Override
                public void connectSuccess(String responseString, int responseCode, ActionHistoryDAO actionHistoryDAO) {
                    if(completion == null){
                        AbxLog.d("completion is null", true);
                        return;
                    }
                    if (CommonUtils.isNullOrEmpty(responseString)) {
                        completion.handle(ActionHistoryError.NULL_RESPONSE_ERROR.getError());
                        return;
                    }

                    JSONArray actionHistoryJSONArray = getActionHistoryJSONArray(responseString);

                    AbxLog.d(actionHistoryJSONArray.toString(), true);

                    actionHistoryDAO.insertActionHistory(actionHistoryJSONArray);

                    List<ActionHistory> actionHistories = new ArrayList<>();
                    for (int i = 0; i < actionHistoryJSONArray.length(); i++) {
                        JSONObject actionHistoryJSONObject = actionHistoryJSONArray.optJSONObject(i);
                        actionHistories.add(ActionHistory.fromJSONObject(actionHistoryJSONObject, true));
                    }

                    completion.handle(Success.of(actionHistories));
                }

                @Override
                public void connectFail(int responseCode, ActionHistoryDAO actionHistoryDAO) {
                    if(completion == null){
                        AbxLog.d("completion is null", true);
                        return;
                    }
                    completion.handle(ActionHistoryError.SYNC_SERVER_ERROR.getError());
                }
            }, actionHistoryDAO);
            manager.execute(apiConnection);

            actionHistoryDAO.getLastServerTimeStamp().onSuccess(lastTimestamp -> {
                AbxLog.d("getLastServerTimeStamp : " + lastTimestamp, true);
                actionHistoryDAO.deleteOutdatedPushData(lastTimestamp);
            }).onFailure(it -> {
                AbxLog.d("getLastServerTimeStamp failed!", true);
            });
        });
    }

    private JSONArray getActionHistoryJSONArray(String response) {
        try {
            JSONObject responseJson = new JSONObject(response);

            return responseJson.optJSONArray(CoreConstants.ACTION_HISTORY_RESPONSE_HISTORIES);
        } catch (JSONException e) {
            AbxLog.e(e, true);
            return new JSONArray();
        }
    }

    @Override
    public void insertPushData(String pushDataString) {
        if (actionHistoryExecutor == null || actionHistoryExecutor.isShutdown())
            return;

        actionHistoryExecutor.submit(() -> actionHistoryDAO.insertPushData(pushDataString));
    }

    @Override
    public void getActionHistory(int skip, int limit, List<String> actionType, Completion<Result<List<ActionHistory>>> completion) {
        if (actionHistoryExecutor == null || actionHistoryExecutor.isShutdown()) {
            completion.handle(Error.of("actionHistoryExecutor is not available now!"));
            return;
        }

        actionHistoryExecutor.submit(() -> {
            completion.handle(actionHistoryDAO.getActionHistory(skip, limit, actionType));
        });
    }

    @Override
    public void getAllActionHistory(List<String> actionType, Completion<Result<List<ActionHistory>>> completion) {
        if (actionHistoryExecutor == null || actionHistoryExecutor.isShutdown()) {
            completion.handle(Error.of("actionHistoryExecutor is not available now!"));
            return;
        }

        actionHistoryExecutor.submit(() -> {
            completion.handle(actionHistoryDAO.getAllActionHistory(actionType));
        });
    }

    @Override
    public void deleteActionHistory(@Nullable String token, String historyId, long timestamp, Completion<Result<Empty>> completion) {
        if (actionHistoryExecutor == null || actionHistoryExecutor.isShutdown()) {
            completion.handle(Error.of("actionHistoryExecutor is not available now!"));
            return;
        }

        actionHistoryExecutor.submit(() -> {
            if (!actionHistoryDAO.deleteActionHistory(historyId, timestamp).isSucceeded()) {
                completion.handle(ActionHistoryError.SQLITE_QUERY_ERROR.getError());
                return;
            }

            ActionHistoryDeleteModel actionHistoryDeleteModel = null;
            try {
                actionHistoryDeleteModel = new ActionHistoryDeleteModelProvider(
                        dataRegistry,
                        androidContext,
                        deviceRealtimeDataProvider,
                        factory,
                        null,
                        historyId,
                        timestamp
                ).provide();
            } catch (Exception e) {
                AbxLog.e(e, true);
                completion.handle(Error.of(e));
            }

            if (actionHistoryDeleteModel == null) {
                completion.handle(ActionHistoryError.NULL_ACTION_HISTORY_API_MODEL_ERROR.getError());
                return;
            }

            Map<String, String> header = new HashMap<>();
            if (token != null)
                header.put("Authorization", token);

            IApiConnection apiConnection = factory.createOrGetAPIConnection()
                    .header(header)
                    .delete(actionHistoryDeleteModel);
            ApiConnectionManager<Void> manager = new ApiConnectionManager<>(new ApiConnectionManager.Result<Void>() {
                @Override
                public void connectSuccess(String responseString, int responseCode, Void unused) {
                    AbxLog.d("deleteActionHistory connect success!!", true);
                    if(completion == null){
                        AbxLog.d("completion is null", true);
                        return;
                    }
                    completion.handle(Success.empty());
                }

                @Override
                public void connectFail(int responseCode, Void unused) {
                    AbxLog.w("deleteActionHistory connect failed!!", true);
                    if(completion == null){
                        AbxLog.d("completion is null", true);
                        return;
                    }
                    completion.handle(ActionHistoryError.DELETE_REMOTE_DB_FAIL_ERROR.getError());
                }
            }, null);
            manager.execute(apiConnection);
        });
    }

    @Override
    public void deleteAllActionHistory(@Nullable String token, @Nullable ActionHistoryIdType idType, Completion<Result<Empty>> completion) {
        if (actionHistoryExecutor == null || actionHistoryExecutor.isShutdown()) {
            completion.handle(Error.of("actionHistoryExecutor is not available now!"));
            return;
        }

        actionHistoryExecutor.submit(() -> {
            if (!actionHistoryDAO.clearSyncedActionHistoryInLocalDB().isSucceeded()) {
                completion.handle(ActionHistoryError.SQLITE_QUERY_ERROR.getError());
                return;
            }

            ActionHistoryDeleteModel actionHistoryDeleteModel = null;
            try {
                actionHistoryDeleteModel = new ActionHistoryDeleteModelProvider(
                        dataRegistry,
                        androidContext,
                        deviceRealtimeDataProvider,
                        factory,
                        idType,
                        null,
                        0
                ).provide();
            } catch (Exception e) {
                AbxLog.e(e, true);
                completion.handle(Error.of(e));
            }

            if (actionHistoryDeleteModel == null) {
                completion.handle(ActionHistoryError.NULL_ACTION_HISTORY_API_MODEL_ERROR.getError());
                return;
            }

            Map<String, String> header = new HashMap<>();
            if (token != null)
                header.put("Authorization", token);

            IApiConnection apiConnection = factory.createOrGetAPIConnection()
                    .header(header)
                    .delete(actionHistoryDeleteModel);
            ApiConnectionManager<Void> apiConnectionManager = new ApiConnectionManager(new ApiConnectionManager.Result<Void>() {
                @Override
                public void connectSuccess(String responseString, int responseCode, Void unused) {
                    AbxLog.d("deleteAllActionHistory connect success!!", true);
                    completion.handle(Success.empty());
                }

                @Override
                public void connectFail(int responseCode, Void unused) {
                    AbxLog.d("deleteAllActionHistory connect failed!!", true);
                    completion.handle(ActionHistoryError.DELETE_REMOTE_DB_FAIL_ERROR.getError());
                }
            }, null);
            apiConnectionManager.execute(apiConnection);
        });
    }

    @Override
    public void clearSyncedActionHistoryInLocalDB(Completion<Result<Empty>> completion) {
        if (actionHistoryExecutor == null || actionHistoryExecutor.isShutdown()) {
            completion.handle(Error.of("actionHistoryExecutor is not available now!"));
            return;
        }

        actionHistoryExecutor.submit(() -> {
            completion.handle(actionHistoryDAO.clearSyncedActionHistoryInLocalDB());
        });
    }

    @Override
    public void clearAllActionHistoryInLocalDB() {
        if (actionHistoryExecutor == null || actionHistoryExecutor.isShutdown())
            return;

        actionHistoryExecutor.submit(() -> {
            actionHistoryDAO.clearAllActionHistoryInLocalDB()
                    .onSuccess(empty -> AbxLog.d("clearAllActionHistoryInLocalDB onSuccess!", true))
                    .onFailure(error -> AbxLog.d("clearAllActionHistoryInLocalDB onFailure: " + error.getMessage(), true));
        });
    }

    @Override
    public void loadDFNId() {
        ApiConnectionManager<DataRegistry> apiConnectionManager = new ApiConnectionManager<>(new ApiConnectionManager.Result<DataRegistry>() {
            @Override
            public void connectSuccess(String responseString, int responseCode, DataRegistry dataRegistry) {
                AbxLog.d("DFN-ID API connect success!! " + responseString, true);
                try {
                    JSONObject responseData = new JSONObject(responseString);
                    JSONObject data = responseData.optJSONObject("data");
                    String dfnId = data != null ? data.optString("dfn_id") : null;

                    if (CommonUtils.isNullOrEmpty(dfnId)) {
                        AbxLog.w("DFN-ID is null or empty", true);
                        return;
                    }

                    dataRegistry.putDataRegistry(new DataUnit(
                            DataRegistryKey.STRING_DFN_ID,
                            dfnId,
                            5,
                            this.getClass().getName(),
                            true
                    ));

                    //call beacon api
                    startTvAttributionBeacon(dfnId);

                } catch (JSONException e) {
                    AbxLog.w(e, true);
                }
            }

            @Override
            public void connectFail(int responseCode, DataRegistry dataRegistry) {
                AbxLog.d("DFN-ID API connect failed!! responseCode : " + responseCode, true);
            }
        }, dataRegistry);

        DFNIDModel dfnidModel = new DFNIDModelProvider(dataRegistry, androidContext, deviceRealtimeDataProvider).provide();
        IApiConnection apiConnection = factory.createOrGetAPIConnection().post(dfnidModel);
        apiConnectionManager.executeWithRetry(apiConnection);
    }

    @Override
    public void startTvAttributionBeacon(String dfnId) {
        if(androidContext == null){
            AbxLog.w("context is null", true);
            return;
        }
        if(CommonUtils.isNullOrEmpty(dfnId)){
            AbxLog.d("dfnId is null", true);
            return;
        }
        if(Build.VERSION.SDK_INT <= 26){
            AbxLog.d("Android OS API 8 is not supported", true);
            return;
        }
        long isActive = dataRegistry.safeGetLong(DataRegistryKey.LONG_S3_CONFIG_TV_ATTRIBUTION_WITHOUT_SUPPORT_LIBRARY_ACTIVE, 0);
        if(isActive != 1){
            AbxLog.d("LONG_S3_CONFIG_TV_ATTRIBUTION_ACTIVE is "+isActive, true);
            return;
        }

        if(!CommonUtils.hasClass("com.pci.beacon.PCI")){
            return;
        }
        AbxLog.i("startTvAttributionBeacon", true);
        Executors.newSingleThreadExecutor().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    PCI.with(androidContext).beaconStart(dfnId, "1004");
                    //PCI.with(androidContext).DMRStart(dfnId, "1004");
                }
                catch (IncompatibleClassChangeError error){}
                catch (Exception e){
                    AbxLog.w(e, true);
                }
            }
        });
    }

    /**
     * 2023.03.27 bobos
     * @see "https://github.com/IGAWorksDev/DFNGitOps/issues/477"
     *
     */
    @Override
    public void sendDataToSkBroadbandSetTopBox() {
        final String SKB_OS_ANDROID = "0";
        final String SKB_OS_IOS = "1";
        final String SKB_OS_EXTRA = "2";
        final String SKB_VENDOR_CODE_SKB = "C00000";
        final String SKB_VENDOR_CODE_SKP = "C01000";
        final String SKB_VENDOR_CODE_ONNURI_DMC = "C02000";
        final String SKB_VENDOR_CODE_IGAWORKS = "C03000";
        final String SKB_VENDOR_CODE_SKSTOA = "C04000";
        final String SKB_VENDOR_CODE_LOTTE_MEMBERS = "C05000";

        long isActive = dataRegistry.safeGetLong(DataRegistryKey.LONG_S3_CONFIG_SK_BROADBAND_ATTRIBUTION_ACTIVE, 0);
        if(isActive != 1){
            AbxLog.d("LONG_S3_CONFIG_SK_BROADBAND_ATTRIBUTION_ACTIVE is "+isActive, true);
            return;
        }
        String network = this.dataRegistry.safeGetString(DataRegistryKey.STRING_NETWORK, null);
        if(!"wifi".equals(network)){
            //셋톱박스와 같은 wifi망에 속해있을때만 동작하므로 wifi 아닐때는 불필요함
            return;
        }
        String gaid = this.dataRegistry.safeGetString(DataRegistryKey.STRING_GAID, null);
        if(CommonUtils.isNullOrEmpty(gaid)){
            //gaid 없으면 동작하지 않음
            return;
        }
        String ipAndPort = dataRegistry.safeGetString(DataRegistryKey.STRING_S3_CONFIG_SK_IP_PORT, "23919300801006000");
        String ip = getSkBroadbandIp(ipAndPort);
        int port = getSkBroadbandPort(ipAndPort);
        int pingPeriodSecond = dataRegistry.safeGetInt(DataRegistryKey.INT_S3_CONFIG_SK_PING_PERIOD_SECOND, 300);//안씀
        int skPingVersion = dataRegistry.safeGetInt(DataRegistryKey.INT_S3_CONFIG_SK_PING_VERSION, 1);//프로토콜 버전
        final String skBroadbandMessage = getSkBroadbandMessage(gaid, skPingVersion, SKB_OS_ANDROID, SKB_VENDOR_CODE_IGAWORKS);
        new Thread(new Runnable() {
            @Override
            public void run() {
                DatagramSocket clientSocket = null;
                try {
                    InetAddress inetAddress = InetAddress.getByName(ip);
                    byte[] sendData = skBroadbandMessage.getBytes();
                    DatagramPacket sendPacket = new DatagramPacket(sendData, sendData.length, inetAddress, port);
                    clientSocket = new DatagramSocket();
                    clientSocket.send(sendPacket);
                    AbxLog.d("adbrix->skb sending completed. "+"ip: "+ip+" port: "+port+"\ndata: "+skBroadbandMessage, true);
                } catch (SocketException e) {
                    AbxLog.w(e, true);
                } catch (UnknownHostException e) {
                    AbxLog.w(e, true);
                } catch (IOException e) {
                    AbxLog.w(e, true);
                } finally {
                    try {
                        if (clientSocket != null) {
                            clientSocket.close();
                        }
                    } catch (Exception e) {
                        // ignore
                    }
                }
            }
        }).start();
    }
    private String getSkBroadbandIp(String ipAndPort) {
        if (ipAndPort == null || ipAndPort.length() != 17)
            return null;

        String tempIp = ipAndPort.substring(0, 12);

        String dottedIp = "";

        for (int i = 0; i < tempIp.length(); i += 3) {
            dottedIp += tempIp.substring(i, i + 3) + ".";
        }
        dottedIp = dottedIp.substring(0, 15);

        return dottedIp;
    }

    private int getSkBroadbandPort(String ipAndPort) {
        if (ipAndPort == null || ipAndPort.length() != 17)
            return -1;

        String tempPort = ipAndPort.substring(12, 17);

        if (tempPort.charAt(0) == '0') {
            tempPort = tempPort.substring(1, 5);
        }

        return Integer.parseInt(tempPort);
    }
    private String getSkBroadbandMessage(String gaid, int protocolVersion, String osCode, String vendorCode){
        StringBuilder mutableFieldBuilder = new StringBuilder();
        mutableFieldBuilder.append("\"os\":");//OS구분
        mutableFieldBuilder.append("\"");
        mutableFieldBuilder.append(osCode);
        mutableFieldBuilder.append("\"");
        mutableFieldBuilder.append(",");
        mutableFieldBuilder.append("\"ccode\":");//제휴사 코드
        mutableFieldBuilder.append("\"");
        mutableFieldBuilder.append(vendorCode);
        mutableFieldBuilder.append("\"");
        String mutableField = Base64.encode(mutableFieldBuilder.toString());
        StringBuilder messageBuilder = new StringBuilder();
        messageBuilder.append(String.format("%04d", protocolVersion));//프로토콜 버전
        messageBuilder.append(gaid);
        messageBuilder.append(String.format("%04d", mutableField.length()));
        messageBuilder.append(mutableField);
        return messageBuilder.toString();
    }

    public InAppMessageDAO getInAppMessageDAO() {
        return inAppMessageDAO;
    }
}
