package io.adbrix.sdk.utils;

import android.content.Context;
import android.database.Cursor;
import android.os.Build;
import android.os.Bundle;
import android.os.CountDownTimer;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.Closeable;
import java.lang.ref.WeakReference;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.TimeZone;
import java.util.Timer;
import java.util.TimerTask;
import java.util.UUID;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.data.NullableJSONObject;
import io.adbrix.sdk.domain.CompatConstants;
import io.adbrix.sdk.domain.CoreConstants;

public class CommonUtils {

    private CommonUtils() {
    }

    public static boolean notNull(Object object) {
        if(object == null) {
            return false;
        }
        else {
            return true;
        }
    }
    public static boolean isNull(Object object) {
        if(object == null) {
            return true;
        }
        else {
            return false;
        }
    }
    public static boolean isNullOrEmpty(String string) {
        return string == null || string.isEmpty();
    }

    public static boolean isNullOrEmpty(Bundle bundle){
        boolean result = false;
        if(bundle == null){
            result = true;
            return result;
        }
        if(bundle.size() == 0){
            result = true;
            return result;
        }
        return result;
    }

    public static boolean isNullOrEmpty(List list){
        boolean result = false;
        if(list == null){
            result = true;
            return result;
        }
        if(list.size() == 0){
            result = true;
            return result;
        }
        return result;
    }
    public static boolean isNullOrEmpty(Queue queue){
        boolean result = false;
        if(queue == null){
            result = true;
            return result;
        }
        if(queue.size() == 0){
            result = true;
            return result;
        }
        return result;
    }

    public static boolean isNullOrEmpty(Map map){
        boolean result = false;
        if(map == null){
            result = true;
            return result;
        }
        if(map.isEmpty()){
            result = true;
            return result;
        }
        return result;
    }
    public static boolean isNullOrEmpty(Object[] value){
        boolean result = false;
        if(value == null){
            result = true;
            return result;
        }
        if(value.length == 0){
            result = true;
            return result;
        }
        return result;
    }
    public static boolean isNullOrEmpty(JSONObject value){
        boolean result = false;
        if(value == null){
            result = true;
            return result;
        }
        if(value.length() == 0){
            result = true;
            return result;
        }
        return result;
    }
    public static boolean isNullOrEmpty(JSONArray value){
        boolean result = false;
        if(value == null){
            result = true;
            return result;
        }
        if(value.length() == 0){
            result = true;
            return result;
        }
        return result;
    }

    public static Bundle convertToBundle(String jsonString){
        Bundle bundle = new Bundle();
        try {
            JSONObject jsonObject = new JSONObject(jsonString);
            bundle = convertToBundle(jsonObject);
        } catch (JSONException e) {
            AbxLog.e(e, false);
        }
        return bundle;
    }

    public static Bundle convertToBundle(JSONObject jsonObject) {
        Bundle bundle = new Bundle();
        try {
            Iterator iter = jsonObject.keys();
            while(iter.hasNext()) {
                String key = (String)iter.next();
                String value = jsonObject.getString(key);
                bundle.putString(key,value);
            }
        }
        catch (Exception e) {
            AbxLog.e(e, false);
        }
        return bundle;
    }

    public static Map<String, Object> convertToMap(Bundle extras) {
        Map<String, Object> map = new HashMap<String, Object>();

        Set<String> ks = extras.keySet();
        Iterator<String> iterator = ks.iterator();
        while (iterator.hasNext()) {
            String key = iterator.next();
            map.put(key, extras.get(key));
        }
        return map;
    }
    public static Map<String, Object> convertToMap(JSONObject jsonObject) {
        Map<String, Object> map = new HashMap<String, Object>();
        try {
            Iterator iter = jsonObject.keys();
            while(iter.hasNext()) {
                String key = (String)iter.next();
                Object value = jsonObject.get(key);
                map.put(key, value);
            }
        }
        catch (Exception e) {
            AbxLog.e(e, false);
        }
        return map;
    }

    public static Bundle convertToBundle(Map<String, String> map){
        Bundle bundle = new Bundle();
        Iterator<String> iterator = map.keySet().iterator();
        while (iterator.hasNext()){
            String key = iterator.next();
            String value = map.get(key);
            bundle.putString(key, value);
        }
        return bundle;
    }

    public static String getUTCDataBaseFormat(long timeStamp) {
        SimpleDateFormat sdfKST = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", Locale.KOREA);
        sdfKST.setTimeZone(TimeZone.getTimeZone("GMT+0"));
        return sdfKST.format(new Date(timeStamp));
    }

    public static String convertNullStringToNull(String string) {
        return string != null && string.equals("null") ? null : string;
    }

    // (현재 시간):(UUID) 형식의 문자열을 만드는 Utility Method이다. Event Id 등을 만드는 데에 사용된다.
    public static String randomUUIDWithCurrentTime() {
        return System.currentTimeMillis() + ":" + UUID.randomUUID().toString();
    }

    public static String getCurrentUTCInDBFormat() {
        DateFormat dateFormat = CoreUtils.createDateFormat(CoreConstants.DB_DATE_FORMAT);
        Date date = new Date(System.currentTimeMillis());
        return dateFormat.format(date);
    }

    public static String randomUUIDWithCurrentTime(long currentTimeMillis) {
        return currentTimeMillis + ":" + UUID.randomUUID().toString();
    }

    public static String getCurrentUTCInDBFormat(long currentTimeMillis) {
        DateFormat dateFormat = CoreUtils.createDateFormat(CoreConstants.DB_DATE_FORMAT);
        Date date = new Date(currentTimeMillis);
        return dateFormat.format(date);
    }

    /**
     * Truncate values in a JSON object. Any string values longer than 1024 characters will be
     * truncated to 1024 characters.
     * Any dictionary with more than 1000 items will be ignored.
     *
     * @param object the object
     * @return the truncated JSON object
     */
    public static JSONObject truncate(JSONObject object, int propertyMaxSize) {
        if (object == null) {
            return new JSONObject();
        }
        Iterator<?> keys = object.keys();
        while (keys.hasNext()) {
            String key = (String) keys.next();
            if (key.length() > CompatConstants.MAX_KEY_LENGTH) {
                AbxLog.w(String.format("Validation CHECK :: Invalid key >> Maximum length is %s. Key %s is too long.", String.valueOf(CompatConstants.MAX_KEY_LENGTH), key.substring(0, CompatConstants.MAX_KEY_LENGTH)), true);
                object.remove(key);
            } else {
                try {
                    Object value = object.get(key);
                    // do not truncate important key
                    if (key.equals(CoreConstants.SPECIAL_KEY) || key.equals(CoreConstants.DEEPLINK_PAYLOAD) || key.equals(CoreConstants.COMMERCE_KEY)) {
                        object.put(key, value);
                    } else {

                        if (!isNullOrEmpty(key)) {

                            if (value.getClass().equals(String.class)) {
                                object.put(key, truncate((String) value));
                            } else if (value.getClass().equals(JSONObject.class)) {
                                object.put(key, truncate((JSONObject) value, propertyMaxSize));  //this method
                            } else if (value.getClass().equals(JSONArray.class)) {
                                object.put(key, truncate((JSONArray) value,propertyMaxSize));
                            }
                        }
                    }
                    if (object.length() > propertyMaxSize) {
                        AbxLog.w("Validation CHECK :: Warning: properties over the limit " + propertyMaxSize + ", exceeded objects be ignored", true);
                        break;
                    }
                } catch (JSONException e) {
                    AbxLog.e(e, true);
                }
            }


        }

        return object;
    }

    /**
     * Truncate a string to 1024 characters.
     *
     * @param value the value
     * @return the truncated string
     */
    public static String truncate(String value) {
        return value.length() <= CoreConstants.MAX_STRING_LENGTH ? value : value.substring(0, CoreConstants.MAX_STRING_LENGTH);
    }

    /**
     * Truncate values in a JSON array. Any string values longer than 1024 characters will be
     * truncated to 1024 characters.
     *
     * @param array the array
     * @return the truncated JSON array
     * @throws JSONException the json exception
     */
    public static JSONArray truncate(JSONArray array, int propertyMaxSize) throws JSONException {
        if (array == null) {
            return new JSONArray();
        }

        for (int i = 0; i < array.length(); i++) {
            Object value = array.get(i);
            if (value.getClass().equals(String.class)) {
                array.put(i, truncate((String) value));
            } else if (value.getClass().equals(JSONObject.class)) {
                array.put(i, truncate((JSONObject) value,propertyMaxSize));
            } else if (value.getClass().equals(JSONArray.class)) {
                array.put(i, truncate((JSONArray) value,propertyMaxSize));
            }
        }
        return array;
    }

    public static Map<String, Object> parseValueWithDataType(Map<String, Object> map, FixType fixType) {
        JSONObject jsonObject = getJSONObjectFromMap(map);
        jsonObject = parseValueWithDataType(jsonObject, fixType);
        map = getMapFromJSONObject(jsonObject);
        return map;
    }

    public static JSONObject getJSONObjectFromMap(Map<String, Object> map) {
        NullableJSONObject jsonObject = new NullableJSONObject();
        if(map == null){
            return jsonObject;
        }
        try {
            for (Map.Entry<String, Object> entry : map.entrySet()) {
                String key = entry.getKey();
                Object value = entry.getValue();
                jsonObject.put(key, value);
            }
        } catch (Exception e) {
            AbxLog.w(e, true);
        }

        return jsonObject;
    }

    public static Map<String, Object> getMapFromJSONObject(JSONObject object) {
        Map<String, Object> map = new HashMap<>();
        if(object == null){
            return map;
        }
        Iterator<String> keysItr = object.keys();
        try {
            while (keysItr.hasNext()) {
                String key = keysItr.next();

                Object value = object.get(key);

                if (value instanceof JSONObject) {
                    value = getMapFromJSONObject((JSONObject) value);
                }
                map.put(key, value);
            }
        } catch (Exception e) {
            AbxLog.w(e, true);
        }
        return map;
    }

    public static String getParseValueWithDataType(Object obj, FixType fixType){
        StringBuilder sb = new StringBuilder();

        if (obj instanceof String) {
            sb.append("string");
        } else if (obj instanceof Boolean) {
            sb.append("boolean");
        } else if (obj instanceof Integer) {
            sb.append("long");
        } else if (obj instanceof Long) {
            sb.append("long");
        } else if (obj instanceof Double) {
            sb.append("double");
        } else if (obj instanceof Float) {
            sb.append("double");
        } else {
            sb.append("string");
        }

        if (FixType.PREFIX == fixType) {
            sb.append(":");
            sb.append(obj);
        } else {
            sb.insert(0, ":");
            sb.insert(0, obj);
        }
        return sb.toString();
    }

    public static JSONObject parseValueWithDataType(JSONObject json, FixType fixType) {
        JSONObject retJsonObject = new JSONObject();
        try {
            if (json != null) {
                Iterator<?> keys = json.keys();
                while (keys.hasNext()) {
                    String key = (String) keys.next();

                    Object obj = json.get(key);
                    if (obj == null || obj == JSONObject.NULL || obj == "") {
                        continue;
                    }
                    String parsedValue = getParseValueWithDataType(obj, fixType);
                    retJsonObject.put(key, parsedValue);
                }
            }
        } catch (JSONException e) {
            AbxLog.e(e, true);
        }

        return retJsonObject;
    }

    public static Object replaceWithJSONNull(Object obj) {
        return obj == null ? JSONObject.NULL : obj;
    }

    public static <T> List<List<T>> split(List<T> list, int splitSize) {
        List<List<T>> lists = new ArrayList<>();
        List<T> beginning;
        List<T> remaining;

        if (list.size() <= splitSize) {
            lists.add(list);
            return lists;
        } else {
            beginning = list.subList(0, splitSize);
            remaining = list.subList(splitSize, list.size());

            lists.add(beginning);
            lists.addAll(split(remaining, splitSize));

            return lists;
        }
    }

    public static String getLanguage(Context context) {
        String lang = Locale.getDefault().toString();
        String newFormatLanguage;
        try {
            if (Build.VERSION.SDK_INT >= 24) { //From Build.VERSION_CODES.N
                if (context != null && context.getResources() != null && context.getResources().getConfiguration() != null &&
                        context.getResources().getConfiguration().getLocales() != null) {
                    if (context.getResources().getConfiguration().getLocales().size() > 0) {
                        String temp = context.getResources().getConfiguration().getLocales().get(0).toString();
                        if (temp != null && !temp.equals("")) lang = temp;
                    }
                }
            }
        } catch (Exception e) {
            AbxLog.e("GetLanguage Error: ",e, true);
            lang = Locale.getDefault().getLanguage(); // System locale
        }
        if (isNullOrEmpty(lang)) {
            return "unknown";
        }
        String[] splitElement = lang.split("_");
        if (splitElement.length < 2) {
            return lang;
        } else if (splitElement.length == 2) {
            return splitElement[0].toUpperCase(Locale.ENGLISH);
        } else {
            splitElement[2] = splitElement[2].replaceAll("#", "");
            newFormatLanguage = splitElement[0].toUpperCase(Locale.ENGLISH) + "-" + splitElement[2].toUpperCase(Locale.ENGLISH);
            return newFormatLanguage;
        }
    }

    public enum FixType {
        PREFIX, SUFFIX
    }
    public static void cancelCountDownTimer(CountDownTimer timer){
        if(timer!=null){
            timer.cancel();
        }
    }
    public static void cancelTimer(Timer timer){
        try {
            if(timer!=null){
                timer.cancel();
                timer.purge();
            }
        }catch (Exception e){
            AbxLog.e(e, true);
        }
    }
    public static void cancelTimer(WeakReference<Timer> timerReference){
        Timer timer = null;
        if(timerReference!=null){
            timer = timerReference.get();
        }
        cancelTimer(timer);
    }
    public static void cancelTimerTask(TimerTask timerTask){
        if(timerTask!=null){
            timerTask.cancel();
        }
    }
    public static void cancelTimerTask(WeakReference<TimerTask> timerTaskReference){
        TimerTask timerTask = null;
        if(timerTaskReference!=null){
            timerTask = timerTaskReference.get();
        }
        cancelTimerTask(timerTask);
    }

    public static void close(Cursor cursor){
        if(cursor == null){
            return;
        }
        if(cursor.isClosed()){
            return;
        }
        cursor.close();
    }
    public static void close(Closeable closeable){
        if(closeable == null){
            return;
        }
        try {
            closeable.close();
        }catch (Exception e){
            AbxLog.e(e, true);
        }
    }
    public static void close(AutoCloseable closeable){
        if(closeable == null){
            return;
        }
        try {
            closeable.close();
        }catch (Exception e){
            AbxLog.e(e, true);
        }
    }

    public static boolean hasClass(String className){
        boolean result = false;
        Class<?> cls = null;
        try {
            cls = Class.forName(className);
        } catch (ClassNotFoundException e) {
            AbxLog.w(className+" is not exist", e, true);
            return result;
        }
        if(cls != null){
            result = true;
            return result;
        }
        return result;
    }
}
